"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscalingManager = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
/**
 * Settings for the manager (coordinator)
 *
 *  Manager coordinates the placement of runner (job executor) instances
 */
class GitlabRunnerAutoscalingManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.globalConfiguration =
            props.globalConfiguration ||
                {
                    concurrent: 10,
                    checkInterval: 0,
                    logFormat: "runner",
                    logLevel: "info",
                };
        this.machineImage = (_b = props.machineImage) !== null && _b !== void 0 ? _b : aws_ec2_1.MachineImage.latestAmazonLinux({
            generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
            virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
            storage: aws_ec2_1.AmazonLinuxStorage.EBS,
            cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
        });
        this.instanceType = (_c = props.instanceType) !== null && _c !== void 0 ? _c : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        this.keyPairName = props.keyPairName;
        this.runners = props.runners;
        this.network = props.network;
        this.cacheBucket = props.cacheBucket;
        this.runnersSecurityGroupName = props.runnersSecurityGroup.securityGroupName;
        this.role =
            props.role ||
                new aws_iam_1.Role(scope, "ManagerRole", {
                    assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {}),
                    managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")],
                    inlinePolicies: {
                        Cache: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                                    Resource: [`${this.cacheBucket.bucketArn}/*`],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListBucket"],
                                    Resource: [`${this.cacheBucket.bucketArn}`],
                                },
                            ],
                        }),
                        Runners: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                                    Resource: ["*"],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "aws:RequestTag/Name": "*gitlab-runner-*",
                                        },
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["Name"],
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringEqualsIfExists: {
                                            "ec2:Region": `${scope.region}`,
                                        },
                                        ArnEqualsIfExists: {
                                            "ec2:Vpc": `arn:aws:ec2:${scope.region}:${scope.account}:vpc/${this.network.vpc.vpcId}`,
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RunInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["InstanceProfile"],
                                            "ec2:InstanceType": (this.runners || []).map((runner) => {
                                                const runnersInstanceType = (this.runners && runner.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
                                                return runnersInstanceType.toString();
                                            }),
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["iam:PassRole"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["RunnersRole"],
                                        },
                                    },
                                },
                            ],
                        }),
                    },
                });
        this.userData = aws_ec2_1.UserData.forLinux({});
        this.userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        this.initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: this.globalConfiguration,
                        runnersConfiguration: this.runners.map((runner) => {
                            var _b;
                            const configuration = runner.configuration;
                            return {
                                ...configuration,
                                machine: {
                                    ...configuration.machine,
                                    machineOptions: {
                                        ...(_b = configuration.machine) === null || _b === void 0 ? void 0 : _b.machineOptions,
                                        instanceType: runner.instanceType.toString(),
                                        ami: runner.machineImage.getImage(scope).imageId,
                                        region: scope.region,
                                        vpcId: this.network.vpc.vpcId,
                                        zone: this.network.availabilityZone.slice(-1),
                                        subnetId: this.network.subnet.subnetId,
                                        securityGroup: this.runnersSecurityGroupName,
                                        usePrivateAddress: true,
                                        iamInstanceProfile: runner.instanceProfile.ref,
                                    },
                                },
                                cache: {
                                    s3: {
                                        serverAddress: `s3.${scope.urlSuffix}`,
                                        bucketName: `${this.cacheBucket.bucketName}`,
                                        bucketLocation: `${scope.region}`,
                                    },
                                },
                            };
                        }),
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
    }
}
exports.GitlabRunnerAutoscalingManager = GitlabRunnerAutoscalingManager;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscalingManager[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscalingManager", version: "0.1.20" };
//# sourceMappingURL=data:application/json;base64,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