"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * CLB listeners are configured for secure (HTTPs or SSL) protocols for client communication
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        const listeners = monocdk_1.Stack.of(node).resolve(node.listeners);
        for (const listener of listeners) {
            const resolvedListener = monocdk_1.Stack.of(node).resolve(listener);
            const protocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.protocol);
            const instanceProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.instanceProtocol);
            if (protocol.toLowerCase() == 'ssl') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'ssl')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
            else if (protocol.toLowerCase() == 'https') {
                if (!(instanceProtocol == undefined ||
                    instanceProtocol.toLowerCase() == 'https')) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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