"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const appsync_1 = require("../rules/appsync");
const athena_1 = require("../rules/athena");
const autoscaling_1 = require("../rules/autoscaling");
const cloud9_1 = require("../rules/cloud9");
const cloudfront_1 = require("../rules/cloudfront");
const codebuild_1 = require("../rules/codebuild");
const cognito_1 = require("../rules/cognito");
const documentdb_1 = require("../rules/documentdb");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecr_1 = require("../rules/ecr");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const eks_1 = require("../rules/eks");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const glue_1 = require("../rules/glue");
const iam_1 = require("../rules/iam");
const kinesis_1 = require("../rules/kinesis");
const kms_1 = require("../rules/kms");
const mediastore_1 = require("../rules/mediastore");
const msk_1 = require("../rules/msk");
const neptune_1 = require("../rules/neptune");
const opensearch_1 = require("../rules/opensearch");
const quicksight_1 = require("../rules/quicksight");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const sqs_1 = require("../rules/sqs");
const stepfunctions_1 = require("../rules/stepfunctions");
const timestream_1 = require("../rules/timestream");
const vpc_1 = require("../rules/vpc");
/**
 * Check Best practices based on AWS Solutions Security Matrix
 *
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'AwsSolutions';
    }
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleSuffixOverride: 'EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default in order to seperate your environment from default resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkVPCSpecified,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: elasticbeanstalk_1.ElasticBeanstalkEC2InstanceLogsToS3,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSVolumeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2SecurityGroupDescription,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceTerminationProtection,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecr_1.ECROpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS2',
            info: 'The ECS Task Definition includes a container definition that directly specifies environment variables.',
            explanation: 'Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionNoEnvironmentVariables,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSClusterCloudWatchContainerInsights,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS7',
            info: 'The ECS Task Definition does not have awslogs logging enabled at the minimum.',
            explanation: 'Container logging allows operators to view and aggregate the logs from the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionContainerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS1',
            info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
            explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterNoEndpointPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS2',
            info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
            explanation: 'EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterControlPlaneLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBNoInboundHttpHttps,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB2',
            info: 'The ELB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBConnectionDraining,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleSuffixOverride: 'S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S10',
            info: 'The S3 Bucket does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleSuffixOverride: 'RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS3',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled.',
            explanation: 'Use multi-AZ deployment configurations for high availability and automatic failover support fully managed by AWS.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLPostgresIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS8',
            info: 'The RDS DB Security Group allows for 0.0.0.0/0 inbound access.',
            explanation: 'RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSRestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLBacktrack,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS15',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level for Amazon Aurora databases or instance level for non Aurora instances helps protect from accidental deletion.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DAXEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterRedisAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterAutomaticMinorVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS1',
            info: 'The Redshift cluster does not require TLS/SSL encryption.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultUsername,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterAuditLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters databse. To enable this feature associate a Resdhsift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterUserActivityLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBCredentialsInSecretsManager,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterLogExports,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: timestream_1.TimestreamDatabaseCustomerManagedKey,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleSuffixOverride: 'VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCNoNACLs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionGeoRestrictions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionWAFIntegration,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR3',
            info: 'The CloudFront distributions does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessIdentity,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWRequestValidation,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAuthorization,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleSuffixOverride: 'AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupCooldownPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupHealthCheck,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupScalingNotifications,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleSuffixOverride: 'SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleSuffixOverride: 'ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: athena_1.AthenaWorkgroupEncryptedQueryResults,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRS3AccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR4',
            info: 'The EMR cluster does not use a security configuration with local disk encryption enabled.',
            explanation: 'Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRLocalDiskEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR5',
            info: 'The EMR cluster does not use a security configuration with encryption in transit enabled and configured.',
            explanation: 'EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMREncryptionInTransit,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRAuthEC2KeyPairOrKerberos,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL1',
            info: 'The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.',
            explanation: 'Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueEncryptedCloudWatchLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL3',
            info: 'The Glue job does not have use a security configuration with job bookmark encryption enabled.',
            explanation: 'Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueJobBookmarkEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataAnalyticsFlinkCheckpointing,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataFirehoseSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataStreamSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataStreamDefaultKeyWhenSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKClientToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchAllowlistedIPs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchDedicatedMasterNode,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNoUnsignedOrAnonymousAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchZoneAwareness,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchSlowLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: quicksight_1.QuicksightSSLConnections,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleSuffixOverride: 'IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoManagedPolicies,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoWildcardPermissions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolStrongPasswordPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cognito_1.CognitoUserPoolMFA,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAdvancedSecurityModeEnforced,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAPIGWAuthorizer,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolNoUnauthenticatedLogins,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleSuffixOverride: 'ASC3',
            info: 'The GraphQL API does not have request leveling logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: appsync_1.AppSyncGraphQLRequestLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineAllLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineXray,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleSuffixOverride: 'SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS3',
            info: 'The SNS Topic does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the topic policy to force requests to use SSL. If SSE is already enabled then this control is auto enforced.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSTopicSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueDLQ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS4',
            info: 'The SQS queue does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSLRequestsOnly,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleSuffixOverride: 'MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS3',
            info: 'The MediaStore container does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreCloudWatchMetricPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerHasContainerPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerCORSPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerLifecyclePolicy,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleSuffixOverride: 'CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectPrivilegedModeDisabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectKMSEncryptedArtifacts,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectManagedImages,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloud9_1.Cloud9InstanceNoIngressSystemsManager,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "1.5.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLXNvbHV0aW9ucy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9wYWNrcy9hd3Mtc29sdXRpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztFQUdFO0FBQ0YscUNBQWtEO0FBQ2xELDBDQUFxRTtBQUNyRSwwQ0FNd0I7QUFDeEIsOENBQWdFO0FBQ2hFLDRDQUF1RTtBQUN2RSxzREFJOEI7QUFDOUIsNENBQXdFO0FBQ3hFLG9EQU02QjtBQUM3QixrREFJNEI7QUFDNUIsOENBTTBCO0FBQzFCLG9EQU02QjtBQUM3QixnREFBc0U7QUFDdEUsc0NBTXNCO0FBQ3RCLHNDQUE2QztBQUM3QyxzQ0FJc0I7QUFDdEIsc0NBQTRDO0FBQzVDLHNDQUdzQjtBQUN0QixzREFNOEI7QUFDOUIsZ0VBSW1DO0FBQ25DLHNDQU1zQjtBQUN0QixzQ0FLc0I7QUFDdEIsd0NBR3VCO0FBQ3ZCLHNDQUE4RTtBQUM5RSw4Q0FLMEI7QUFDMUIsc0NBQTREO0FBQzVELG9EQU82QjtBQUM3QixzQ0FJc0I7QUFDdEIsOENBTTBCO0FBQzFCLG9EQVM2QjtBQUM3QixvREFBK0Q7QUFDL0Qsc0NBVXNCO0FBQ3RCLGdEQVcyQjtBQUMzQixvQ0FLcUI7QUFDckIsa0RBSTRCO0FBQzVCLDREQUF3RTtBQUN4RSxzQ0FBd0U7QUFDeEUsc0NBSXNCO0FBQ3RCLDBEQUdnQztBQUNoQyxvREFBMkU7QUFDM0Usc0NBQThEO0FBRTlEOzs7R0FHRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsa0JBQU87SUFDN0MsWUFBWSxLQUFvQjtRQUM5QixLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDYixJQUFJLENBQUMsUUFBUSxHQUFHLGNBQWMsQ0FBQztJQUNqQyxDQUFDO0lBQ00sS0FBSyxDQUFDLElBQWdCO1FBQzNCLElBQUksSUFBSSxZQUFZLHFCQUFXLEVBQUU7WUFDL0IsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN4QixJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3hCLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDMUIsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2hDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDaEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMxQixJQUFJLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDbkMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMzQixJQUFJLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDdkMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzlCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNoQztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssWUFBWSxDQUFDLElBQWlCO1FBQ3BDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSw0RUFBNEU7WUFDbEYsV0FBVyxFQUNULGlGQUFpRjtZQUNuRixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwrQ0FBNEI7WUFDbEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsMEVBQTBFO1lBQ2hGLFdBQVcsRUFDVCw4TEFBOEw7WUFDaE0sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0RBQXFDO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDRFQUE0RTtZQUNsRixXQUFXLEVBQ1QsNk9BQTZPO1lBQy9PLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLHNEQUFtQztZQUN6QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxpRUFBaUU7WUFDdkUsV0FBVyxFQUNULHFjQUFxYztZQUN2YyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQkFBb0I7WUFDMUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUseUNBQXlDO1lBQy9DLFdBQVcsRUFDVCw4TkFBOE47WUFDaE8sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkJBQXFCO1lBQzNCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGlEQUFpRDtZQUN2RCxXQUFXLEVBQ1QseUZBQXlGO1lBQzNGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlDQUEyQjtZQUNqQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSw4RkFBOEY7WUFDcEcsV0FBVyxFQUNULDZGQUE2RjtZQUMvRixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQ0FBb0M7WUFDMUMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsaUZBQWlGO1lBQ3ZGLFdBQVcsRUFDVCxxSEFBcUg7WUFDdkgsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsc0NBQWdDO1lBQ3RDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHdDQUF3QztZQUM5QyxXQUFXLEVBQ1QsdUZBQXVGO1lBQ3pGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLG1CQUFhO1lBQ25CLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHdHQUF3RztZQUM5RyxXQUFXLEVBQ1Qsb1RBQW9UO1lBQ3RULEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDZDQUF1QztZQUM3QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSw2REFBNkQ7WUFDbkUsV0FBVyxFQUNULGdKQUFnSjtZQUNsSixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyQ0FBcUM7WUFDM0MsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsK0VBQStFO1lBQ3JGLFdBQVcsRUFDVCx1RkFBdUY7WUFDekYsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsdUNBQWlDO1lBQ3ZDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDZFQUE2RTtZQUNuRixXQUFXLEVBQ1QseVFBQXlRO1lBQzNRLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUFnQztZQUN0QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwySEFBMkg7WUFDakksV0FBVyxFQUNULDhNQUE4TTtZQUNoTixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBMEI7WUFDaEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsb0VBQW9FO1lBQzFFLFdBQVcsRUFDVCw2S0FBNks7WUFDL0ssS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkJBQXFCO1lBQzNCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDRDQUE0QztZQUNsRCxXQUFXLEVBQ1QsMkdBQTJHO1lBQzdHLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHVCQUFpQjtZQUN2QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxvREFBb0Q7WUFDMUQsV0FBVyxFQUNULCtRQUErUTtZQUNqUixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyQkFBcUI7WUFDM0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsMkZBQTJGO1lBQ2pHLFdBQVcsRUFDVCw2T0FBNk87WUFDL08sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsc0NBQWdDO1lBQ3RDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGtHQUFrRztZQUN4RyxXQUFXLEVBQ1Qsa0lBQWtJO1lBQ3BJLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhCQUF3QjtZQUM5QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssWUFBWSxDQUFDLElBQWlCO1FBQ3BDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLElBQUksRUFBRSxnREFBZ0Q7WUFDdEQsV0FBVyxFQUNULGdJQUFnSTtZQUNsSSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyQkFBc0I7WUFDNUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixJQUFJLEVBQUUsbUVBQW1FO1lBQ3pFLFdBQVcsRUFDVCw2RkFBNkY7WUFDL0YsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQW1DO1lBQ3pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsSUFBSSxFQUFFLG9EQUFvRDtZQUMxRCxXQUFXLEVBQ1QsNEVBQTRFO1lBQzlFLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUFtQztZQUN6QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxxREFBcUQ7WUFDM0QsV0FBVyxFQUNULHFTQUFxUztZQUN2UyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw0QkFBdUI7WUFDN0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsbURBQW1EO1lBQ3pELFdBQVcsRUFDVCxrVkFBa1Y7WUFDcFYsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0JBQVk7WUFDbEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQjtRQUN0QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsaUZBQWlGO1lBQ3ZGLFdBQVcsRUFDVCx1SkFBdUo7WUFDekosS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUseUJBQW1CO1lBQ3pCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHdFQUF3RTtZQUM5RSxXQUFXLEVBQ1QsbUhBQW1IO1lBQ3JILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHVCQUFpQjtZQUN2QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSw2RkFBNkY7WUFDbkcsV0FBVyxFQUNULDBMQUEwTDtZQUM1TCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBMEI7WUFDaEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsZ0VBQWdFO1lBQ3RFLFdBQVcsRUFDVCxpTUFBaU07WUFDbk0sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMEJBQW9CO1lBQzFCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE9BQU87WUFDM0IsSUFBSSxFQUFFLGtGQUFrRjtZQUN4RixXQUFXLEVBQ1QsNkZBQTZGO1lBQy9GLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBDQUFvQztZQUMxQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxPQUFPO1lBQzNCLElBQUksRUFBRSx1RUFBdUU7WUFDN0UsV0FBVyxFQUNULHVNQUF1TTtZQUN6TSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1QkFBaUI7WUFDdkIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsT0FBTztZQUMzQixJQUFJLEVBQUUsMkRBQTJEO1lBQ2pFLFdBQVcsRUFBRSxxREFBcUQ7WUFDbEUsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOEJBQXdCO1lBQzlCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE9BQU87WUFDM0IsSUFBSSxFQUFFLCtEQUErRDtZQUNyRSxXQUFXLEVBQ1QsMkZBQTJGO1lBQzdGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBCQUFvQjtZQUMxQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxPQUFPO1lBQzNCLElBQUksRUFBRSxxRkFBcUY7WUFDM0YsV0FBVyxFQUNULGtLQUFrSztZQUNwSyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQ0FBb0M7WUFDMUMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsT0FBTztZQUMzQixJQUFJLEVBQUUsaUhBQWlIO1lBQ3ZILFdBQVcsRUFDVCxpR0FBaUc7WUFDbkcsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0JBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGtFQUFrRTtZQUN4RSxXQUFXLEVBQ1QsdVNBQXVTO1lBQ3pTLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLDhCQUFtQjtZQUN6QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwrREFBK0Q7WUFDckUsV0FBVyxFQUNULDBLQUEwSztZQUM1SyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1QkFBWTtZQUNsQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxzREFBc0Q7WUFDNUQsV0FBVyxFQUNULDJKQUEySjtZQUM3SixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxxQ0FBdUI7WUFDN0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsNkZBQTZGO1lBQ25HLFdBQVcsRUFDVCwySUFBMkk7WUFDN0ksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsK0NBQWlDO1lBQ3ZDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDRFQUE0RTtZQUNsRixXQUFXLEVBQ1QsbUZBQW1GO1lBQ3JGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDRDQUE4QjtZQUNwQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSx5REFBeUQ7WUFDL0QsV0FBVyxFQUNULHdLQUF3SztZQUMxSyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4Q0FBZ0M7WUFDdEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsZ0ZBQWdGO1lBQ3RGLFdBQVcsRUFDVCxzSkFBc0o7WUFDeEosS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOENBQWdDO1lBQ3RDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsSUFBSSxFQUFFLHFFQUFxRTtZQUMzRSxXQUFXLEVBQ1QsbUZBQW1GO1lBQ3JGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLCtCQUFxQjtZQUMzQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLElBQUksRUFBRSx1RUFBdUU7WUFDN0UsV0FBVyxFQUNULDBLQUEwSztZQUM1SyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxvREFBMEM7WUFDaEQsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixJQUFJLEVBQUUsK0ZBQStGO1lBQ3JHLFdBQVcsRUFDVCxzTEFBc0w7WUFDeEwsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkNBQW1DO1lBQ3pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsSUFBSSxFQUFFLGtFQUFrRTtZQUN4RSxXQUFXLEVBQ1QsNkhBQTZIO1lBQy9ILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUE4QjtZQUNwQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxJQUFJO1lBQ3hCLElBQUksRUFBRSwyRUFBMkU7WUFDakYsV0FBVyxFQUNULHFIQUFxSDtZQUN2SCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwrQkFBcUI7WUFDM0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsMkRBQTJEO1lBQ2pFLFdBQVcsRUFDVCwwSUFBMEk7WUFDNUksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsZ0NBQXFCO1lBQzNCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLG1EQUFtRDtZQUN6RCxXQUFXLEVBQ1QsOEpBQThKO1lBQ2hLLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLCtCQUFvQjtZQUMxQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSwyREFBMkQ7WUFDakUsV0FBVyxFQUNULGdLQUFnSztZQUNsSyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw0Q0FBaUM7WUFDdkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsc0RBQXNEO1lBQzVELFdBQVcsRUFDVCxrSkFBa0o7WUFDcEosS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQTZCO1lBQ25DLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDJEQUEyRDtZQUNqRSxXQUFXLEVBQ1Qsd0VBQXdFO1lBQzFFLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUEyQjtZQUNqQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxnRUFBZ0U7WUFDdEUsV0FBVyxFQUFFLHdEQUF3RDtZQUNyRSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQ0FBK0I7WUFDckMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsOENBQThDO1lBQ3BELFdBQVcsRUFDVCwrREFBK0Q7WUFDakUsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsc0NBQTJCO1lBQ2pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDZEQUE2RDtZQUNuRSxXQUFXLEVBQ1QscUhBQXFIO1lBQ3ZILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUE2QjtZQUNuQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwyRkFBMkY7WUFDakcsV0FBVyxFQUNULHdKQUF3SjtZQUMxSixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBcUI7WUFDM0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsbUVBQW1FO1lBQ3pFLFdBQVcsRUFDVCw2TkFBNk47WUFDL04sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkNBQWtDO1lBQ3hDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLG1FQUFtRTtZQUN6RSxXQUFXLEVBQ1QsNkhBQTZIO1lBQy9ILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhDQUFpQztZQUN2QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSx5REFBeUQ7WUFDL0QsV0FBVyxFQUNULGtKQUFrSjtZQUNwSixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw0Q0FBK0I7WUFDckMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsNEZBQTRGO1lBQ2xHLFdBQVcsRUFDVCx3aUJBQXdpQjtZQUMxaUIsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0RBQXFDO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGdHQUFnRztZQUN0RyxXQUFXLEVBQ1Qsc0xBQXNMO1lBQ3hMLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLG1EQUFzQztZQUM1QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwwR0FBMEc7WUFDaEgsV0FBVyxFQUNULG9TQUFvUztZQUN0UyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx3Q0FBMkI7WUFDakMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUseUZBQXlGO1lBQy9GLFdBQVcsRUFDVCwyVUFBMlU7WUFDN1UsS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsaURBQW9DO1lBQzFDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxvQkFBb0IsQ0FBQyxJQUFpQjtRQUM1QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsMERBQTBEO1lBQ2hFLFdBQVcsRUFDVCxzVEFBc1Q7WUFDeFQsS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsZ0JBQVU7WUFDaEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsK0NBQStDO1lBQ3JELFdBQVcsRUFDVCw4VUFBOFU7WUFDaFYsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0JBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDJEQUEyRDtZQUNqRSxXQUFXLEVBQ1QsbUxBQW1MO1lBQ3JMLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLGtEQUFxQztZQUMzQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxtRUFBbUU7WUFDekUsV0FBVyxFQUNULG1LQUFtSztZQUNySyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSxpREFBb0M7WUFDMUMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsb0VBQW9FO1lBQzFFLFdBQVcsRUFDVCxnSUFBZ0k7WUFDbEksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsZ0RBQW1DO1lBQ3pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLG1GQUFtRjtZQUN6RixXQUFXLEVBQ1QsdU9BQXVPO1lBQ3pPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdEQUFtQztZQUN6QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxrRkFBa0Y7WUFDeEYsV0FBVyxFQUNULDBHQUEwRztZQUM1RyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx5REFBNEM7WUFDbEQsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsT0FBTztZQUMzQixJQUFJLEVBQUUsK0NBQStDO1lBQ3JELFdBQVcsRUFDVCxvR0FBb0c7WUFDdEcsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMEJBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE9BQU87WUFDM0IsSUFBSSxFQUFFLHdEQUF3RDtZQUM5RCxXQUFXLEVBQ1QseU1BQXlNO1lBQzNNLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhCQUFzQjtZQUM1QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxPQUFPO1lBQzNCLElBQUksRUFBRSw4REFBOEQ7WUFDcEUsV0FBVyxFQUNULDhPQUE4TztZQUNoUCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSw4QkFBc0I7WUFDNUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsT0FBTztZQUMzQixJQUFJLEVBQUUsMkNBQTJDO1lBQ2pELFdBQVcsRUFDVCx5TEFBeUw7WUFDM0wsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMEJBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE9BQU87WUFDM0IsSUFBSSxFQUFFLDhFQUE4RTtZQUNwRixXQUFXLEVBQ1QsNkhBQTZIO1lBQy9ILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLG9DQUE0QjtZQUNsQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0sseUJBQXlCLENBQUMsSUFBaUI7UUFDakQsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLHlEQUF5RDtZQUMvRCxXQUFXLEVBQ1QseUtBQXlLO1lBQzNLLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDRDQUE4QjtZQUNwQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSx5RUFBeUU7WUFDL0UsV0FBVyxFQUNULGlIQUFpSDtZQUNuSCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx5Q0FBMkI7WUFDakMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsdUZBQXVGO1lBQzdGLFdBQVcsRUFDVCxzTEFBc0w7WUFDeEwsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0RBQW9DO1lBQzFDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxvQkFBb0IsQ0FBQyxJQUFpQjtRQUM1QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsa0VBQWtFO1lBQ3hFLFdBQVcsRUFDVCw4SEFBOEg7WUFDaEksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0NBQXNCO1lBQzVCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDRFQUE0RTtZQUNsRixXQUFXLEVBQ1Qsa0VBQWtFO1lBQ3BFLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFEQUF5QztZQUMvQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxxRUFBcUU7WUFDM0UsV0FBVyxFQUNULCtEQUErRDtZQUNqRSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxtREFBdUM7WUFDN0MsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQjtRQUN0QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsc0RBQXNEO1lBQzVELFdBQVcsRUFDVCxzSEFBc0g7WUFDeEgsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkNBQW9DO1lBQzFDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLG1EQUFtRDtZQUN6RCxXQUFXLEVBQ1QseUlBQXlJO1lBQzNJLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdCQUFrQjtZQUN4QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwyRkFBMkY7WUFDakcsV0FBVyxFQUNULHFIQUFxSDtZQUN2SCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw0QkFBc0I7WUFDNUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsMEdBQTBHO1lBQ2hILFdBQVcsRUFDVCx3R0FBd0c7WUFDMUcsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNEJBQXNCO1lBQzVCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLG9GQUFvRjtZQUMxRixXQUFXLEVBQ1QsbU9BQW1PO1lBQ3JPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlDQUEyQjtZQUNqQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSx1R0FBdUc7WUFDN0csV0FBVyxFQUNULG9JQUFvSTtZQUN0SSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSxrQ0FBMkI7WUFDakMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsK0ZBQStGO1lBQ3JHLFdBQVcsRUFDVCw0RkFBNEY7WUFDOUYsS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsK0JBQXdCO1lBQzlCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLG1GQUFtRjtZQUN6RixXQUFXLEVBQ1QsMkhBQTJIO1lBQzdILEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLGdEQUFzQztZQUM1QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxxRkFBcUY7WUFDM0YsV0FBVyxFQUNULHdHQUF3RztZQUMxRyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBc0I7WUFDNUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsc0VBQXNFO1lBQzVFLFdBQVcsRUFDVCxvT0FBb087WUFDdE8sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOEJBQW9CO1lBQzFCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGtHQUFrRztZQUN4RyxXQUFXLEVBQ1Qsa1FBQWtRO1lBQ3BRLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLDRDQUFrQztZQUN4QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwyRUFBMkU7WUFDakYsV0FBVyxFQUNULDhHQUE4RztZQUNoSCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQkFBb0I7WUFDMUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsK0RBQStEO1lBQ3JFLFdBQVcsRUFDVCx5RkFBeUY7WUFDM0YsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMEJBQW9CO1lBQzFCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHVFQUF1RTtZQUM3RSxXQUFXLEVBQ1QsMFFBQTBRO1lBQzVRLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNCQUFnQjtZQUN0QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxnRUFBZ0U7WUFDdEUsV0FBVyxFQUNULDhQQUE4UDtZQUNoUSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBbUI7WUFDekIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsOEVBQThFO1lBQ3BGLFdBQVcsRUFDVCx5TUFBeU07WUFDM00sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkNBQThCO1lBQ3BDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLHdGQUF3RjtZQUM5RixXQUFXLEVBQ1Qsc0ZBQXNGO1lBQ3hGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFDQUF3QjtZQUM5QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxvRUFBb0U7WUFDMUUsV0FBVyxFQUNULGdJQUFnSTtZQUNsSSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwQ0FBNkI7WUFDbkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUseUZBQXlGO1lBQy9GLFdBQVcsRUFDVCw2SEFBNkg7WUFDL0gsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsa0RBQXFDO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLHFFQUFxRTtZQUMzRSxXQUFXLEVBQ1QsdVJBQXVSO1lBQ3pSLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLG9DQUF1QjtZQUM3QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSx5RUFBeUU7WUFDL0UsV0FBVyxFQUNULDZIQUE2SDtZQUMvSCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxzQ0FBeUI7WUFDL0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsbUhBQW1IO1lBQ3pILFdBQVcsRUFDVCw0RkFBNEY7WUFDOUYsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkNBQThCO1lBQ3BDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLHNFQUFzRTtZQUM1RSxXQUFXLEVBQ1Qsc01BQXNNO1lBQ3hNLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFDQUF3QjtZQUM5QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssdUJBQXVCLENBQUMsSUFBaUI7UUFDL0MsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHlEQUF5RDtZQUMvRCxXQUFXLEVBQ1QsMk9BQTJPO1lBQzdPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBCQUFvQjtZQUMxQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwrSEFBK0g7WUFDckksV0FBVyxFQUNULDZIQUE2SDtZQUMvSCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4QkFBd0I7WUFDOUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsMkxBQTJMO1lBQ2pNLFdBQVcsRUFDVCxpSEFBaUg7WUFDbkgsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkNBQW1DO1lBQ3pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDZDQUE2QztZQUNuRCxXQUFXLEVBQ1QscUtBQXFLO1lBQ3ZLLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLDRCQUFrQjtZQUN4QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSwyRUFBMkU7WUFDakYsV0FBVyxFQUNULGlHQUFpRztZQUNuRyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxxREFBMkM7WUFDakQsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsZ0VBQWdFO1lBQ3RFLFdBQVcsRUFDVCxpTEFBaUw7WUFDbkwsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQThCO1lBQ3BDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHlIQUF5SDtZQUMvSCxXQUFXLEVBQ1QsNkpBQTZKO1lBQy9KLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdEQUFzQztZQUM1QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxNQUFNO1lBQzFCLElBQUksRUFBRSxxRUFBcUU7WUFDM0UsV0FBVyxFQUNULG9QQUFvUDtZQUN0UCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxrQ0FBNEI7WUFDbEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsd0RBQXdEO1lBQzlELFdBQVcsRUFDVCw2R0FBNkc7WUFDL0csS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOENBQTZCO1lBQ25DLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsSUFBaUI7UUFDdkMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLGlFQUFpRTtZQUN2RSxXQUFXLEVBQ1Qsd0pBQXdKO1lBQzFKLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUE0QjtZQUNsQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxpRUFBaUU7WUFDdkUsV0FBVyxFQUNULHdGQUF3RjtZQUMxRixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyREFBMkM7WUFDakQsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsd0RBQXdEO1lBQzlELFdBQVcsRUFDVCxpTkFBaU47WUFDbk4sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNENBQTRCO1lBQ2xDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSywyQkFBMkIsQ0FBQyxJQUFpQjtRQUNuRCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsNkRBQTZEO1lBQ25FLFdBQVcsRUFDVCwyR0FBMkc7WUFDN0csS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUscUJBQWU7WUFDckIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUscURBQXFEO1lBQzNELFdBQVcsRUFDVCx1VkFBdVY7WUFDelYsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkJBQXVCO1lBQzdCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDZEQUE2RDtZQUNuRSxXQUFXLEVBQ1QsMkdBQTJHO1lBQzdHLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlCQUFXO1lBQ2pCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLDBIQUEwSDtZQUNoSSxXQUFXLEVBQ1QsdUlBQXVJO1lBQ3pJLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlCQUFXO1lBQ2pCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLE1BQU07WUFDMUIsSUFBSSxFQUFFLHFEQUFxRDtZQUMzRCxXQUFXLEVBQ1QseVJBQXlSO1lBQzNSLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDZCQUF1QjtZQUM3QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssa0JBQWtCLENBQUMsSUFBaUI7UUFDMUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDBFQUEwRTtZQUNoRixXQUFXLEVBQ1QsK0hBQStIO1lBQ2pJLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDZDQUFnQztZQUN0QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSxnRUFBZ0U7WUFDdEUsV0FBVyxFQUNULHlTQUF5UztZQUMzUyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwrQ0FBa0M7WUFDeEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUseUZBQXlGO1lBQy9GLFdBQVcsRUFDVCxpSUFBaUk7WUFDbkksS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsNkNBQWdDO1lBQ3RDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLDhEQUE4RDtZQUNwRSxXQUFXLEVBQ1Qsb0xBQW9MO1lBQ3RMLEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLGtEQUFxQztZQUMzQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSx5REFBeUQ7WUFDL0QsV0FBVyxFQUNULCtLQUErSztZQUNqTCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSwwQ0FBNkI7WUFDbkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsTUFBTTtZQUMxQixJQUFJLEVBQUUsOERBQThEO1lBQ3BFLFdBQVcsRUFDVCxpSEFBaUg7WUFDbkgsS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsK0NBQWtDO1lBQ3hDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxtQkFBbUIsQ0FBQyxJQUFpQjtRQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsb0RBQW9EO1lBQzFELFdBQVcsRUFDVCxvUUFBb1E7WUFDdFEsS0FBSyxFQUFFLDBCQUFlLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsa0RBQXNDO1lBQzVDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLGtCQUFrQixFQUFFLEtBQUs7WUFDekIsSUFBSSxFQUFFLG1FQUFtRTtZQUN6RSxXQUFXLEVBQ1QsaUlBQWlJO1lBQ25JLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlEQUFxQztZQUMzQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixrQkFBa0IsRUFBRSxLQUFLO1lBQ3pCLElBQUksRUFBRSx3SkFBd0o7WUFDOUosV0FBVyxFQUNULHNHQUFzRztZQUN4RyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSx5Q0FBNkI7WUFDbkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2Isa0JBQWtCLEVBQUUsS0FBSztZQUN6QixJQUFJLEVBQUUsc0ZBQXNGO1lBQzVGLFdBQVcsRUFDVCx5SkFBeUo7WUFDM0osS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsOENBQXFDO1lBQzNDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUFwd0NILGdEQXF3Q0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IENmblJlc291cmNlLCBJQ29uc3RydWN0IH0gZnJvbSAnbW9ub2Nkayc7XG5pbXBvcnQgeyBOYWdQYWNrLCBOYWdNZXNzYWdlTGV2ZWwsIE5hZ1BhY2tQcm9wcyB9IGZyb20gJy4uL25hZy1wYWNrJztcbmltcG9ydCB7XG4gIEFQSUdXQWNjZXNzTG9nZ2luZyxcbiAgQVBJR1dBc3NvY2lhdGVkV2l0aFdBRixcbiAgQVBJR1dBdXRob3JpemF0aW9uLFxuICBBUElHV0V4ZWN1dGlvbkxvZ2dpbmdFbmFibGVkLFxuICBBUElHV1JlcXVlc3RWYWxpZGF0aW9uLFxufSBmcm9tICcuLi9ydWxlcy9hcGlndyc7XG5pbXBvcnQgeyBBcHBTeW5jR3JhcGhRTFJlcXVlc3RMb2dnaW5nIH0gZnJvbSAnLi4vcnVsZXMvYXBwc3luYyc7XG5pbXBvcnQgeyBBdGhlbmFXb3JrZ3JvdXBFbmNyeXB0ZWRRdWVyeVJlc3VsdHMgfSBmcm9tICcuLi9ydWxlcy9hdGhlbmEnO1xuaW1wb3J0IHtcbiAgQXV0b1NjYWxpbmdHcm91cENvb2xkb3duUGVyaW9kLFxuICBBdXRvU2NhbGluZ0dyb3VwSGVhbHRoQ2hlY2ssXG4gIEF1dG9TY2FsaW5nR3JvdXBTY2FsaW5nTm90aWZpY2F0aW9ucyxcbn0gZnJvbSAnLi4vcnVsZXMvYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHsgQ2xvdWQ5SW5zdGFuY2VOb0luZ3Jlc3NTeXN0ZW1zTWFuYWdlciB9IGZyb20gJy4uL3J1bGVzL2Nsb3VkOSc7XG5pbXBvcnQge1xuICBDbG91ZEZyb250RGlzdHJpYnV0aW9uQWNjZXNzTG9nZ2luZyxcbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkdlb1Jlc3RyaWN0aW9ucyxcbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk5vT3V0ZGF0ZWRTU0wsXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25TM09yaWdpbkFjY2Vzc0lkZW50aXR5LFxuICBDbG91ZEZyb250RGlzdHJpYnV0aW9uV0FGSW50ZWdyYXRpb24sXG59IGZyb20gJy4uL3J1bGVzL2Nsb3VkZnJvbnQnO1xuaW1wb3J0IHtcbiAgQ29kZUJ1aWxkUHJvamVjdEtNU0VuY3J5cHRlZEFydGlmYWN0cyxcbiAgQ29kZUJ1aWxkUHJvamVjdE1hbmFnZWRJbWFnZXMsXG4gIENvZGVCdWlsZFByb2plY3RQcml2aWxlZ2VkTW9kZURpc2FibGVkLFxufSBmcm9tICcuLi9ydWxlcy9jb2RlYnVpbGQnO1xuaW1wb3J0IHtcbiAgQ29nbml0b1VzZXJQb29sQWR2YW5jZWRTZWN1cml0eU1vZGVFbmZvcmNlZCxcbiAgQ29nbml0b1VzZXJQb29sQVBJR1dBdXRob3JpemVyLFxuICBDb2duaXRvVXNlclBvb2xNRkEsXG4gIENvZ25pdG9Vc2VyUG9vbE5vVW5hdXRoZW50aWNhdGVkTG9naW5zLFxuICBDb2duaXRvVXNlclBvb2xTdHJvbmdQYXNzd29yZFBvbGljeSxcbn0gZnJvbSAnLi4vcnVsZXMvY29nbml0byc7XG5pbXBvcnQge1xuICBEb2N1bWVudERCQ2x1c3RlckJhY2t1cFJldGVudGlvblBlcmlvZCxcbiAgRG9jdW1lbnREQkNsdXN0ZXJFbmNyeXB0aW9uQXRSZXN0LFxuICBEb2N1bWVudERCQ2x1c3RlckxvZ0V4cG9ydHMsXG4gIERvY3VtZW50REJDbHVzdGVyTm9uRGVmYXVsdFBvcnQsXG4gIERvY3VtZW50REJDcmVkZW50aWFsc0luU2VjcmV0c01hbmFnZXIsXG59IGZyb20gJy4uL3J1bGVzL2RvY3VtZW50ZGInO1xuaW1wb3J0IHsgREFYRW5jcnlwdGVkLCBEeW5hbW9EQlBJVFJFbmFibGVkIH0gZnJvbSAnLi4vcnVsZXMvZHluYW1vZGInO1xuaW1wb3J0IHtcbiAgRUMyRUJTVm9sdW1lRW5jcnlwdGVkLFxuICBFQzJJbnN0YW5jZURldGFpbGVkTW9uaXRvcmluZ0VuYWJsZWQsXG4gIEVDMkluc3RhbmNlVGVybWluYXRpb25Qcm90ZWN0aW9uLFxuICBFQzJSZXN0cmljdGVkSW5ib3VuZCxcbiAgRUMyU2VjdXJpdHlHcm91cERlc2NyaXB0aW9uLFxufSBmcm9tICcuLi9ydWxlcy9lYzInO1xuaW1wb3J0IHsgRUNST3BlbkFjY2VzcyB9IGZyb20gJy4uL3J1bGVzL2Vjcic7XG5pbXBvcnQge1xuICBFQ1NDbHVzdGVyQ2xvdWRXYXRjaENvbnRhaW5lckluc2lnaHRzLFxuICBFQ1NUYXNrRGVmaW5pdGlvbkNvbnRhaW5lckxvZ2dpbmcsXG4gIEVDU1Rhc2tEZWZpbml0aW9uTm9FbnZpcm9ubWVudFZhcmlhYmxlcyxcbn0gZnJvbSAnLi4vcnVsZXMvZWNzJztcbmltcG9ydCB7IEVGU0VuY3J5cHRlZCB9IGZyb20gJy4uL3J1bGVzL2Vmcyc7XG5pbXBvcnQge1xuICBFS1NDbHVzdGVyQ29udHJvbFBsYW5lTG9ncyxcbiAgRUtTQ2x1c3Rlck5vRW5kcG9pbnRQdWJsaWNBY2Nlc3MsXG59IGZyb20gJy4uL3J1bGVzL2Vrcyc7XG5pbXBvcnQge1xuICBFbGFzdGlDYWNoZUNsdXN0ZXJJblZQQyxcbiAgRWxhc3RpQ2FjaGVDbHVzdGVyTm9uRGVmYXVsdFBvcnQsXG4gIEVsYXN0aUNhY2hlUmVkaXNDbHVzdGVyRW5jcnlwdGlvbixcbiAgRWxhc3RpQ2FjaGVSZWRpc0NsdXN0ZXJNdWx0aUFaLFxuICBFbGFzdGlDYWNoZVJlZGlzQ2x1c3RlclJlZGlzQXV0aCxcbn0gZnJvbSAnLi4vcnVsZXMvZWxhc3RpY2FjaGUnO1xuaW1wb3J0IHtcbiAgRWxhc3RpY0JlYW5zdGFsa0VDMkluc3RhbmNlTG9nc1RvUzMsXG4gIEVsYXN0aWNCZWFuc3RhbGtNYW5hZ2VkVXBkYXRlc0VuYWJsZWQsXG4gIEVsYXN0aWNCZWFuc3RhbGtWUENTcGVjaWZpZWQsXG59IGZyb20gJy4uL3J1bGVzL2VsYXN0aWNiZWFuc3RhbGsnO1xuaW1wb3J0IHtcbiAgQ0xCQ29ubmVjdGlvbkRyYWluaW5nLFxuICBDTEJOb0luYm91bmRIdHRwSHR0cHMsXG4gIEVMQkNyb3NzWm9uZUxvYWRCYWxhbmNpbmdFbmFibGVkLFxuICBFTEJMb2dnaW5nRW5hYmxlZCxcbiAgRUxCVGxzSHR0cHNMaXN0ZW5lcnNPbmx5LFxufSBmcm9tICcuLi9ydWxlcy9lbGInO1xuaW1wb3J0IHtcbiAgRU1SQXV0aEVDMktleVBhaXJPcktlcmJlcm9zLFxuICBFTVJFbmNyeXB0aW9uSW5UcmFuc2l0LFxuICBFTVJMb2NhbERpc2tFbmNyeXB0aW9uLFxuICBFTVJTM0FjY2Vzc0xvZ2dpbmcsXG59IGZyb20gJy4uL3J1bGVzL2Vtcic7XG5pbXBvcnQge1xuICBHbHVlRW5jcnlwdGVkQ2xvdWRXYXRjaExvZ3MsXG4gIEdsdWVKb2JCb29rbWFya0VuY3J5cHRlZCxcbn0gZnJvbSAnLi4vcnVsZXMvZ2x1ZSc7XG5pbXBvcnQgeyBJQU1Ob01hbmFnZWRQb2xpY2llcywgSUFNTm9XaWxkY2FyZFBlcm1pc3Npb25zIH0gZnJvbSAnLi4vcnVsZXMvaWFtJztcbmltcG9ydCB7XG4gIEtpbmVzaXNEYXRhQW5hbHl0aWNzRmxpbmtDaGVja3BvaW50aW5nLFxuICBLaW5lc2lzRGF0YUZpcmVob3NlU1NFLFxuICBLaW5lc2lzRGF0YVN0cmVhbURlZmF1bHRLZXlXaGVuU1NFLFxuICBLaW5lc2lzRGF0YVN0cmVhbVNTRSxcbn0gZnJvbSAnLi4vcnVsZXMva2luZXNpcyc7XG5pbXBvcnQgeyBLTVNCYWNraW5nS2V5Um90YXRpb25FbmFibGVkIH0gZnJvbSAnLi4vcnVsZXMva21zJztcbmltcG9ydCB7XG4gIE1lZGlhU3RvcmVDbG91ZFdhdGNoTWV0cmljUG9saWN5LFxuICBNZWRpYVN0b3JlQ29udGFpbmVyQWNjZXNzTG9nZ2luZyxcbiAgTWVkaWFTdG9yZUNvbnRhaW5lckNPUlNQb2xpY3ksXG4gIE1lZGlhU3RvcmVDb250YWluZXJIYXNDb250YWluZXJQb2xpY3ksXG4gIE1lZGlhU3RvcmVDb250YWluZXJMaWZlY3ljbGVQb2xpY3ksXG4gIE1lZGlhU3RvcmVDb250YWluZXJTU0xSZXF1ZXN0c09ubHksXG59IGZyb20gJy4uL3J1bGVzL21lZGlhc3RvcmUnO1xuaW1wb3J0IHtcbiAgTVNLQnJva2VyTG9nZ2luZyxcbiAgTVNLQnJva2VyVG9Ccm9rZXJUTFMsXG4gIE1TS0NsaWVudFRvQnJva2VyVExTLFxufSBmcm9tICcuLi9ydWxlcy9tc2snO1xuaW1wb3J0IHtcbiAgTmVwdHVuZUNsdXN0ZXJBdXRvbWF0aWNNaW5vclZlcnNpb25VcGdyYWRlLFxuICBOZXB0dW5lQ2x1c3RlckJhY2t1cFJldGVudGlvblBlcmlvZCxcbiAgTmVwdHVuZUNsdXN0ZXJFbmNyeXB0aW9uQXRSZXN0LFxuICBOZXB0dW5lQ2x1c3RlcklBTUF1dGgsXG4gIE5lcHR1bmVDbHVzdGVyTXVsdGlBWixcbn0gZnJvbSAnLi4vcnVsZXMvbmVwdHVuZSc7XG5pbXBvcnQge1xuICBPcGVuU2VhcmNoQWxsb3dsaXN0ZWRJUHMsXG4gIE9wZW5TZWFyY2hEZWRpY2F0ZWRNYXN0ZXJOb2RlLFxuICBPcGVuU2VhcmNoRW5jcnlwdGVkQXRSZXN0LFxuICBPcGVuU2VhcmNoSW5WUENPbmx5LFxuICBPcGVuU2VhcmNoTm9kZVRvTm9kZUVuY3J5cHRpb24sXG4gIE9wZW5TZWFyY2hOb1Vuc2lnbmVkT3JBbm9ueW1vdXNBY2Nlc3MsXG4gIE9wZW5TZWFyY2hTbG93TG9nc1RvQ2xvdWRXYXRjaCxcbiAgT3BlblNlYXJjaFpvbmVBd2FyZW5lc3MsXG59IGZyb20gJy4uL3J1bGVzL29wZW5zZWFyY2gnO1xuaW1wb3J0IHsgUXVpY2tzaWdodFNTTENvbm5lY3Rpb25zIH0gZnJvbSAnLi4vcnVsZXMvcXVpY2tzaWdodCc7XG5pbXBvcnQge1xuICBBdXJvcmFNeVNRTEJhY2t0cmFjayxcbiAgQXVyb3JhTXlTUUxMb2dnaW5nLFxuICBBdXJvcmFNeVNRTFBvc3RncmVzSUFNQXV0aCxcbiAgUkRTSW5zdGFuY2VCYWNrdXBFbmFibGVkLFxuICBSRFNJbnN0YW5jZURlbGV0aW9uUHJvdGVjdGlvbkVuYWJsZWQsXG4gIFJEU011bHRpQVpTdXBwb3J0LFxuICBSRFNOb25EZWZhdWx0UG9ydCxcbiAgUkRTUmVzdHJpY3RlZEluYm91bmQsXG4gIFJEU1N0b3JhZ2VFbmNyeXB0ZWQsXG59IGZyb20gJy4uL3J1bGVzL3Jkcyc7XG5pbXBvcnQge1xuICBSZWRzaGlmdENsdXN0ZXJBdWRpdExvZ2dpbmcsXG4gIFJlZHNoaWZ0QmFja3VwRW5hYmxlZCxcbiAgUmVkc2hpZnRDbHVzdGVyRW5jcnlwdGlvbkF0UmVzdCxcbiAgUmVkc2hpZnRDbHVzdGVySW5WUEMsXG4gIFJlZHNoaWZ0Q2x1c3Rlck5vbkRlZmF1bHRQb3J0LFxuICBSZWRzaGlmdENsdXN0ZXJOb25EZWZhdWx0VXNlcm5hbWUsXG4gIFJlZHNoaWZ0Q2x1c3RlclB1YmxpY0FjY2VzcyxcbiAgUmVkc2hpZnRDbHVzdGVyVXNlckFjdGl2aXR5TG9nZ2luZyxcbiAgUmVkc2hpZnRDbHVzdGVyVmVyc2lvblVwZ3JhZGUsXG4gIFJlZHNoaWZ0UmVxdWlyZVRsc1NTTCxcbn0gZnJvbSAnLi4vcnVsZXMvcmVkc2hpZnQnO1xuaW1wb3J0IHtcbiAgUzNCdWNrZXRMZXZlbFB1YmxpY0FjY2Vzc1Byb2hpYml0ZWQsXG4gIFMzQnVja2V0TG9nZ2luZ0VuYWJsZWQsXG4gIFMzQnVja2V0U2VydmVyU2lkZUVuY3J5cHRpb25FbmFibGVkLFxuICBTM0J1Y2tldFNTTFJlcXVlc3RzT25seSxcbn0gZnJvbSAnLi4vcnVsZXMvczMnO1xuaW1wb3J0IHtcbiAgU2FnZU1ha2VyTm90ZWJvb2tJbnN0YW5jZUtNU0tleUNvbmZpZ3VyZWQsXG4gIFNhZ2VNYWtlck5vdGVib29rSW5WUEMsXG4gIFNhZ2VNYWtlck5vdGVib29rTm9EaXJlY3RJbnRlcm5ldEFjY2Vzcyxcbn0gZnJvbSAnLi4vcnVsZXMvc2FnZW1ha2VyJztcbmltcG9ydCB7IFNlY3JldHNNYW5hZ2VyUm90YXRpb25FbmFibGVkIH0gZnJvbSAnLi4vcnVsZXMvc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHsgU05TRW5jcnlwdGVkS01TLCBTTlNUb3BpY1NTTFJlcXVlc3RzT25seSB9IGZyb20gJy4uL3J1bGVzL3Nucyc7XG5pbXBvcnQge1xuICBTUVNRdWV1ZURMUSxcbiAgU1FTUXVldWVTU0UsXG4gIFNRU1F1ZXVlU1NMUmVxdWVzdHNPbmx5LFxufSBmcm9tICcuLi9ydWxlcy9zcXMnO1xuaW1wb3J0IHtcbiAgU3RlcEZ1bmN0aW9uU3RhdGVNYWNoaW5lQWxsTG9nc1RvQ2xvdWRXYXRjaCxcbiAgU3RlcEZ1bmN0aW9uU3RhdGVNYWNoaW5lWHJheSxcbn0gZnJvbSAnLi4vcnVsZXMvc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBUaW1lc3RyZWFtRGF0YWJhc2VDdXN0b21lck1hbmFnZWRLZXkgfSBmcm9tICcuLi9ydWxlcy90aW1lc3RyZWFtJztcbmltcG9ydCB7IFZQQ0Zsb3dMb2dzRW5hYmxlZCwgVlBDTm9OQUNMcyB9IGZyb20gJy4uL3J1bGVzL3ZwYyc7XG5cbi8qKlxuICogQ2hlY2sgQmVzdCBwcmFjdGljZXMgYmFzZWQgb24gQVdTIFNvbHV0aW9ucyBTZWN1cml0eSBNYXRyaXhcbiAqXG4gKi9cbmV4cG9ydCBjbGFzcyBBd3NTb2x1dGlvbnNDaGVja3MgZXh0ZW5kcyBOYWdQYWNrIHtcbiAgY29uc3RydWN0b3IocHJvcHM/OiBOYWdQYWNrUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgdGhpcy5wYWNrTmFtZSA9ICdBd3NTb2x1dGlvbnMnO1xuICB9XG4gIHB1YmxpYyB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5SZXNvdXJjZSkge1xuICAgICAgdGhpcy5jaGVja0NvbXB1dGUobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrU3RvcmFnZShub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tEYXRhYmFzZXMobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrTmV0d29ya0RlbGl2ZXJ5KG5vZGUpO1xuICAgICAgdGhpcy5jaGVja01hbmFnZW1lbnRHb3Zlcm5hbmNlKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja01hY2hpbmVMZWFybmluZyhub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tBbmFseXRpY3Mobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrU2VjdXJpdHlDb21wbGlhbmNlKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja1NlcnZlcmxlc3Mobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrQXBwbGljYXRpb25JbnRlZ3JhdGlvbihub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tNZWRpYVNlcnZpY2VzKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja0RldmVsb3BlclRvb2xzKG5vZGUpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBDb21wdXRlIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0NvbXB1dGUobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFQjEnLFxuICAgICAgaW5mbzogJ1RoZSBFbGFzdGljIEJlYW5zdGFsayBlbnZpcm9ubWVudCBpcyBub3QgY29uZmlndXJlZCB0byB1c2UgYSBzcGVjaWZpYyBWUEMuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVXNlIGEgbm9uLWRlZmF1bHQgaW4gb3JkZXIgdG8gc2VwZXJhdGUgeW91ciBlbnZpcm9ubWVudCBmcm9tIGRlZmF1bHQgcmVzb3VyY2VzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRWxhc3RpY0JlYW5zdGFsa1ZQQ1NwZWNpZmllZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUIzJyxcbiAgICAgIGluZm86ICdUaGUgRWxhc3RpYyBCZWFuc3RhbGsgZW52aXJvbm1lbnQgZG9lcyBub3QgaGF2ZSBtYW5hZ2VkIHVwZGF0ZXMgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmFibGUgbWFuYWdlZCBwbGF0Zm9ybSB1cGRhdGVzIGZvciBiZWFuc3RhbGsgZW52aXJvbm1lbnRzIGluIG9yZGVyIHRvIHJlY2VpdmUgYnVnIGZpeGVzLCBzb2Z0d2FyZSB1cGRhdGVzIGFuZCBuZXcgZmVhdHVyZXMuIE1hbmFnZWQgcGxhdGZvcm0gdXBkYXRlcyBwZXJmb3JtIGltbXV0YWJsZSBlbnZpcm9ubWVudCB1cGRhdGVzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRWxhc3RpY0JlYW5zdGFsa01hbmFnZWRVcGRhdGVzRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUI0JyxcbiAgICAgIGluZm86ICdUaGUgRWxhc3RpYyBCZWFuc3RhbGsgZW52aXJvbm1lbnQgZG9lcyBub3QgdXBsb2FkIEVDMiBJbnN0YW5jZSBsb2dzIHRvIFMzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlYW5zdGFsayBlbnZpcm9ubWVudCBsb2dzIHNob3VsZCBiZSByZXRhaW5lZCBhbmQgdXBsb2FkZWQgdG8gQW1hem9uIFMzIGluIG9yZGVyIHRvIGtlZXAgdGhlIGxvZ2dpbmcgZGF0YSBmb3IgZnV0dXJlIGF1ZGl0cywgaGlzdG9yaWNhbCBwdXJwb3NlcyBvciB0byB0cmFjayBhbmQgYW5hbHl6ZSB0aGUgRUIgYXBwbGljYXRpb24gZW52aXJvbm1lbnQgYmVoYXZpb3IgZm9yIGEgbG9uZyBwZXJpb2Qgb2YgdGltZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogRWxhc3RpY0JlYW5zdGFsa0VDMkluc3RhbmNlTG9nc1RvUzMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0VDMjMnLFxuICAgICAgaW5mbzogJ1RoZSBTZWN1cml0eSBHcm91cCBhbGxvd3MgZm9yIDAuMC4wLjAvMCBvciA6Oi8wIGluYm91bmQgYWNjZXNzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0xhcmdlIHBvcnQgcmFuZ2VzLCB3aGVuIG9wZW4sIGV4cG9zZSBpbnN0YW5jZXMgdG8gdW53YW50ZWQgYXR0YWNrcy4gTW9yZSB0aGFuIHRoYXQsIHRoZXkgbWFrZSB0cmFjZWFiaWxpdHkgb2YgdnVsbmVyYWJpbGl0aWVzIHZlcnkgZGlmZmljdWx0LiBGb3IgaW5zdGFuY2UsIHlvdXIgd2ViIHNlcnZlcnMgbWF5IG9ubHkgcmVxdWlyZSA4MCBhbmQgNDQzIHBvcnRzIHRvIGJlIG9wZW4sIGJ1dCBub3QgYWxsLiBPbmUgb2YgdGhlIG1vc3QgY29tbW9uIG1pc3Rha2VzIG9ic2VydmVkIGlzIHdoZW4gIGFsbCBwb3J0cyBmb3IgMC4wLjAuMC8wIHJhbmdlIGFyZSBvcGVuIGluIGEgcnVzaCB0byBhY2Nlc3MgdGhlIGluc3RhbmNlLiBFQzIgaW5zdGFuY2VzIG11c3QgZXhwb3NlIG9ubHkgdG8gdGhvc2UgcG9ydHMgZW5hYmxlZCBvbiB0aGUgY29ycmVzcG9uZGluZyBzZWN1cml0eSBncm91cCBsZXZlbC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDMlJlc3RyaWN0ZWRJbmJvdW5kLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFQzI2JyxcbiAgICAgIGluZm86ICdUaGUgRUJTIHZvbHVtZSBoYXMgZW5jcnlwdGlvbiBkaXNhYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgIFwiV2l0aCBFQlMgZW5jcnlwdGlvbiwgeW91IGFyZW4ndCByZXF1aXJlZCB0byBidWlsZCwgbWFpbnRhaW4sIGFuZCBzZWN1cmUgeW91ciBvd24ga2V5IG1hbmFnZW1lbnQgaW5mcmFzdHJ1Y3R1cmUuIEVCUyBlbmNyeXB0aW9uIHVzZXMgS01TIGtleXMgd2hlbiBjcmVhdGluZyBlbmNyeXB0ZWQgdm9sdW1lcyBhbmQgc25hcHNob3RzLiBUaGlzIGhlbHBzIHByb3RlY3QgZGF0YSBhdCByZXN0LlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDMkVCU1ZvbHVtZUVuY3J5cHRlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUMyNycsXG4gICAgICBpbmZvOiAnVGhlIFNlY3VyaXR5IEdyb3VwIGRvZXMgbm90IGhhdmUgYSBkZXNjcmlwdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdEZXNjcmlwdGlvbnMgaGVscCBzaW1wbGlmeSBvcGVyYXRpb25zIGFuZCByZW1vdmUgYW55IG9wcG9ydHVuaXRpZXMgZm9yIG9wZXJhdG9yIGVycm9ycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDMlNlY3VyaXR5R3JvdXBEZXNjcmlwdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUMyOCcsXG4gICAgICBpbmZvOiAnVGhlIEVDMiBpbnN0YW5jZS9BdXRvU2NhbGluZyBsYXVuY2ggY29uZmlndXJhdGlvbiBkb2VzIG5vdCBoYXZlIGRldGFpbGVkIG1vbml0b3JpbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdNb25pdG9yaW5nIGRhdGEgaGVscHMgbWFrZSBiZXR0ZXIgZGVjaXNpb25zIG9uIGFyY2hpdGVjdGluZyBhbmQgbWFuYWdpbmcgY29tcHV0ZSByZXNvdXJjZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFQzJJbnN0YW5jZURldGFpbGVkTW9uaXRvcmluZ0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0VDMjknLFxuICAgICAgaW5mbzogJ1RoZSBFQzIgaW5zdGFuY2UgaXMgbm90IHBhcnQgb2YgYW4gQVNHIGFuZCBoYXMgVGVybWluYXRpb24gUHJvdGVjdGlvbiBkaXNhYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUZXJtaW5hdGlvbiBQcm90ZWN0aW9uIHNhZmV0eSBmZWF0dXJlIGVuYWJsZWQgaW4gb3JkZXIgdG8gcHJvdGVjdCB0aGUgaW5zdGFuY2VzIGZyb20gYmVpbmcgYWNjaWRlbnRhbGx5IHRlcm1pbmF0ZWQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFQzJJbnN0YW5jZVRlcm1pbmF0aW9uUHJvdGVjdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUNSMScsXG4gICAgICBpbmZvOiAnVGhlIEVDUiBSZXBvc2l0b3J5IGFsbG93cyBvcGVuIGFjY2Vzcy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdSZW1vdmluZyAqIHByaW5jaXBhbHMgaW4gYW4gRUNSIFJlcG9zaXRvcnkgaGVscHMgcHJvdGVjdCBhZ2FpbnN0IHVuYXV0aG9yaXplZCBhY2Nlc3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFQ1JPcGVuQWNjZXNzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFQ1MyJyxcbiAgICAgIGluZm86ICdUaGUgRUNTIFRhc2sgRGVmaW5pdGlvbiBpbmNsdWRlcyBhIGNvbnRhaW5lciBkZWZpbml0aW9uIHRoYXQgZGlyZWN0bHkgc3BlY2lmaWVzIGVudmlyb25tZW50IHZhcmlhYmxlcy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdVc2Ugc2VjcmV0cyB0byBpbmplY3QgZW52aXJvbm1lbnQgdmFyaWFibGVzIGR1cmluZyBjb250YWluZXIgc3RhcnR1cCBmcm9tIEFXUyBTeXN0ZW1zIE1hbmFnZXIgUGFyYW1ldGVyIFN0b3JlIG9yIFNlY3JldHMgTWFuYWdlciBpbnN0ZWFkIG9mIGRpcmVjdGx5IHNwZWNpZnlpbmcgcGxhaW50ZXh0IGVudmlyb25tZW50IHZhcmlhYmxlcy4gVXBkYXRlcyB0byBkaXJlY3QgZW52aXJvbm1lbnQgdmFyaWFibGVzIHJlcXVpcmUgb3BlcmF0b3JzIHRvIGNoYW5nZSB0YXNrIGRlZmluaXRpb25zIGFuZCBwZXJmb3JtIG5ldyBkZXBsb3ltZW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDU1Rhc2tEZWZpbml0aW9uTm9FbnZpcm9ubWVudFZhcmlhYmxlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUNTNCcsXG4gICAgICBpbmZvOiAnVGhlIEVDUyBDbHVzdGVyIGhhcyBDbG91ZFdhdGNoIENvbnRhaW5lciBJbnNpZ2h0cyBkaXNhYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdDbG91ZFdhdGNoIENvbnRhaW5lciBJbnNpZ2h0cyBhbGxvdyBvcGVyYXRvcnMgdG8gZ2FpbiBhIGJldHRlciBwZXJzcGVjdGl2ZSBvbiBob3cgdGhlIGNsdXN0ZXLigJlzIGFwcGxpY2F0aW9ucyBhbmQgbWljcm9zZXJ2aWNlcyBhcmUgcGVyZm9ybWluZy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDU0NsdXN0ZXJDbG91ZFdhdGNoQ29udGFpbmVySW5zaWdodHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0VDUzcnLFxuICAgICAgaW5mbzogJ1RoZSBFQ1MgVGFzayBEZWZpbml0aW9uIGRvZXMgbm90IGhhdmUgYXdzbG9ncyBsb2dnaW5nIGVuYWJsZWQgYXQgdGhlIG1pbmltdW0uJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQ29udGFpbmVyIGxvZ2dpbmcgYWxsb3dzIG9wZXJhdG9ycyB0byB2aWV3IGFuZCBhZ2dyZWdhdGUgdGhlIGxvZ3MgZnJvbSB0aGUgY29udGFpbmVyLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRUNTVGFza0RlZmluaXRpb25Db250YWluZXJMb2dnaW5nLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFS1MxJyxcbiAgICAgIGluZm86IFwiVGhlIEVLUyBjbHVzdGVyJ3MgS3ViZXJuZXRlcyBBUEkgc2VydmVyIGVuZHBvaW50IGhhcyBwdWJsaWMgYWNjZXNzIGVuYWJsZWQuXCIsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgXCJBIGNsdXN0ZXIncyBLdWJlcm5ldGVzIEFQSSBzZXJ2ZXIgZW5kcG9pbnQgc2hvdWxkIG5vdCBiZSBwdWJsaWNseSBhY2Nlc3NpYmxlIGZyb20gdGhlIEludGVybmV0IGluIG9yZGVyIHRvIGF2b2lkIGV4cG9zaW5nIHByaXZhdGUgZGF0YSBhbmQgbWluaW1pemluZyBzZWN1cml0eSByaXNrcy4gVGhlIEFQSSBzZXJ2ZXIgZW5kcG9pbnRzIHNob3VsZCBvbmx5IGJlIGFjY2Vzc2libGUgZnJvbSB3aXRoaW4gYSBBV1MgVmlydHVhbCBQcml2YXRlIENsb3VkIChWUEMpLlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVLU0NsdXN0ZXJOb0VuZHBvaW50UHVibGljQWNjZXNzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFS1MyJyxcbiAgICAgIGluZm86IFwiVGhlIEVLUyBDbHVzdGVyIGRvZXMgbm90IHB1Ymxpc2ggJ2FwaScsICdhdWRpdCcsICdhdXRoZW50aWNhdG9yLCAnY29udHJvbGxlck1hbmFnZXInLCBhbmQgJ3NjaGVkdWxlcicgY29udHJvbCBwbGFuZSBsb2dzLlwiLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFS1MgY29udHJvbCBwbGFuZSBsb2dnaW5nIHByb3ZpZGVzIGF1ZGl0IGFuZCBkaWFnbm9zdGljIGxvZ3MgZGlyZWN0bHkgZnJvbSB0aGUgQW1hem9uIEVLUyBjb250cm9sIHBsYW5lIHRvIENsb3VkV2F0Y2ggTG9ncyBpbiB5b3VyIGFjY291bnQuIFRoZXNlIGxvZ3MgbWFrZSBpdCBlYXN5IGZvciB5b3UgdG8gc2VjdXJlIGFuZCBydW4geW91ciBjbHVzdGVycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVLU0NsdXN0ZXJDb250cm9sUGxhbmVMb2dzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFTEIxJyxcbiAgICAgIGluZm86ICdUaGUgQ0xCIGlzIHVzZWQgZm9yIGluY29taW5nIEhUVFAvSFRUUFMgdHJhZmZpYy4gVXNlIEFMQnMgaW5zdGVhZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdIVFRQL0hUVFBTIGFwcGxpY2F0aW9ucyAobW9ub2xpdGhpYyBvciBjb250YWluZXJpemVkKSBzaG91bGQgdXNlIHRoZSBBTEIgaW5zdGVhZCBvZiB0aGUgQ0xCIGZvciBlbmhhbmNlZCBpbmNvbWluZyB0cmFmZmljIGRpc3RyaWJ1dGlvbiwgYmV0dGVyIHBlcmZvcm1hbmNlIGFuZCBsb3dlciBjb3N0cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENMQk5vSW5ib3VuZEh0dHBIdHRwcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUxCMicsXG4gICAgICBpbmZvOiAnVGhlIEVMQiBkb2VzIG5vdCBoYXZlIGFjY2VzcyBsb2dzIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQWNjZXNzIGxvZ3MgYWxsb3cgb3BlcmF0b3JzIHRvIHRvIGFuYWx5emUgdHJhZmZpYyBwYXR0ZXJucyBhbmQgaWRlbnRpZnkgYW5kIHRyb3VibGVzaG9vdCBzZWN1cml0eSBpc3N1ZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFTEJMb2dnaW5nRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRUxCMycsXG4gICAgICBpbmZvOiAnVGhlIENMQiBkb2VzIG5vdCBoYXZlIGNvbm5lY3Rpb24gZHJhaW5pbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdXaXRoIENvbm5lY3Rpb24gRHJhaW5pbmcgZmVhdHVyZSBlbmFibGVkLCBpZiBhbiBFQzIgYmFja2VuZCBpbnN0YW5jZSBmYWlscyBoZWFsdGggY2hlY2tzIFRoZSBDTEIgd2lsbCBub3Qgc2VuZCBhbnkgbmV3IHJlcXVlc3RzIHRvIHRoZSB1bmhlYWx0aHkgaW5zdGFuY2UuIEhvd2V2ZXIsIGl0IHdpbGwgc3RpbGwgYWxsb3cgZXhpc3RpbmcgKGluLWZsaWdodCkgcmVxdWVzdHMgdG8gY29tcGxldGUgZm9yIHRoZSBkdXJhdGlvbiBvZiB0aGUgY29uZmlndXJlZCB0aW1lb3V0LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQ0xCQ29ubmVjdGlvbkRyYWluaW5nLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFTEI0JyxcbiAgICAgIGluZm86ICdUaGUgQ0xCIGRvZXMgbm90IHVzZSBhdCBsZWFzdCB0d28gQVpzIHdpdGggdGhlIENyb3NzLVpvbmUgTG9hZCBCYWxhbmNpbmcgZmVhdHVyZSBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0NMQnMgY2FuIGRpc3RyaWJ1dGUgdGhlIHRyYWZmaWMgZXZlbmx5IGFjcm9zcyBhbGwgYmFja2VuZCBpbnN0YW5jZXMuIFRvIHVzZSBDcm9zcy1ab25lIExvYWQgQmFsYW5jaW5nIGF0IG9wdGltYWwgbGV2ZWwsIHRoZSBzeXN0ZW0gc2hvdWxkIG1haW50YWluIGFuIGVxdWFsIEVDMiBjYXBhY2l0eSBkaXN0cmlidXRpb24gaW4gZWFjaCBvZiB0aGUgQVpzIHJlZ2lzdGVyZWQgd2l0aCB0aGUgbG9hZCBiYWxhbmNlci4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVMQkNyb3NzWm9uZUxvYWRCYWxhbmNpbmdFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFTEI1JyxcbiAgICAgIGluZm86ICdUaGUgQ0xCIGxpc3RlbmVyIGlzIG5vdCBjb25maWd1cmVkIGZvciBzZWN1cmUgKEhUVFBzIG9yIFNTTCkgcHJvdG9jb2xzIGZvciBjbGllbnQgY29tbXVuaWNhdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgSFRUUHMgb3IgU1NMIHByb3RvY29scyBlbmFibGUgc2VjdXJlIGNvbW11bmljYXRpb24gYnkgZW5jcnlwdGluZyB0aGUgY29tbXVuaWNhdGlvbiBiZXR3ZWVuIHRoZSBjbGllbnQgYW5kIHRoZSBsb2FkIGJhbGFuY2VyLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRUxCVGxzSHR0cHNMaXN0ZW5lcnNPbmx5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBTdG9yYWdlIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1N0b3JhZ2Uobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdTMScsXG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldCBoYXMgc2VydmVyIGFjY2VzcyBsb2dzIGRpc2FibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBidWNrZXQgc2hvdWxkIGhhdmUgc2VydmVyIGFjY2VzcyBsb2dnaW5nIGVuYWJsZWQgdG8gcHJvdmlkZSBkZXRhaWxlZCByZWNvcmRzIGZvciB0aGUgcmVxdWVzdHMgdGhhdCBhcmUgbWFkZSB0byB0aGUgYnVja2V0LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUzNCdWNrZXRMb2dnaW5nRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUzInLFxuICAgICAgaW5mbzogJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSBwdWJsaWMgYWNjZXNzIHJlc3RyaWN0ZWQgYW5kIGJsb2NrZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIGJ1Y2tldCBzaG91bGQgaGF2ZSBwdWJsaWMgYWNjZXNzIHJlc3RyaWN0ZWQgYW5kIGJsb2NrZWQgdG8gcHJldmVudCB1bmF1dGhvcml6ZWQgYWNjZXNzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUzNCdWNrZXRMZXZlbFB1YmxpY0FjY2Vzc1Byb2hpYml0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1MzJyxcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGRlZmF1bHQgZW5jcnlwdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBidWNrZXQgc2hvdWxkIG1pbmltYWxseSBoYXZlIFNTRSBlbmFibGVkIHRvIGhlbHAgcHJvdGVjdCBkYXRhLWF0LXJlc3QuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTM0J1Y2tldFNlcnZlclNpZGVFbmNyeXB0aW9uRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUzEwJyxcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IHJlcXVpcmUgcmVxdWVzdHMgdG8gdXNlIFNTTC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdZb3UgY2FuIHVzZSBIVFRQUyAoVExTKSB0byBoZWxwIHByZXZlbnQgcG90ZW50aWFsIGF0dGFja2VycyBmcm9tIGVhdmVzZHJvcHBpbmcgb24gb3IgbWFuaXB1bGF0aW5nIG5ldHdvcmsgdHJhZmZpYyB1c2luZyBwZXJzb24taW4tdGhlLW1pZGRsZSBvciBzaW1pbGFyIGF0dGFja3MuIFlvdSBzaG91bGQgYWxsb3cgb25seSBlbmNyeXB0ZWQgY29ubmVjdGlvbnMgb3ZlciBIVFRQUyAoVExTKSB1c2luZyB0aGUgYXdzOlNlY3VyZVRyYW5zcG9ydCBjb25kaXRpb24gb24gQW1hem9uIFMzIGJ1Y2tldCBwb2xpY2llcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFMzQnVja2V0U1NMUmVxdWVzdHNPbmx5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFRlMxJyxcbiAgICAgIGluZm86ICdUaGUgRUZTIGlzIG5vdCBjb25maWd1cmVkIGZvciBlbmNyeXB0aW9uIGF0IHJlc3QuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQnkgdXNpbmcgYW4gZW5jcnlwdGVkIGZpbGUgc3lzdGVtLCBkYXRhIGFuZCBtZXRhZGF0YSBhcmUgYXV0b21hdGljYWxseSBlbmNyeXB0ZWQgYmVmb3JlIGJlaW5nIHdyaXR0ZW4gdG8gdGhlIGZpbGUgc3lzdGVtLiBTaW1pbGFybHksIGFzIGRhdGEgYW5kIG1ldGFkYXRhIGFyZSByZWFkLCB0aGV5IGFyZSBhdXRvbWF0aWNhbGx5IGRlY3J5cHRlZCBiZWZvcmUgYmVpbmcgcHJlc2VudGVkIHRvIHRoZSBhcHBsaWNhdGlvbi4gVGhlc2UgcHJvY2Vzc2VzIGFyZSBoYW5kbGVkIHRyYW5zcGFyZW50bHkgYnkgRUZTIHdpdGhvdXQgcmVxdWlyaW5nIG1vZGlmaWNhdGlvbiBvZiBhcHBsaWNhdGlvbnMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFRlNFbmNyeXB0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIERhdGFiYXNlIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0RhdGFiYXNlcyhub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JEUzInLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgaW5zdGFuY2Ugb3IgQXVyb3JhIERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBzdG9yYWdlIGVuY3J5cHRpb24gZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdTdG9yYWdlIGVuY3J5cHRpb24gaGVscHMgcHJvdGVjdCBkYXRhLWF0LXJlc3QgYnkgZW5jcnlwdGluZyB0aGUgdW5kZXJseWluZyBzdG9yYWdlLCBhdXRvbWF0ZWQgYmFja3VwcywgcmVhZCByZXBsaWNhcywgYW5kIHNuYXBzaG90cyBmb3IgdGhlIGRhdGFiYXNlLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUkRTU3RvcmFnZUVuY3J5cHRlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUkRTMycsXG4gICAgICBpbmZvOiAnVGhlIG5vbi1BdXJvcmEgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgbXVsdGktQVogc3VwcG9ydCBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzZSBtdWx0aS1BWiBkZXBsb3ltZW50IGNvbmZpZ3VyYXRpb25zIGZvciBoaWdoIGF2YWlsYWJpbGl0eSBhbmQgYXV0b21hdGljIGZhaWxvdmVyIHN1cHBvcnQgZnVsbHkgbWFuYWdlZCBieSBBV1MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSRFNNdWx0aUFaU3VwcG9ydCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUkRTNicsXG4gICAgICBpbmZvOiAnVGhlIFJEUyBBdXJvcmEgTXlTUUwvUG9zdGdyZXNTUUwgY2x1c3RlciBkb2VzIG5vdCBoYXZlIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgXCJXaXRoIElBTSBEYXRhYmFzZSBBdXRoZW50aWNhdGlvbiBlbmFibGVkLCB0aGUgc3lzdGVtIGRvZXNuJ3QgaGF2ZSB0byB1c2UgYSBwYXNzd29yZCB3aGVuIGNvbm5lY3RpbmcgdG8gdGhlIE15U1FML1Bvc3RncmVTUUwgZGF0YWJhc2UgaW5zdGFuY2VzLCBpbnN0ZWFkIGl0IHVzZXMgYW4gYXV0aGVudGljYXRpb24gdG9rZW4uXCIsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQXVyb3JhTXlTUUxQb3N0Z3Jlc0lBTUF1dGgsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JEUzgnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgREIgU2VjdXJpdHkgR3JvdXAgYWxsb3dzIGZvciAwLjAuMC4wLzAgaW5ib3VuZCBhY2Nlc3MuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnUkRTIERCIHNlY3VyaXR5IGdyb3VwcyBzaG91bGQgbm90IGFsbG93IGFjY2VzcyBmcm9tIDAuMC4wLjAvMCAoaS5lLiBhbnl3aGVyZSwgZXZlcnkgbWFjaGluZSB0aGF0IGhhcyB0aGUgYWJpbGl0eSB0byBlc3RhYmxpc2ggYSBjb25uZWN0aW9uKSBpbiBvcmRlciB0byByZWR1Y2UgdGhlIHJpc2sgb2YgdW5hdXRob3JpemVkIGFjY2Vzcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJEU1Jlc3RyaWN0ZWRJbmJvdW5kLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdSRFMxMCcsXG4gICAgICBpbmZvOiAnVGhlIFJEUyBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGRlbGV0aW9uIHByb3RlY3Rpb24gZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgZGVsZXRpb24gcHJvdGVjdGlvbiBmZWF0dXJlIGhlbHBzIHByb3RlY3QgdGhlIGRhdGFiYXNlIGZyb20gYmVpbmcgYWNjaWRlbnRhbGx5IGRlbGV0ZWQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSRFNJbnN0YW5jZURlbGV0aW9uUHJvdGVjdGlvbkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JEUzExJyxcbiAgICAgIGluZm86ICdUaGUgUkRTIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyIHVzZXMgdGhlIGRlZmF1bHQgZW5kcG9pbnQgcG9ydC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdQb3J0IG9iZnVzY2F0aW9uICh1c2luZyBhIG5vbiBkZWZhdWx0IGVuZHBvaW50IHBvcnQpIGFkZHMgYW4gYWRkaXRpb25hbCBsYXllciBvZiBkZWZlbnNlIGFnYWluc3Qgbm9uLXRhcmdldGVkIGF0dGFja3MgKGkuZS4gTXlTUUwvQXVyb3JhIHBvcnQgMzMwNiwgU1FMIFNlcnZlciBwb3J0IDE0MzMsIFBvc3RncmVTUUwgcG9ydCA1NDMyLCBldGMpLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUkRTTm9uRGVmYXVsdFBvcnQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JEUzEzJyxcbiAgICAgIGluZm86ICdUaGUgUkRTIGluc3RhbmNlIGlzIG5vdCBjb25maWd1cmVkIGZvciBhdXRvbWF0ZWQgYmFja3Vwcy4nLFxuICAgICAgZXhwbGFuYXRpb246ICdBdXRvbWF0ZWQgYmFja3VwcyBhbGxvdyBmb3IgcG9pbnQtaW4tdGltZSByZWNvdmVyeS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJEU0luc3RhbmNlQmFja3VwRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUkRTMTQnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgQXVyb3JhIE15U1FMIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBCYWNrdHJhY2sgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCYWNrdHJhY2sgaGVscHMgb3JkZXIgdG8gcmV3aW5kIGNsdXN0ZXIgdGFibGVzIHRvIGEgc3BlY2lmaWMgdGltZSwgd2l0aG91dCB1c2luZyBiYWNrdXBzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQXVyb3JhTXlTUUxCYWNrdHJhY2ssXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JEUzE1JyxcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIGluc3RhbmNlIG9yIEF1cm9yYSBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgZGVsZXRpb24gcHJvdGVjdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuYWJsaW5nIERlbGV0aW9uIFByb3RlY3Rpb24gYXQgdGhlIGNsdXN0ZXIgbGV2ZWwgZm9yIEFtYXpvbiBBdXJvcmEgZGF0YWJhc2VzIG9yIGluc3RhbmNlIGxldmVsIGZvciBub24gQXVyb3JhIGluc3RhbmNlcyBoZWxwcyBwcm90ZWN0IGZyb20gYWNjaWRlbnRhbCBkZWxldGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJEU0luc3RhbmNlRGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUkRTMTYnLFxuICAgICAgaW5mbzogJ1RoZSBSRFMgQXVyb3JhIE15U1FMIHNlcnZlcmxlc3MgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1ZGl0LCBlcnJvciwgZ2VuZXJhbCwgYW5kIHNsb3dxdWVyeSBMb2cgRXhwb3J0cyBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoaXMgYWxsb3dzIG9wZXJhdG9ycyB0byB1c2UgQ2xvdWRXYXRjaCB0byB2aWV3IGxvZ3MgdG8gaGVscCBkaWFnbm9zZSBwcm9ibGVtcyBpbiB0aGUgZGF0YWJhc2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBdXJvcmFNeVNRTExvZ2dpbmcsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0REQjMnLFxuICAgICAgaW5mbzogJ1RoZSBEeW5hbW9EQiB0YWJsZSBkb2VzIG5vdCBoYXZlIFBvaW50LWluLXRpbWUgUmVjb3ZlcnkgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdEeW5hbW9EQiBjb250aW51b3VzIGJhY2t1cHMgcmVwcmVzZW50IGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgaW5zdXJhbmNlIGFnYWluc3QgYWNjaWRlbnRhbCBsb3NzIG9mIGRhdGEgb24gdG9wIG9mIG9uLWRlbWFuZCBiYWNrdXBzLiBUaGUgRHluYW1vREIgc2VydmljZSBjYW4gYmFjayB1cCB0aGUgZGF0YSB3aXRoIHBlci1zZWNvbmQgZ3JhbnVsYXJpdHkgYW5kIHJlc3RvcmUgaXQgdG8gYW55IHNpbmdsZSBzZWNvbmQgZnJvbSB0aGUgdGltZSBQSVRSIHdhcyBlbmFibGVkIHVwIHRvIHRoZSBwcmlvciAzNSBkYXlzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBEeW5hbW9EQlBJVFJFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdEREI0JyxcbiAgICAgIGluZm86ICdUaGUgREFYIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRGF0YSBpbiBjYWNoZSwgY29uZmlndXJhdGlvbiBkYXRhIGFuZCBsb2cgZmlsZXMgc2hvdWxkIGJlIGVuY3J5cHRlZCB1c2luZyBTZXJ2ZXItU2lkZSBFbmNyeXB0aW9uIGluIG9yZGVyIHRvIHByb3RlY3QgZnJvbSB1bmF1dGhvcml6ZWQgYWNjZXNzIHRvIHRoZSB1bmRlcmx5aW5nIHN0b3JhZ2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBEQVhFbmNyeXB0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0FFQzEnLFxuICAgICAgaW5mbzogJ1RoZSBFbGFzdGlDYWNoZSBjbHVzdGVyIGlzIG5vdCBwcm92aXNpb25lZCBpbiBhIFZQQy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdQcm92aXNpb25pbmcgdGhlIGNsdXN0ZXIgd2l0aGluIGEgVlBDIGFsbG93cyBmb3IgYmV0dGVyIGZsZXhpYmlsaXR5IGFuZCBjb250cm9sIG92ZXIgdGhlIGNhY2hlIGNsdXN0ZXJzIHNlY3VyaXR5LCBhdmFpbGFiaWxpdHksIHRyYWZmaWMgcm91dGluZyBhbmQgbW9yZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVsYXN0aUNhY2hlQ2x1c3RlckluVlBDLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBRUMzJyxcbiAgICAgIGluZm86ICdUaGUgRWxhc3RpQ2FjaGUgUmVkaXMgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGJvdGggZW5jcnlwdGlvbiBpbiB0cmFuc2l0IGFuZCBhdCByZXN0IGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5jcnlwdGlvbiBpbiB0cmFuc2l0IGhlbHBzIHNlY3VyZSBjb21tdW5pY2F0aW9ucyB0byB0aGUgY2x1c3Rlci4gRW5jcnlwdGlvbiBhdCByZXN0IGhlbHBzIHByb3RlY3QgZGF0YSBhdCByZXN0IGZyb20gdW5hdXRob3JpemVkIGFjY2Vzcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVsYXN0aUNhY2hlUmVkaXNDbHVzdGVyRW5jcnlwdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQUVDNCcsXG4gICAgICBpbmZvOiAnVGhlIEVsYXN0aUNhY2hlIFJlZGlzIGNsdXN0ZXIgaXMgbm90IGRlcGxveWVkIGluIGEgTXVsdGktQVogY29uZmlndXJhdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgY2x1c3RlciBzaG91bGQgdXNlIGEgTXVsdGktQVogZGVwbG95bWVudCBjb25maWd1cmF0aW9uIGZvciBoaWdoIGF2YWlsYWJpbGl0eS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVsYXN0aUNhY2hlUmVkaXNDbHVzdGVyTXVsdGlBWixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQUVDNScsXG4gICAgICBpbmZvOiAnVGhlIEVsYXN0aUNhY2hlIGNsdXN0ZXIgdXNlcyB0aGUgZGVmYXVsdCBlbmRwb2ludCBwb3J0LicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1BvcnQgb2JmdXNjYXRpb24gKHVzaW5nIGEgbm9uIGRlZmF1bHQgZW5kcG9pbnQgcG9ydCkgYWRkcyBhbiBhZGRpdGlvbmFsIGxheWVyIG9mIGRlZmVuc2UgYWdhaW5zdCBub24tdGFyZ2V0ZWQgYXR0YWNrcyAoaS5lLiBSZWRpcyBwb3J0IDYzNzkgYW5kIE1lbWNhY2hlZCBwb3J0IDExMjExKS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVsYXN0aUNhY2hlQ2x1c3Rlck5vbkRlZmF1bHRQb3J0LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBRUM2JyxcbiAgICAgIGluZm86ICdUaGUgRWxhc3RpQ2FjaGUgUmVkaXMgY2x1c3RlciBkb2VzIG5vdCB1c2UgUmVkaXMgQVVUSCBmb3IgdXNlciBhdXRoZW50aWNhdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdSZWRpcyBhdXRoZW50aWNhdGlvbiB0b2tlbnMgZW5hYmxlIFJlZGlzIHRvIHJlcXVpcmUgYSB0b2tlbiAocGFzc3dvcmQpIGJlZm9yZSBhbGxvd2luZyBjbGllbnRzIHRvIGV4ZWN1dGUgY29tbWFuZHMsIHRoZXJlYnkgaW1wcm92aW5nIGRhdGEgc2VjdXJpdHkuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFbGFzdGlDYWNoZVJlZGlzQ2x1c3RlclJlZGlzQXV0aCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTjEnLFxuICAgICAgaW5mbzogJ1RoZSBOZXB0dW5lIERCIGNsdXN0ZXIgaXMgbm90IGRlcGxveWVkIGluIGEgTXVsdGktQVogY29uZmlndXJhdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgY2x1c3RlciBzaG91bGQgdXNlIGEgTXVsdGktQVogZGVwbG95bWVudCBjb25maWd1cmF0aW9uIGZvciBoaWdoIGF2YWlsYWJpbGl0eS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE5lcHR1bmVDbHVzdGVyTXVsdGlBWixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTjInLFxuICAgICAgaW5mbzogJ1RoZSBOZXB0dW5lIERCIGluc3RhbmNlIGRvZXMgaGF2ZSBBdXRvIE1pbm9yIFZlcnNpb24gVXBncmFkZSBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBOZXB0dW5lIHNlcnZpY2UgcmVndWxhcmx5IHJlbGVhc2VzIGVuZ2luZSB1cGRhdGVzLiBFbmFibGluZyBBdXRvIE1pbm9yIFZlcnNpb24gVXBncmFkZSB3aWxsIGFsbG93IHRoZSBzZXJ2aWNlIHRvIGF1dG9tYXRpY2FsbHkgYXBwbHkgdGhlc2UgdXBncmFkZXMgdG8gREIgSW5zdGFuY2VzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogTmVwdHVuZUNsdXN0ZXJBdXRvbWF0aWNNaW5vclZlcnNpb25VcGdyYWRlLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdOMycsXG4gICAgICBpbmZvOiAnVGhlIE5lcHR1bmUgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGEgcmVhc29uYWJsZSBtaW5pbXVtIGJhY2t1cCByZXRlbnRpb24gcGVyaW9kIGNvbmZpZ3VyZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIHJldGVudGlvbiBwZXJpb2QgcmVwcmVzZW50cyB0aGUgbnVtYmVyIG9mIGRheXMgdG8gcmV0YWluIGF1dG9tYXRlZCBzbmFwc2hvdHMuIEEgbWluaW11bSByZXRlbnRpb24gcGVyaW9kIG9mIDcgZGF5cyBpcyByZWNvbW1lbmRlZCBidXQgY2FuIGJlIGFkanVzdCB0byBtZWV0IHN5c3RlbSByZXF1aXJlbWVudHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBOZXB0dW5lQ2x1c3RlckJhY2t1cFJldGVudGlvblBlcmlvZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTjQnLFxuICAgICAgaW5mbzogJ1RoZSBOZXB0dW5lIERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmNyeXB0aW5nIGRhdGEtYXQtcmVzdCBwcm90ZWN0cyBkYXRhIGNvbmZpZGVudGlhbGl0eSBhbmQgcHJldmVudHMgdW5hdXRob3JpemVkIHVzZXJzIGZyb20gYWNjZXNzaW5nIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE5lcHR1bmVDbHVzdGVyRW5jcnlwdGlvbkF0UmVzdCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTjUnLFxuICAgICAgaW5mbzogJ1RoZSBOZXB0dW5lIERCIGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBJQU0gRGF0YWJhc2UgQXV0aGVudGljYXRpb24gZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgIFwiV2l0aCBJQU0gRGF0YWJhc2UgQXV0aGVudGljYXRpb24gZW5hYmxlZCwgdGhlIHN5c3RlbSBkb2Vzbid0IGhhdmUgdG8gdXNlIGEgcGFzc3dvcmQgd2hlbiBjb25uZWN0aW5nIHRvIHRoZSBjbHVzdGVyLlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE5lcHR1bmVDbHVzdGVySUFNQXV0aCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlMxJyxcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCByZXF1aXJlIFRMUy9TU0wgZW5jcnlwdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmFibGluZyB0aGUgXCJyZXF1aXJlX3NzbFwiIHBhcmFtZXRlciBzZWN1cmVzIGRhdGEtaW4tdHJhbnNpdCBieSBlbmNyeXB0aW5nIHRoZSBjb25uZWN0aW9uIGJldHdlZW4gdGhlIGNsaWVudHMgYW5kIHRoZSBSZWRzaGlmdCBjbHVzdGVycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0UmVxdWlyZVRsc1NTTCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlMyJyxcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBpcyBub3QgcHJvdmlzaW9uZWQgaW4gYSBWUEMuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnUHJvdmlzaW9uaW5nIHRoZSBjbHVzdGVyIHdpdGhpbiBhIFZQQyBhbGxvd3MgZm9yIGJldHRlciBmbGV4aWJpbGl0eSBhbmQgY29udHJvbCBvdmVyIHRoZSBSZWRzaGlmdCBjbHVzdGVycyBzZWN1cml0eSwgYXZhaWxhYmlsaXR5LCB0cmFmZmljIHJvdXRpbmcgYW5kIG1vcmUuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJJblZQQyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlMzJyxcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciB1c2VzIHRoZSBkZWZhdWx0IFwiYXdzdXNlclwiIHVzZXJuYW1lLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzaW5nIGEgY3VzdG9tIG1hc3RlciB1c2VyIG5hbWUgaW5zdGVhZCBvZiB0aGUgZGVmYXVsdCBtYXN0ZXIgdXNlciBuYW1lIChpLmUuIFwiYXdzdXNlclwiKSBwcm92aWRlcyBhbiBhZGRpdGlvbmFsIGxheWVyIG9mIGRlZmVuc2UgYWdhaW5zdCBub24tdGFyZ2V0ZWQgYXR0YWNrcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0Q2x1c3Rlck5vbkRlZmF1bHRVc2VybmFtZSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlM0JyxcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciB1c2VzIHRoZSBkZWZhdWx0IGVuZHBvaW50IHBvcnQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnUG9ydCBvYmZ1c2NhdGlvbiAodXNpbmcgYSBub24gZGVmYXVsdCBlbmRwb2ludCBwb3J0KSBhZGRzIGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgZGVmZW5zZSBhZ2FpbnN0IG5vbi10YXJnZXRlZCBhdHRhY2tzIChpLmUuIFJlZHNoaWZ0IHBvcnQgNTQzOSkuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJOb25EZWZhdWx0UG9ydCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlM1JyxcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1ZGl0IGxvZ2dpbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBdWRpdCBsb2dnaW5nIGhlbHBzIG9wZXJhdG9ycyB0cm91Ymxlc2hvb3QgaXNzdWVzIGFuZCBlbnN1cmUgc2VjdXJpdHkuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJBdWRpdExvZ2dpbmcsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JTNicsXG4gICAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246ICdFbmNyeXB0aW5nIGRhdGEtYXQtcmVzdCBwcm90ZWN0cyBkYXRhIGNvbmZpZGVudGlhbGl0eS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0Q2x1c3RlckVuY3J5cHRpb25BdFJlc3QsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JTOCcsXG4gICAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgaXMgcHVibGljbHkgYWNjZXNzaWJsZS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdEaXNhYmxpbmcgcHVibGljIGFjY2Vzc2liaWxpdHkgaGVscHMgbWluaW1pemUgc2VjdXJpdHkgcmlza3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJQdWJsaWNBY2Nlc3MsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1JTOScsXG4gICAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSB2ZXJzaW9uIHVwZ3JhZGUgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdWZXJzaW9uIFVwZ3JhZGUgbXVzdCBlbmFibGVkIHRvIGVuYWJsZSB0aGUgY2x1c3RlciB0byBhdXRvbWF0aWNhbGx5IHJlY2VpdmUgdXBncmFkZXMgZHVyaW5nIHRoZSBtYWludGVuYW5jZSB3aW5kb3cuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJWZXJzaW9uVXBncmFkZSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlMxMCcsXG4gICAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSBhIHJldGVudGlvbiBwZXJpb2QgZm9yIGF1dG9tYXRlZCBzbmFwc2hvdHMgY29uZmlndXJlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgcmV0ZW50aW9uIHBlcmlvZCByZXByZXNlbnRzIHRoZSBudW1iZXIgb2YgZGF5cyB0byByZXRhaW4gYXV0b21hdGVkIHNuYXBzaG90cy4gQSBwb3NpdGl2ZSByZXRlbnRpb24gcGVyaW9kIHNob3VsZCBiZSBzZXQgdG8gY29uZmlndXJlIHRoaXMgZmVhdHVyZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0QmFja3VwRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnUlMxMScsXG4gICAgICBpbmZvOiAnVGhlIFJlZHNoaWZ0IGNsdXN0ZXIgZG9lcyBub3QgaGF2ZSB1c2VyIGFjdGl2aXR5IGxvZ2dpbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdVc2VyIGFjdGl2aXR5IGxvZ2dpbmcgbG9ncyBlYWNoIHF1ZXJ5IGJlZm9yZSBpdCBpcyBwZXJmb3JtZWQgb24gdGhlIGNsdXN0ZXJzIGRhdGFic2UuIFRvIGVuYWJsZSB0aGlzIGZlYXR1cmUgYXNzb2NpYXRlIGEgUmVzZGhzaWZ0IENsdXN0ZXIgUGFyYW1ldGVyIEdyb3VwIHdpdGggdGhlIFwiZW5hYmxlX3VzZXJfYWN0aXZpdHlfbG9nZ2luZ1wiIHBhcmFtZXRlciBzZXQgdG8gXCJ0cnVlXCIuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJVc2VyQWN0aXZpdHlMb2dnaW5nLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdET0MxJyxcbiAgICAgIGluZm86ICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuY3J5cHRpbmcgZGF0YS1hdC1yZXN0IHByb3RlY3RzIGRhdGEgY29uZmlkZW50aWFsaXR5IGFuZCBwcmV2ZW50cyB1bmF1dGhvcml6ZWQgdXNlcnMgZnJvbSBhY2Nlc3Npbmcgc2Vuc2l0aXZlIGluZm9ybWF0aW9uLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRG9jdW1lbnREQkNsdXN0ZXJFbmNyeXB0aW9uQXRSZXN0LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdET0MyJyxcbiAgICAgIGluZm86ICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciB1c2VzIHRoZSBkZWZhdWx0IGVuZHBvaW50IHBvcnQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnUG9ydCBvYmZ1c2NhdGlvbiAodXNpbmcgYSBub24gZGVmYXVsdCBlbmRwb2ludCBwb3J0KSBhZGRzIGFuIGFkZGl0aW9uYWwgbGF5ZXIgb2YgZGVmZW5zZSBhZ2FpbnN0IG5vbi10YXJnZXRlZCBhdHRhY2tzIChpLmUuIE1vbmdvREIgcG9ydCAyNzAxNykuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBEb2N1bWVudERCQ2x1c3Rlck5vbkRlZmF1bHRQb3J0LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdET0MzJyxcbiAgICAgIGluZm86ICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHRoZSB1c2VybmFtZSBhbmQgcGFzc3dvcmQgc3RvcmVkIGluIFNlY3JldHMgTWFuYWdlci4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgIFwiU2VjcmV0cyBNYW5hZ2VyIGVuYWJsZXMgb3BlcmF0b3JzIHRvIHJlcGxhY2UgaGFyZGNvZGVkIGNyZWRlbnRpYWxzIGluIHlvdXIgY29kZSwgaW5jbHVkaW5nIHBhc3N3b3Jkcywgd2l0aCBhbiBBUEkgY2FsbCB0byBTZWNyZXRzIE1hbmFnZXIgdG8gcmV0cmlldmUgdGhlIHNlY3JldCBwcm9ncmFtbWF0aWNhbGx5LiBUaGlzIGhlbHBzIGVuc3VyZSB0aGUgc2VjcmV0IGNhbid0IGJlIGNvbXByb21pc2VkIGJ5IHNvbWVvbmUgZXhhbWluaW5nIHN5c3RlbSBjb2RlLCBiZWNhdXNlIHRoZSBzZWNyZXQgbm8gbG9uZ2VyIGV4aXN0cyBpbiB0aGUgY29kZS4gQWxzbywgb3BlcmF0b3JzIGNhbiBjb25maWd1cmUgU2VjcmV0cyBNYW5hZ2VyIHRvIGF1dG9tYXRpY2FsbHkgcm90YXRlIHRoZSBzZWNyZXQgZm9yIHlvdSBhY2NvcmRpbmcgdG8gYSBzcGVjaWZpZWQgc2NoZWR1bGUuIFRoaXMgZW5hYmxlcyB5b3UgdG8gcmVwbGFjZSBsb25nLXRlcm0gc2VjcmV0cyB3aXRoIHNob3J0LXRlcm0gb25lcywgc2lnbmlmaWNhbnRseSByZWR1Y2luZyB0aGUgcmlzayBvZiBjb21wcm9taXNlLlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IERvY3VtZW50REJDcmVkZW50aWFsc0luU2VjcmV0c01hbmFnZXIsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0RPQzQnLFxuICAgICAgaW5mbzogJ1RoZSBEb2N1bWVudCBEQiBjbHVzdGVyIGRvZXMgbm90IGhhdmUgYSByZWFzb25hYmxlIG1pbmltdW0gYmFja3VwIHJldGVudGlvbiBwZXJpb2QgY29uZmlndXJlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGUgcmV0ZW50aW9uIHBlcmlvZCByZXByZXNlbnRzIHRoZSBudW1iZXIgb2YgZGF5cyB0byByZXRhaW4gYXV0b21hdGVkIHNuYXBzaG90cy4gQSBtaW5pbXVtIHJldGVudGlvbiBwZXJpb2Qgb2YgNyBkYXlzIGlzIHJlY29tbWVuZGVkIGJ1dCBjYW4gYmUgYWRqdXN0IHRvIG1lZXQgc3lzdGVtIHJlcXVpcmVtZW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IERvY3VtZW50REJDbHVzdGVyQmFja3VwUmV0ZW50aW9uUGVyaW9kLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdET0M1JyxcbiAgICAgIGluZm86ICdUaGUgRG9jdW1lbnQgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGF1dGhlbnRpY2F0ZSwgY3JlYXRlSW5kZXgsIGFuZCBkcm9wQ29sbGVjdGlvbiBMb2cgRXhwb3J0cyBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoaXMgYWxsb3dzIG9wZXJhdG9ycyB0byB1c2UgQ2xvdWRXYXRjaCB0byB2aWV3IGxvZ3MgdG8gaGVscCBkaWFnbm9zZSBwcm9ibGVtcyBpbiB0aGUgZGF0YWJhc2UuIFRoZSBldmVudHMgcmVjb3JkZWQgYnkgdGhlIEFXUyBEb2N1bWVudERCIGF1ZGl0IGxvZ3MgaW5jbHVkZSBzdWNjZXNzZnVsIGFuZCBmYWlsZWQgYXV0aGVudGljYXRpb24gYXR0ZW1wdHMsIGNyZWF0aW5nIGluZGV4ZXMgb3IgZHJvcHBpbmcgYSBjb2xsZWN0aW9uIGluIGEgZGF0YWJhc2Ugd2l0aGluIHRoZSBEb2N1bWVudERCIGNsdXN0ZXIuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBEb2N1bWVudERCQ2x1c3RlckxvZ0V4cG9ydHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1RTMycsXG4gICAgICBpbmZvOiAnVGhlIFRpbWVzdHJlYW0gZGF0YWJhc2UgZG9lcyBub3QgdXNlIGEgQ3VzdG9tZXIgTWFuYWdlZCBLTVMgS2V5IGZvciBhdCByZXN0IGVuY3J5cHRpb24uJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQWxsIFRpbWVzdHJlYW0gdGFibGVzIGluIGEgZGF0YWJhc2UgYXJlIGVuY3J5cHRlZCBhdCByZXN0IGJ5IGRlZmF1bHQgdXNpbmcgQVdTIE1hbmFnZWQgS2V5LiBUaGVzZSBrZXlzIGFyZSByb3RhdGVkIGV2ZXJ5IHRocmVlIHllYXJzLiBEYXRhIGF0IHJlc3QgbXVzdCBiZSBlbmNyeXB0ZWQgdXNpbmcgQ01LcyBpZiB5b3UgcmVxdWlyZSBtb3JlIGNvbnRyb2wgb3ZlciB0aGUgcGVybWlzc2lvbnMgYW5kIGxpZmVjeWNsZSBvZiB5b3VyIGtleXMsIGluY2x1ZGluZyB0aGUgYWJpbGl0eSB0byBoYXZlIHRoZW0gYXV0b21hdGljYWxseSByb3RhdGVkIG9uIGFuIGFubnVhbCBiYXNpcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogVGltZXN0cmVhbURhdGFiYXNlQ3VzdG9tZXJNYW5hZ2VkS2V5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBOZXR3b3JrIGFuZCBEZWxpdmVyeSBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tOZXR3b3JrRGVsaXZlcnkobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdWUEMzJyxcbiAgICAgIGluZm86ICdBIE5ldHdvcmsgQUNMIG9yIE5ldHdvcmsgQUNMIGVudHJ5IGhhcyBiZWVuIGltcGxlbWVudGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ05ldHdvcmsgQUNMcyBzaG91bGQgYmUgdXNlZCBzcGFyaW5nbHkgZm9yIHRoZSBmb2xsb3dpbmcgcmVhc29uczogdGhleSBjYW4gYmUgY29tcGxleCB0byBtYW5hZ2UsIHRoZXkgYXJlIHN0YXRlbGVzcywgZXZlcnkgSVAgYWRkcmVzcyBtdXN0IGJlIGV4cGxpY2l0bHkgb3BlbmVkIGluIGVhY2ggKGluYm91bmQvb3V0Ym91bmQpIGRpcmVjdGlvbiwgYW5kIHRoZXkgYWZmZWN0IGEgY29tcGxldGUgc3VibmV0LiBVc2Ugc2VjdXJpdHkgZ3JvdXBzIHdoZW4gcG9zc2libGUgYXMgdGhleSBhcmUgc3RhdGVmdWwgYW5kIGVhc2llciB0byBtYW5hZ2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICAgIHJ1bGU6IFZQQ05vTkFDTHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1ZQQzcnLFxuICAgICAgaW5mbzogJ1RoZSBWUEMgZG9lcyBub3QgaGF2ZSBhbiBhc3NvY2lhdGVkIEZsb3cgTG9nLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1ZQQyBGbG93IExvZ3MgY2FwdHVyZSBuZXR3b3JrIGZsb3cgaW5mb3JtYXRpb24gZm9yIGEgVlBDLCBzdWJuZXQsIG9yIG5ldHdvcmsgaW50ZXJmYWNlIGFuZCBzdG9yZXMgaXQgaW4gQW1hem9uIENsb3VkV2F0Y2ggTG9ncy4gRmxvdyBsb2cgZGF0YSBjYW4gaGVscCBjdXN0b21lcnMgdHJvdWJsZXNob290IG5ldHdvcmsgaXNzdWVzOyBmb3IgZXhhbXBsZSwgdG8gZGlhZ25vc2Ugd2h5IHNwZWNpZmljIHRyYWZmaWMgaXMgbm90IHJlYWNoaW5nIGFuIGluc3RhbmNlLCB3aGljaCBtaWdodCBiZSBhIHJlc3VsdCBvZiBvdmVybHkgcmVzdHJpY3RpdmUgc2VjdXJpdHkgZ3JvdXAgcnVsZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBWUENGbG93TG9nc0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0NGUjEnLFxuICAgICAgaW5mbzogJ1RoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBtYXkgcmVxdWlyZSBHZW8gcmVzdHJpY3Rpb25zLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0dlbyByZXN0cmljdGlvbiBtYXkgbmVlZCB0byBiZSBlbmFibGVkIGZvciB0aGUgZGlzdHJpYnV0aW9uIGluIG9yZGVyIHRvIGFsbG93IG9yIGRlbnkgYSBjb3VudHJ5IGluIG9yZGVyIHRvIGFsbG93IG9yIHJlc3RyaWN0IHVzZXJzIGluIHNwZWNpZmljIGxvY2F0aW9ucyBmcm9tIGFjY2Vzc2luZyBjb250ZW50LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBDbG91ZEZyb250RGlzdHJpYnV0aW9uR2VvUmVzdHJpY3Rpb25zLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdDRlIyJyxcbiAgICAgIGluZm86ICdUaGUgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gbWF5IHJlcXVpcmUgaW50ZWdyYXRpb24gd2l0aCBBV1MgV0FGLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBXZWIgQXBwbGljYXRpb24gRmlyZXdhbGwgY2FuIGhlbHAgcHJvdGVjdCBhZ2FpbnN0IGFwcGxpY2F0aW9uLWxheWVyIGF0dGFja3MgdGhhdCBjYW4gY29tcHJvbWlzZSB0aGUgc2VjdXJpdHkgb2YgdGhlIHN5c3RlbSBvciBwbGFjZSB1bm5lY2Vzc2FyeSBsb2FkIG9uIHRoZW0uJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICAgIHJ1bGU6IENsb3VkRnJvbnREaXN0cmlidXRpb25XQUZJbnRlZ3JhdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ0ZSMycsXG4gICAgICBpbmZvOiAnVGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9ucyBkb2VzIG5vdCBoYXZlIGFjY2VzcyBsb2dnaW5nIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5hYmxpbmcgYWNjZXNzIGxvZ3MgaGVscHMgb3BlcmF0b3JzIHRyYWNrIGFsbCB2aWV3ZXIgcmVxdWVzdHMgZm9yIHRoZSBjb250ZW50IGRlbGl2ZXJlZCB0aHJvdWdoIHRoZSBDb250ZW50IERlbGl2ZXJ5IE5ldHdvcmsuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDbG91ZEZyb250RGlzdHJpYnV0aW9uQWNjZXNzTG9nZ2luZyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ0ZSNScsXG4gICAgICBpbmZvOiAnVGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9ucyB1c2VzIFNTTHYzIG9yIFRMU3YxIGZvciBjb21tdW5pY2F0aW9uIHRvIHRoZSBvcmlnaW4uJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVnVsbmVyYWJpbGl0aWVzIGhhdmUgYmVlbiBhbmQgY29udGludWUgdG8gYmUgZGlzY292ZXJlZCBpbiB0aGUgZGVwcmVjYXRlZCBTU0wgYW5kIFRMUyBwcm90b2NvbHMuIFVzaW5nIGEgc2VjdXJpdHkgcG9saWN5IHdpdGggbWluaW11bSBUTFN2MS4xIG9yIFRMU3YxLjIgYW5kIGFwcHJvcHJpYXRlIHNlY3VyaXR5IGNpcGhlcnMgZm9yIEhUVFBTIGhlbHBzIHByb3RlY3Qgdmlld2VyIGNvbm5lY3Rpb25zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQ2xvdWRGcm9udERpc3RyaWJ1dGlvbk5vT3V0ZGF0ZWRTU0wsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0NGUjYnLFxuICAgICAgaW5mbzogJ1RoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBkb2VzIG5vdCB1c2UgYW4gb3JpZ2luIGFjY2VzcyBpZGVudGl0eSBhbiBTMyBvcmlnaW4uJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnT3JpZ2luIGFjY2VzcyBpZGVudGl0aWVzIGhlbHAgd2l0aCBzZWN1cml0eSBieSByZXN0cmljdGluZyBhbnkgZGlyZWN0IGFjY2VzcyB0byBvYmplY3RzIHRocm91Z2ggUzMgVVJMcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENsb3VkRnJvbnREaXN0cmlidXRpb25TM09yaWdpbkFjY2Vzc0lkZW50aXR5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBUElHMScsXG4gICAgICBpbmZvOiAnVGhlIEFQSSBkb2VzIG5vdCBoYXZlIGFjY2VzcyBsb2dnaW5nIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5hYmxpbmcgYWNjZXNzIGxvZ3MgaGVscHMgb3BlcmF0b3JzIHZpZXcgd2hvIGFjY2Vzc2VkIGFuIEFQSSBhbmQgaG93IHRoZSBjYWxsZXIgYWNjZXNzZWQgdGhlIEFQSS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEFQSUdXQWNjZXNzTG9nZ2luZyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQVBJRzInLFxuICAgICAgaW5mbzogJ1RoZSBSRVNUIEFQSSBkb2VzIG5vdCBoYXZlIHJlcXVlc3QgdmFsaWRhdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBBUEkgc2hvdWxkIGhhdmUgYmFzaWMgcmVxdWVzdCB2YWxpZGF0aW9uIGVuYWJsZWQuIElmIHRoZSBBUEkgaXMgaW50ZWdyYXRlZCB3aXRoIGN1c3RvbSBzb3VyY2UgKExhbWJkYSwgRUNTLCBldGMuLikgaW4gdGhlIGJhY2tlbmQsIGRlZXBlciBpbnB1dCB2YWxpZGF0aW9uIHNob3VsZCBiZSBjb25zaWRlcmVkIGZvciBpbXBsZW1lbnRhdGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEFQSUdXUmVxdWVzdFZhbGlkYXRpb24sXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0FQSUczJyxcbiAgICAgIGluZm86ICdUaGUgUkVTVCBBUEkgc3RhZ2UgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBBV1MgV0FGdjIgd2ViIEFDTC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBV1MgV0FGdjIgaXMgYSB3ZWIgYXBwbGljYXRpb24gZmlyZXdhbGwgdGhhdCBoZWxwcyBwcm90ZWN0IHdlYiBhcHBsaWNhdGlvbnMgYW5kIEFQSXMgZnJvbSBhdHRhY2tzIGJ5IGFsbG93aW5nIGNvbmZpZ3VyZWQgcnVsZXMgdG8gYWxsb3csIGJsb2NrLCBvciBtb25pdG9yIChjb3VudCkgd2ViIHJlcXVlc3RzIGJhc2VkIG9uIGN1c3RvbWl6YWJsZSBydWxlcyBhbmQgY29uZGl0aW9ucyB0aGF0IGFyZSBkZWZpbmVkLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBBUElHV0Fzc29jaWF0ZWRXaXRoV0FGLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBUElHNCcsXG4gICAgICBpbmZvOiAnVGhlIEFQSSBkb2VzIG5vdCBpbXBsZW1lbnQgYXV0aG9yaXphdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdJbiBtb3N0IGNhc2VzIGFuIEFQSSBuZWVkcyB0byBoYXZlIGFuIGF1dGhlbnRpY2F0aW9uIGFuZCBhdXRob3JpemF0aW9uIGltcGxlbWVudGF0aW9uIHN0cmF0ZWd5LiBUaGlzIGluY2x1ZGVzIHVzaW5nIHN1Y2ggYXBwcm9hY2hlcyBhcyBJQU0sIENvZ25pdG8gVXNlciBQb29scywgQ3VzdG9tIGF1dGhvcml6ZXIsIGV0Yy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEFQSUdXQXV0aG9yaXphdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQVBJRzYnLFxuICAgICAgaW5mbzogJ1RoZSBSRVNUIEFQSSBTdGFnZSBkb2VzIG5vdCBoYXZlIENsb3VkV2F0Y2ggbG9nZ2luZyBlbmFibGVkIGZvciBhbGwgbWV0aG9kcy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmFibGluZyBDbG91ZFdhdGNoIGxvZ3MgYXQgdGhlIHN0YWdlIGxldmVsIGhlbHBzIG9wZXJhdG9ycyB0byB0cmFjayBhbmQgYW5hbHl6ZSBleGVjdXRpb24gYmVoYXZpb3IgYXQgdGhlIEFQSSBzdGFnZSBsZXZlbC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEFQSUdXRXhlY3V0aW9uTG9nZ2luZ0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIE1hbmFnZW1lbnQgYW5kIEdvdmVybmFuY2UgU2VydmljZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrTWFuYWdlbWVudEdvdmVybmFuY2Uobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBUzEnLFxuICAgICAgaW5mbzogJ1RoZSBBdXRvIFNjYWxpbmcgR3JvdXAgZG9lcyBub3QgaGF2ZSBhIGNvb2xkb3duIHBlcmlvZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBIGNvb2xkb3duIHBlcmlvZCB0ZW1wb3JhcmlseSBzdXNwZW5kcyBhbnkgc2NhbGluZyBhY3Rpdml0aWVzIGluIG9yZGVyIHRvIGFsbG93IHRoZSBuZXdseSBsYXVuY2hlZCBFQzIgaW5zdGFuY2Uocykgc29tZSB0aW1lIHRvIHN0YXJ0IGhhbmRsaW5nIHRoZSBhcHBsaWNhdGlvbiB0cmFmZmljLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQXV0b1NjYWxpbmdHcm91cENvb2xkb3duUGVyaW9kLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdBUzInLFxuICAgICAgaW5mbzogJ1RoZSBBdXRvIFNjYWxpbmcgR3JvdXAgZG9lcyBub3QgaGF2ZSBwcm9wZXJseSBjb25maWd1cmVkIGhlYWx0aCBjaGVja3MuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIGhlYWx0aCBjaGVjayBmZWF0dXJlIGVuYWJsZXMgdGhlIHNlcnZpY2UgdG8gZGV0ZWN0IHdoZXRoZXIgaXRzIHJlZ2lzdGVyZWQgRUMyIGluc3RhbmNlcyBhcmUgaGVhbHRoeSBvciBub3QuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBdXRvU2NhbGluZ0dyb3VwSGVhbHRoQ2hlY2ssXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0FTMycsXG4gICAgICBpbmZvOiAnVGhlIEF1dG8gU2NhbGluZyBHcm91cCBkb2VzIG5vdCBoYXZlIG5vdGlmaWNhdGlvbnMgY29uZmlndXJlZCBmb3IgYWxsIHNjYWxpbmcgZXZlbnRzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ05vdGlmaWNhdGlvbnMgb24gRUMyIGluc3RhbmNlIGxhdW5jaCwgbGF1bmNoIGVycm9yLCB0ZXJtaW5hdGlvbiwgYW5kIHRlcm1pbmF0aW9uIGVycm9ycyBhbGxvdyBvcGVyYXRvcnMgdG8gZ2FpbiBiZXR0ZXIgaW5zaWdodHMgaW50byBzeXN0ZW1zIGF0dHJpYnV0ZXMgc3VjaCBhcyBhY3Rpdml0eSBhbmQgaGVhbHRoLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQXV0b1NjYWxpbmdHcm91cFNjYWxpbmdOb3RpZmljYXRpb25zLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBNYWNoaW5lIExlYXJuaW5nIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja01hY2hpbmVMZWFybmluZyhub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1NNMScsXG4gICAgICBpbmZvOiAnVGhlIFNhZ2VNYWtlciBub3RlYm9vayBpbnN0YW5jZSBpcyBub3QgcHJvdmlzaW9uZWQgaW5zaWRlIGEgVlBDLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1Byb3Zpc2lvbmluZyB0aGUgbm90ZWJvb2sgaW5zdGFuY2VzIGluc2lkZSBhIFZQQyBlbmFibGVzIHRoZSBub3RlYm9vayB0byBhY2Nlc3MgVlBDLW9ubHkgcmVzb3VyY2VzIHN1Y2ggYXMgRUZTIGZpbGUgc3lzdGVtcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFNhZ2VNYWtlck5vdGVib29rSW5WUEMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1NNMicsXG4gICAgICBpbmZvOiAnVGhlIFNhZ2VNYWtlciBub3RlYm9vayBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIGFuIGVuY3J5cHRlZCBzdG9yYWdlIHZvbHVtZS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmNyeXB0aW5nIHN0b3JhZ2Ugdm9sdW1lcyBoZWxwcyBwcm90ZWN0IFNhZ2VNYWtlciBkYXRhLWF0LXJlc3QuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTYWdlTWFrZXJOb3RlYm9va0luc3RhbmNlS01TS2V5Q29uZmlndXJlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnU00zJyxcbiAgICAgIGluZm86ICdUaGUgU2FnZU1ha2VyIG5vdGVib29rIGluc3RhbmNlIGhhcyBkaXJlY3QgaW50ZXJuZXQgYWNjZXNzIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRGlzYWJsaW5nIHB1YmxpYyBhY2Nlc3NpYmlsaXR5IGhlbHBzIG1pbmltaXplIHNlY3VyaXR5IHJpc2tzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbmFseXRpY3MgU2VydmljZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQW5hbHl0aWNzKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQVRIMScsXG4gICAgICBpbmZvOiAnVGhlIEF0aGVuYSB3b3JrZ3JvdXAgZG9lcyBub3QgZW5jcnlwdCBxdWVyeSByZXN1bHRzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuY3J5cHRpbmcgcXVlcnkgcmVzdWx0cyBzdG9yZWQgaW4gUzMgaGVscHMgc2VjdXJlIGRhdGEgdG8gbWVldCBjb21wbGlhbmNlIHJlcXVpcmVtZW50cyBmb3IgZGF0YS1hdC1yZXN0IGVuY3J5cHRpb24uJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBdGhlbmFXb3JrZ3JvdXBFbmNyeXB0ZWRRdWVyeVJlc3VsdHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0VNUjInLFxuICAgICAgaW5mbzogJ1RoZSBFTVIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIFMzIGxvZ2dpbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdVcGxvYWRpbmcgbG9ncyB0byBTMyBlbmFibGVzIHRoZSBzeXN0ZW0gdG8ga2VlcCB0aGUgbG9nZ2luZyBkYXRhIGZvciBoaXN0b3JpY2FsIHB1cnBvc2VzIG9yIHRvIHRyYWNrIGFuZCBhbmFseXplIHRoZSBjbHVzdGVycyBiZWhhdmlvci4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVNUlMzQWNjZXNzTG9nZ2luZyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRU1SNCcsXG4gICAgICBpbmZvOiAnVGhlIEVNUiBjbHVzdGVyIGRvZXMgbm90IHVzZSBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24gd2l0aCBsb2NhbCBkaXNrIGVuY3J5cHRpb24gZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdMb2NhbCBkaXNrIGVuY3J5cHRpb24gdXNlcyBhIGNvbWJpbmF0aW9uIG9mIG9wZW4tc291cmNlIEhERlMgZW5jcnlwdGlvbiBhbmQgTFVLUyBlbmNyeXB0aW9uIHRvIHNlY3VyZSBkYXRhIGF0IHJlc3QuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFTVJMb2NhbERpc2tFbmNyeXB0aW9uLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdFTVI1JyxcbiAgICAgIGluZm86ICdUaGUgRU1SIGNsdXN0ZXIgZG9lcyBub3QgdXNlIGEgc2VjdXJpdHkgY29uZmlndXJhdGlvbiB3aXRoIGVuY3J5cHRpb24gaW4gdHJhbnNpdCBlbmFibGVkIGFuZCBjb25maWd1cmVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VNUiBDbHVzdGVycyBzaG91bGQgaGF2ZSBhIG1ldGhvZCBmb3IgZW5jcnlwdGluZyBkYXRhIGluIHRyYW5zaXQgdXNpbmcgVHJhbnNwb3J0IExheWVyIFNlY3VyaXR5IChUTFMpLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRU1SRW5jcnlwdGlvbkluVHJhbnNpdCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnRU1SNicsXG4gICAgICBpbmZvOiAnVGhlIEVNUiBjbHVzdGVyIGRvZXMgbm90IGltcGxlbWVudCBhdXRoZW50aWNhdGlvbiB2aWEgYW4gRUMyIEtleSBQYWlyIG9yIEtlcmJlcm9zLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1NTSCBjbGllbnRzIGNhbiB1c2UgYW4gRUMyIGtleSBwYWlyIHRvIGF1dGhlbnRpY2F0ZSB0byBjbHVzdGVyIGluc3RhbmNlcy4gQWx0ZXJuYXRpdmVseSwgd2l0aCBFTVIgcmVsZWFzZSB2ZXJzaW9uIDUuMTAuMCBvciBsYXRlciwgc29sdXRpb25zIGNhbiBjb25maWd1cmUgS2VyYmVyb3MgdG8gYXV0aGVudGljYXRlIHVzZXJzIGFuZCBTU0ggY29ubmVjdGlvbnMgdG8gdGhlIG1hc3RlciBub2RlLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRU1SQXV0aEVDMktleVBhaXJPcktlcmJlcm9zLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdHTDEnLFxuICAgICAgaW5mbzogJ1RoZSBHbHVlIGNyYXdsZXIgb3Igam9iIGRvZXMgbm90IHVzZSBhIHNlY3VyaXR5IGNvbmZpZ3VyYXRpb24gd2l0aCBDbG91ZFdhdGNoIExvZyBlbmNyeXB0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5hYmxpbmcgZW5jcnlwdGlvbiBhdCByZXN0IGhlbHBzIHByZXZlbnQgdW5hdXRob3JpemVkIHVzZXJzIGZyb20gZ2V0dGluZyBhY2Nlc3MgdG8gdGhlIGxvZ2dpbmcgZGF0YSBwdWJsaXNoZWQgdG8gQ2xvdWRXYXRjaCBMb2dzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBHbHVlRW5jcnlwdGVkQ2xvdWRXYXRjaExvZ3MsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0dMMycsXG4gICAgICBpbmZvOiAnVGhlIEdsdWUgam9iIGRvZXMgbm90IGhhdmUgdXNlIGEgc2VjdXJpdHkgY29uZmlndXJhdGlvbiB3aXRoIGpvYiBib29rbWFyayBlbmNyeXB0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnSm9iIGJvb2ttYXJrIGVuY3J5cHRpb24gZW5jcnlwdHMgYm9va21hcmsgZGF0YSBiZWZvcmUgaXQgaXMgc2VudCB0byBBbWF6b24gUzMgZm9yIHN0b3JhZ2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICAgIHJ1bGU6IEdsdWVKb2JCb29rbWFya0VuY3J5cHRlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnS0RBMycsXG4gICAgICBpbmZvOiAnVGhlIEtpbmVzaXMgRGF0YSBBbmFseXRpY3MgRmxpbmsgQXBwbGljYXRpb24gZG9lcyBub3QgaGF2ZSBjaGVja3BvaW50aW5nIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQ2hlY2twb2ludHMgYXJlIGJhY2t1cHMgb2YgYXBwbGljYXRpb24gc3RhdGUgdGhhdCBLREEgYXV0b21hdGljYWxseSBjcmVhdGVzIHBlcmlvZGljYWxseSBhbmQgdXNlcyB0byByZXN0b3JlIGZyb20gZmF1bHRzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBLaW5lc2lzRGF0YUFuYWx5dGljc0ZsaW5rQ2hlY2twb2ludGluZyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnS0RGMScsXG4gICAgICBpbmZvOiAnVGhlIEtpbmVzaXMgRGF0YSBGaXJlaG9zZSBkZWxpdmVyeSBzdHJlYW0gZG9lcyBoYXZlIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUaGlzIGFsbG93cyB0aGUgc3lzdGVtIHRvIG1lZXQgc3RyaWN0IHJlZ3VsYXRvcnkgcmVxdWlyZW1lbnRzIGFuZCBlbmhhbmNlIHRoZSBzZWN1cml0eSBvZiBzeXN0ZW0gZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEtpbmVzaXNEYXRhRmlyZWhvc2VTU0UsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0tEUzEnLFxuICAgICAgaW5mbzogJ1RoZSBLaW5lc2lzIERhdGEgU3RyZWFtIGRvZXMgbm90IGhhcyBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICBcIkRhdGEgaXMgZW5jcnlwdGVkIGJlZm9yZSBpdCdzIHdyaXR0ZW4gdG8gdGhlIEtpbmVzaXMgc3RyZWFtIHN0b3JhZ2UgbGF5ZXIsIGFuZCBkZWNyeXB0ZWQgYWZ0ZXIgaXTigJlzIHJldHJpZXZlZCBmcm9tIHN0b3JhZ2UuIFRoaXMgYWxsb3dzIHRoZSBzeXN0ZW0gdG8gbWVldCBzdHJpY3QgcmVndWxhdG9yeSByZXF1aXJlbWVudHMgYW5kIGVuaGFuY2UgdGhlIHNlY3VyaXR5IG9mIHN5c3RlbSBkYXRhLlwiLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEtpbmVzaXNEYXRhU3RyZWFtU1NFLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdLRFMzJyxcbiAgICAgIGluZm86ICdUaGUgS2luZXNpcyBEYXRhIFN0cmVhbSBzcGVjaWZpZXMgc2VydmVyLXNpZGUgZW5jcnlwdGlvbiBhbmQgZG9lcyBub3QgdXNlIHRoZSBcImF3cy9raW5lc2lzXCIga2V5LicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0N1c3RvbWVyIE1hbmFnZWQgS2V5cyBjYW4gaW5jdXIgYWRkaXRpb25hbCBjb3N0cyB0aGF0IHNjYWxlIHdpdGggdGhlIGFtb3VudCBvZiBjb25zdW1lcnMgYW5kIHByb2R1Y2Vycy4gRW5zdXJlIHRoYXQgQ3VzdG9tZXIgTWFuYWdlZCBLZXlzIGFyZSByZXF1aXJlZCBmb3IgY29tcGxpYW5jZSBiZWZvcmUgdXNpbmcgdGhlbSAoaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0cmVhbXMvbGF0ZXN0L2Rldi9jb3N0cy1wZXJmb3JtYW5jZS5odG1sKS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogS2luZXNpc0RhdGFTdHJlYW1EZWZhdWx0S2V5V2hlblNTRSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTVNLMicsXG4gICAgICBpbmZvOiAnVGhlIE1TSyBjbHVzdGVyIHVzZXMgcGxhaW50ZXh0IGNvbW11bmljYXRpb24gYmV0d2VlbiBjbGllbnRzIGFuZCBicm9rZXJzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RMUyBvbmx5IGNvbW11bmljYXRpb24gc2VjdXJlcyBkYXRhLWluLXRyYW5zaXQgYnkgZW5jcnlwdGluZyB0aGUgY29ubmVjdGlvbiBiZXR3ZWVuIHRoZSBjbGllbnRzIGFuZCBicm9rZXJzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogTVNLQ2xpZW50VG9Ccm9rZXJUTFMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ01TSzMnLFxuICAgICAgaW5mbzogJ1RoZSBNU0sgY2x1c3RlciB1c2VzIHBsYWludGV4dCBjb21tdW5pY2F0aW9uIGJldHdlZW4gYnJva2Vycy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUTFMgY29tbXVuaWNhdGlvbiBzZWN1cmVzIGRhdGEtaW4tdHJhbnNpdCBieSBlbmNyeXB0aW5nIHRoZSBjb25uZWN0aW9uIGJldHdlZW4gYnJva2Vycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE1TS0Jyb2tlclRvQnJva2VyVExTLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdNU0s2JyxcbiAgICAgIGluZm86ICdUaGUgTVNLIGNsdXN0ZXIgZG9lcyBub3Qgc2VuZCBicm9rZXIgbG9ncyB0byBhIHN1cHBvcnRlZCBkZXN0aW5hdGlvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCcm9rZXIgbG9ncyBlbmFibGUgb3BlcmF0b3JzIHRvIHRyb3VibGVzaG9vdCBBcGFjaGUgS2Fma2EgYXBwbGljYXRpb25zIGFuZCB0byBhbmFseXplIHRoZWlyIGNvbW11bmljYXRpb25zIHdpdGggdGhlIE1TSyBjbHVzdGVyLiBUaGUgY2x1c3RlciBjYW4gZGVsaXZlciBsb2dzIHRvIHRoZSBmb2xsb3dpbmcgcmVzb3VyY2VzOiBhIENsb3VkV2F0Y2ggbG9nIGdyb3VwLCBhbiBTMyBidWNrZXQsIGEgS2luZXNpcyBEYXRhIEZpcmVob3NlIGRlbGl2ZXJ5IHN0cmVhbS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE1TS0Jyb2tlckxvZ2dpbmcsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ09TMScsXG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gaXMgbm90IHByb3Zpc2lvbmVkIGluc2lkZSBhIFZQQy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdQcm92aXNpb25pbmcgdGhlIGRvbWFpbiB3aXRoaW4gYSBWUEMgZW5hYmxlcyBiZXR0ZXIgZmxleGliaWxpdHkgYW5kIGNvbnRyb2wgb3ZlciB0aGUgY2x1c3RlcnMgYWNjZXNzIGFuZCBzZWN1cml0eSBhcyB0aGlzIGZlYXR1cmUga2VlcHMgYWxsIHRyYWZmaWMgYmV0d2VlbiB0aGUgVlBDIGFuZCBPcGVuU2VhcmNoIGRvbWFpbnMgd2l0aGluIHRoZSBBV1MgbmV0d29yayBpbnN0ZWFkIG9mIGdvaW5nIG92ZXIgdGhlIHB1YmxpYyBJbnRlcm5ldC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE9wZW5TZWFyY2hJblZQQ09ubHksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ09TMicsXG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgaGF2ZSBub2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuYWJsaW5nIHRoZSBub2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBmZWF0dXJlIGFkZHMgYW4gZXh0cmEgbGF5ZXIgb2YgZGF0YSBwcm90ZWN0aW9uIG9uIHRvcCBvZiB0aGUgZXhpc3RpbmcgRVMgc2VjdXJpdHkgZmVhdHVyZXMgc3VjaCBhcyBIVFRQUyBjbGllbnQgdG8gY2x1c3RlciBlbmNyeXB0aW9uIGFuZCBkYXRhLWF0LXJlc3QgZW5jcnlwdGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE9wZW5TZWFyY2hOb2RlVG9Ob2RlRW5jcnlwdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnT1MzJyxcbiAgICAgIGluZm86ICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBvbmx5IGdyYW50IGFjY2VzcyB2aWEgYWxsb3dsaXN0ZWQgSVAgYWRkcmVzc2VzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzaW5nIGFsbG93bGlzdGVkIElQIGFkZHJlc3NlcyBoZWxwcyBwcm90ZWN0IHRoZSBkb21haW4gYWdhaW5zdCB1bmF1dGhvcml6ZWQgYWNjZXNzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogT3BlblNlYXJjaEFsbG93bGlzdGVkSVBzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdPUzQnLFxuICAgICAgaW5mbzogJ1RoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGRvZXMgbm90IHVzZSBkZWRpY2F0ZWQgbWFzdGVyIG5vZGVzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzaW5nIGRlZGljYXRlZCBtYXN0ZXIgbm9kZXMgaGVscHMgaW1wcm92ZSBlbnZpcm9ubWVudGFsIHN0YWJpbGl0eSBieSBvZmZsb2FkaW5nIGFsbCB0aGUgbWFuYWdlbWVudCB0YXNrcyBmcm9tIHRoZSBkYXRhIG5vZGVzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogT3BlblNlYXJjaERlZGljYXRlZE1hc3Rlck5vZGUsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ09TNScsXG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgYWxsb3cgZm9yIHVuc2lnbmVkIHJlcXVlc3RzIG9yIGFub255bW91cyBhY2Nlc3MuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnUmVzdHJpY3RpbmcgcHVibGljIGFjY2VzcyBoZWxwcyBwcmV2ZW50IHVuYXV0aG9yaXplZCBhY2Nlc3MgYW5kIHByZXZlbnRzIGFueSB1bnNpZ25lZCByZXF1ZXN0cyB0byBiZSBtYWRlIHRvIHRoZSByZXNvdXJjZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBPcGVuU2VhcmNoTm9VbnNpZ25lZE9yQW5vbnltb3VzQWNjZXNzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdPUzcnLFxuICAgICAgaW5mbzogJ1RoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGRvZXMgbm90IGhhdmUgWm9uZSBBd2FyZW5lc3MgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbmFibGluZyBjcm9zcy16b25lIHJlcGxpY2F0aW9uIChab25lIEF3YXJlbmVzcykgaW5jcmVhc2VzIHRoZSBhdmFpbGFiaWxpdHkgb2YgdGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gYnkgYWxsb2NhdGluZyB0aGUgbm9kZXMgYW5kIHJlcGxpY2F0ZSB0aGUgZGF0YSBhY3Jvc3MgdHdvIEFacyBpbiB0aGUgc2FtZSByZWdpb24gaW4gb3JkZXIgdG8gcHJldmVudCBkYXRhIGxvc3MgYW5kIG1pbmltaXplIGRvd250aW1lIGluIHRoZSBldmVudCBvZiBub2RlIG9yIEFaIGZhaWx1cmUuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBPcGVuU2VhcmNoWm9uZUF3YXJlbmVzcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnT1M4JyxcbiAgICAgIGluZm86ICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuY3J5cHRpbmcgZGF0YS1hdC1yZXN0IHByb3RlY3RzIGRhdGEgY29uZmlkZW50aWFsaXR5IGFuZCBwcmV2ZW50cyB1bmF1dGhvcml6ZWQgdXNlcnMgZnJvbSBhY2Nlc3Npbmcgc2Vuc2l0aXZlIGluZm9ybWF0aW9uLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogT3BlblNlYXJjaEVuY3J5cHRlZEF0UmVzdCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnT1M5JyxcbiAgICAgIGluZm86ICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBtaW5pbWFsbHkgcHVibGlzaCBTRUFSQ0hfU0xPV19MT0dTIGFuZCBJTkRFWF9TTE9XX0xPR1MgdG8gQ2xvdWRXYXRjaCBMb2dzLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZXNlIGxvZ3MgZW5hYmxlIG9wZXJhdG9ycyB0byBnYWluIGZ1bGwgaW5zaWdodCBpbnRvIHRoZSBwZXJmb3JtYW5jZSBvZiB0aGVzZSBvcGVyYXRpb25zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogT3BlblNlYXJjaFNsb3dMb2dzVG9DbG91ZFdhdGNoLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdRUzEnLFxuICAgICAgaW5mbzogJ1RoZSBRdWlja3NpZ2h0IGRhdGEgc291cmNlcyBjb25uZWN0aW9uIGlzIG5vdCBjb25maWd1cmVkIHRvIHVzZSBTU0wuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnU1NMIHNlY3VyZXMgY29tbXVuaWNhdGlvbnMgdG8gZGF0YSBzb3VyY2VzLCBlc3BlY2lhbGx5IHdoZW4gdXNpbmcgcHVibGljIG5ldHdvcmtzLiBVc2luZyBTU0wgd2l0aCBRdWlja1NpZ2h0IHJlcXVpcmVzIHRoZSB1c2Ugb2YgY2VydGlmaWNhdGVzIHNpZ25lZCBieSBhIHB1YmxpY2x5LXJlY29nbml6ZWQgY2VydGlmaWNhdGUgYXV0aG9yaXR5LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUXVpY2tzaWdodFNTTENvbm5lY3Rpb25zLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBTZWN1cml0eSBhbmQgQ29tcGxpYW5jZSBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTZWN1cml0eUNvbXBsaWFuY2Uobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdJQU00JyxcbiAgICAgIGluZm86ICdUaGUgSUFNIHVzZXIsIHJvbGUsIG9yIGdyb3VwIHVzZXMgQVdTIG1hbmFnZWQgcG9saWNpZXMuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQW4gQVdTwqBtYW5hZ2VkIHBvbGljecKgaXMgYSBzdGFuZGFsb25lwqBwb2xpY3nCoHRoYXQgaXMgY3JlYXRlZCBhbmQgYWRtaW5pc3RlcmVkIGJ5IEFXUy4gQ3VycmVudGx5LCBtYW55IEFXUyBtYW5hZ2VkIHBvbGljaWVzIGRvIG5vdCByZXN0cmljdCByZXNvdXJjZSBzY29wZS4gUmVwbGFjZSBBV1MgbWFuYWdlZCBwb2xpY2llcyB3aXRoIHN5c3RlbSBzcGVjaWZpYyAoY3VzdG9tZXIpIG1hbmFnZWQgcG9saWNpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBJQU1Ob01hbmFnZWRQb2xpY2llcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnSUFNNScsXG4gICAgICBpbmZvOiAnVGhlIElBTSBlbnRpdHkgY29udGFpbnMgd2lsZGNhcmQgcGVybWlzc2lvbnMgYW5kIGRvZXMgbm90IGhhdmUgYSBjZGtfbmFnIHJ1bGUgc3VwcHJlc3Npb24gd2l0aCBldmlkZW5jZSBmb3IgdGhvc2UgcGVybWlzc2lvbi4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdNZXRhZGF0YSBleHBsYWluaW5nIHRoZSBldmlkZW5jZSAoZS5nLiB2aWEgc3VwcG9ydGluZyBsaW5rcykgZm9yIHdpbGRjYXJkIHBlcm1pc3Npb25zIGFsbG93cyBmb3IgdHJhbnNwYXJlbmN5IHRvIG9wZXJhdG9ycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IElBTU5vV2lsZGNhcmRQZXJtaXNzaW9ucyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ09HMScsXG4gICAgICBpbmZvOiAnVGhlIENvZ25pdG8gdXNlciBwb29sIGRvZXMgbm90IGhhdmUgYSBwYXNzd29yZCBwb2xpY3kgdGhhdCBtaW5pbWFsbHkgc3BlY2lmeSBhIHBhc3N3b3JkIGxlbmd0aCBvZiBhdCBsZWFzdCA4IGNoYXJhY3RlcnMsIGFzIHdlbGwgYXMgcmVxdWlyaW5nIHVwcGVyY2FzZSwgbnVtZXJpYywgYW5kIHNwZWNpYWwgY2hhcmFjdGVycy4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdTdHJvbmcgcGFzc3dvcmQgcG9saWNpZXMgaW5jcmVhc2Ugc3lzdGVtIHNlY3VyaXR5IGJ5IGVuY291cmFnaW5nIHVzZXJzIHRvIGNyZWF0ZSByZWxpYWJsZSBhbmQgc2VjdXJlIHBhc3N3b3Jkcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENvZ25pdG9Vc2VyUG9vbFN0cm9uZ1Bhc3N3b3JkUG9saWN5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdDT0cyJyxcbiAgICAgIGluZm86ICdUaGUgQ29nbml0byB1c2VyIHBvb2wgZG9lcyBub3QgcmVxdWlyZSBNRkEuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnTXVsdGktZmFjdG9yIGF1dGhlbnRpY2F0aW9uIChNRkEpIGluY3JlYXNlcyBzZWN1cml0eSBmb3IgdGhlIGFwcGxpY2F0aW9uIGJ5IGFkZGluZyBhbm90aGVyIGF1dGhlbnRpY2F0aW9uIG1ldGhvZCwgYW5kIG5vdCByZWx5aW5nIHNvbGVseSBvbiB1c2VyIG5hbWUgYW5kIHBhc3N3b3JkLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBDb2duaXRvVXNlclBvb2xNRkEsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0NPRzMnLFxuICAgICAgaW5mbzogJ1RoZSBDb2duaXRvIHVzZXIgcG9vbCBkb2VzIG5vdCBoYXZlIEFkdmFuY2VkU2VjdXJpdHlNb2RlIHNldCB0byBFTkZPUkNFRC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBZHZhbmNlZCBzZWN1cml0eSBmZWF0dXJlcyBlbmFibGUgdGhlIHN5c3RlbSB0byBkZXRlY3QgYW5kIGFjdCB1cG9uIG1hbGljaW91cyBzaWduLWluIGF0dGVtcHRzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQ29nbml0b1VzZXJQb29sQWR2YW5jZWRTZWN1cml0eU1vZGVFbmZvcmNlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ09HNCcsXG4gICAgICBpbmZvOiAnVGhlIEFQSSBHVyBtZXRob2QgZG9lcyBub3QgdXNlIGEgQ29nbml0byB1c2VyIHBvb2wgYXV0aG9yaXplci4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBUEkgR2F0ZXdheSB2YWxpZGF0ZXMgdGhlIHRva2VucyBmcm9tIGEgc3VjY2Vzc2Z1bCB1c2VyIHBvb2wgYXV0aGVudGljYXRpb24sIGFuZCB1c2VzIHRoZW0gdG8gZ3JhbnQgeW91ciB1c2VycyBhY2Nlc3MgdG8gcmVzb3VyY2VzIGluY2x1ZGluZyBMYW1iZGEgZnVuY3Rpb25zLCBvciB5b3VyIG93biBBUEkuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDb2duaXRvVXNlclBvb2xBUElHV0F1dGhvcml6ZXIsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0NPRzcnLFxuICAgICAgaW5mbzogJ1RoZSBDb2duaXRvIGlkZW50aXR5IHBvb2wgYWxsb3dzIGZvciB1bmF1dGhlbnRpY2F0ZWQgbG9naW5zIGFuZCBkb2VzIG5vdCBoYXZlIGEgY2RrX25hZyBydWxlIHN1cHByZXNzaW9uIHdpdGggYSByZWFzb24uJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnSW4gbWFueSBjYXNlcyBhcHBsaWNhdGlvbnMgZG8gbm90IHdhcnJhbnQgdW5hdXRoZW50aWNhdGVkIGd1ZXN0IGFjY2VzcyBhcHBsaWNhdGlvbnMuIE1ldGFkYXRhIGV4cGxhaW5pbmcgdGhlIHVzZSBjYXNlIGFsbG93cyBmb3IgdHJhbnNwYXJlbmN5IHRvIG9wZXJhdG9ycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENvZ25pdG9Vc2VyUG9vbE5vVW5hdXRoZW50aWNhdGVkTG9naW5zLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdLTVM1JyxcbiAgICAgIGluZm86ICdUaGUgS01TIFN5bW1ldHJpYyBrZXkgZG9lcyBub3QgaGF2ZSBhdXRvbWF0aWMga2V5IHJvdGF0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnS01TIGtleSByb3RhdGlvbiBhbGxvdyBhIHN5c3RlbSB0byBzZXQgYW4geWVhcmx5IHJvdGF0aW9uIHNjaGVkdWxlIGZvciBhIEtNUyBrZXkgc28gd2hlbiBhIEFXUyBLTVMga2V5IGlzIHJlcXVpcmVkIHRvIGVuY3J5cHQgbmV3IGRhdGEsIHRoZSBLTVMgc2VydmljZSBjYW4gYXV0b21hdGljYWxseSB1c2UgdGhlIGxhdGVzdCB2ZXJzaW9uIG9mIHRoZSBIU0EgYmFja2luZyBrZXkgdG8gcGVyZm9ybSB0aGUgZW5jcnlwdGlvbi4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEtNU0JhY2tpbmdLZXlSb3RhdGlvbkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1NNRzQnLFxuICAgICAgaW5mbzogJ1RoZSBzZWNyZXQgZG9lcyBub3QgaGF2ZSBhdXRvbWF0aWMgcm90YXRpb24gc2NoZWR1bGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FXUyBTZWNyZXRzIE1hbmFnZXIgY2FuIGJlIGNvbmZpZ3VyZWQgdG8gYXV0b21hdGljYWxseSByb3RhdGUgdGhlIHNlY3JldCBmb3IgYSBzZWN1cmVkIHNlcnZpY2Ugb3IgZGF0YWJhc2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTZWNyZXRzTWFuYWdlclJvdGF0aW9uRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2VydmVybGVzcyBTZXJ2aWNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTZXJ2ZXJsZXNzKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQVNDMycsXG4gICAgICBpbmZvOiAnVGhlIEdyYXBoUUwgQVBJIGRvZXMgbm90IGhhdmUgcmVxdWVzdCBsZXZlbGluZyBsb2dnaW5nIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnSXQgaXMgaW1wb3J0YW50IHRvIHVzZSBDbG91ZFdhdGNoIExvZ3MgdG8gbG9nIG1ldHJpY3Mgc3VjaCBhcyB3aG8gaGFzIGFjY2Vzc2VkIHRoZSBHcmFwaFFMIEFQSSwgaG93IHRoZSBjYWxsZXIgYWNjZXNzZWQgdGhlIEFQSSwgYW5kIGludmFsaWQgcmVxdWVzdHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBcHBTeW5jR3JhcGhRTFJlcXVlc3RMb2dnaW5nLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdTRjEnLFxuICAgICAgaW5mbzogJ1RoZSBTdGVwIEZ1bmN0aW9uIGRvZXMgbm90IGxvZyBcIkFMTFwiIGV2ZW50cyB0byBDbG91ZFdhdGNoIExvZ3MuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnTG9nZ2luZyBcIkFMTFwiIGV2ZW50cyB0byBDbG91ZFdhdGNoIGxvZ3MgaGVscCBvcGVyYXRvcnMgdHJvdWJsZXNob290IGFuZCBhdWRpdCBzeXN0ZW1zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU3RlcEZ1bmN0aW9uU3RhdGVNYWNoaW5lQWxsTG9nc1RvQ2xvdWRXYXRjaCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnU0YyJyxcbiAgICAgIGluZm86ICdUaGUgU3RlcCBGdW5jdGlvbiBkb2VzIG5vdCBoYXZlIFgtUmF5IHRyYWNpbmcgZW5hYmxlZC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdYLXJheSBwcm92aWRlcyBhbiBlbmQtdG8tZW5kIHZpZXcgb2YgaG93IGFuIGFwcGxpY2F0aW9uIGlzIHBlcmZvcm1pbmcuIFRoaXMgaGVscHMgb3BlcmF0b3JzIHRvIGRpc2NvdmVyIHBlcmZvcm1hbmNlIGlzc3VlcywgZGV0ZWN0IHBlcm1pc3Npb24gcHJvYmxlbXMsIGFuZCB0cmFjayByZXF1ZXN0cyBtYWRlIHRvIGFuZCBmcm9tIG90aGVyIEFXUyBzZXJ2aWNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFN0ZXBGdW5jdGlvblN0YXRlTWFjaGluZVhyYXksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEFwcGxpY2F0aW9uIEludGVncmF0aW9uIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0FwcGxpY2F0aW9uSW50ZWdyYXRpb24obm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdTTlMyJyxcbiAgICAgIGluZm86ICdUaGUgU05TIFRvcGljIGRvZXMgbm90IGhhdmUgc2VydmVyLXNpZGUgZW5jcnlwdGlvbiBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1NlcnZlciBzaWRlIGVuY3J5cHRpb24gYWRkcyBhZGRpdGlvbmFsIHByb3RlY3Rpb24gb2Ygc2Vuc2l0aXZlIGRhdGEgZGVsaXZlcmVkIGFzIG1lc3NhZ2VzIHRvIHN1YnNjcmliZXJzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU05TRW5jcnlwdGVkS01TLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdTTlMzJyxcbiAgICAgIGluZm86ICdUaGUgU05TIFRvcGljIGRvZXMgbm90IHJlcXVpcmUgcmVxdWVzdHMgdG8gdXNlIFNTTC4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdXaXRob3V0IEhUVFBTIChUTFMpLCBhIG5ldHdvcmstYmFzZWQgYXR0YWNrZXIgY2FuIGVhdmVzZHJvcCBvbiBuZXR3b3JrIHRyYWZmaWMgb3IgbWFuaXB1bGF0ZSBpdCwgdXNpbmcgYW4gYXR0YWNrIHN1Y2ggYXMgbWFuLWluLXRoZS1taWRkbGUuIEFsbG93IG9ubHkgZW5jcnlwdGVkIGNvbm5lY3Rpb25zIG92ZXIgSFRUUFMgKFRMUykgdXNpbmcgdGhlIGF3czpTZWN1cmVUcmFuc3BvcnQgY29uZGl0aW9uIGluIHRoZSB0b3BpYyBwb2xpY3kgdG8gZm9yY2UgcmVxdWVzdHMgdG8gdXNlIFNTTC4gSWYgU1NFIGlzIGFscmVhZHkgZW5hYmxlZCB0aGVuIHRoaXMgY29udHJvbCBpcyBhdXRvIGVuZm9yY2VkLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU05TVG9waWNTU0xSZXF1ZXN0c09ubHksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1NRUzInLFxuICAgICAgaW5mbzogJ1RoZSBTUVMgUXVldWUgZG9lcyBub3QgaGF2ZSBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIGVuYWJsZWQuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnU2VydmVyIHNpZGUgZW5jcnlwdGlvbiBhZGRzIGFkZGl0aW9uYWwgcHJvdGVjdGlvbiBvZiBzZW5zaXRpdmUgZGF0YSBkZWxpdmVyZWQgYXMgbWVzc2FnZXMgdG8gc3Vic2NyaWJlcnMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTUVNRdWV1ZVNTRSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnU1FTMycsXG4gICAgICBpbmZvOiAnVGhlIFNRUyBxdWV1ZSBkb2VzIG5vdCBoYXZlIGEgZGVhZC1sZXR0ZXIgcXVldWUgKERMUSkgZW5hYmxlZCBvciBoYXZlIGEgY2RrX25hZyBydWxlIHN1cHByZXNzaW9uIGluZGljYXRpbmcgaXQgaXMgYSBETFEuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVXNpbmcgYSBETFEgaGVscHMgbWFpbnRhaW4gdGhlIHF1ZXVlIGZsb3cgYW5kIGF2b2lkIGxvc2luZyBkYXRhIGJ5IGRldGVjdGluZyBhbmQgbWl0aWdhdGluZyBmYWlsdXJlcyBhbmQgc2VydmljZSBkaXNydXB0aW9ucyBvbiB0aW1lLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU1FTUXVldWVETFEsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ1NRUzQnLFxuICAgICAgaW5mbzogJ1RoZSBTUVMgcXVldWUgZG9lcyBub3QgcmVxdWlyZSByZXF1ZXN0cyB0byB1c2UgU1NMLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1dpdGhvdXQgSFRUUFMgKFRMUyksIGEgbmV0d29yay1iYXNlZCBhdHRhY2tlciBjYW4gZWF2ZXNkcm9wIG9uIG5ldHdvcmsgdHJhZmZpYyBvciBtYW5pcHVsYXRlIGl0LCB1c2luZyBhbiBhdHRhY2sgc3VjaCBhcyBtYW4taW4tdGhlLW1pZGRsZS4gQWxsb3cgb25seSBlbmNyeXB0ZWQgY29ubmVjdGlvbnMgb3ZlciBIVFRQUyAoVExTKSB1c2luZyB0aGUgYXdzOlNlY3VyZVRyYW5zcG9ydCBjb25kaXRpb24gaW4gdGhlIHF1ZXVlIHBvbGljeSB0byBmb3JjZSByZXF1ZXN0cyB0byB1c2UgU1NMLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU1FTUXVldWVTU0xSZXF1ZXN0c09ubHksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIE1lZGlhIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja01lZGlhU2VydmljZXMobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdNUzEnLFxuICAgICAgaW5mbzogJ1RoZSBNZWRpYVN0b3JlIGNvbnRhaW5lciBkb2VzIG5vdCBoYXZlIGNvbnRhaW5lciBhY2Nlc3MgbG9nZ2luZyBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBjb250YWluZXIgc2hvdWxkIGhhdmUgYWNjZXNzIGxvZ2dpbmcgZW5hYmxlZCB0byBwcm92aWRlIGRldGFpbGVkIHJlY29yZHMgZm9yIHRoZSByZXF1ZXN0cyB0aGF0IGFyZSBtYWRlIHRvIHRoZSBjb250YWluZXIuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBNZWRpYVN0b3JlQ29udGFpbmVyQWNjZXNzTG9nZ2luZyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTVMzJyxcbiAgICAgIGluZm86ICdUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgcmVxdWlyZSByZXF1ZXN0cyB0byB1c2UgU1NMLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1lvdSBjYW4gdXNlIEhUVFBTIChUTFMpIHRvIGhlbHAgcHJldmVudCBwb3RlbnRpYWwgYXR0YWNrZXJzIGZyb20gZWF2ZXNkcm9wcGluZyBvbiBvciBtYW5pcHVsYXRpbmcgbmV0d29yayB0cmFmZmljIHVzaW5nIHBlcnNvbi1pbi10aGUtbWlkZGxlIG9yIHNpbWlsYXIgYXR0YWNrcy4gWW91IHNob3VsZCBhbGxvdyBvbmx5IGVuY3J5cHRlZCBjb25uZWN0aW9ucyBvdmVyIEhUVFBTIChUTFMpIHVzaW5nIHRoZSBhd3M6U2VjdXJlVHJhbnNwb3J0IGNvbmRpdGlvbiBvbiBNZWRpYVN0b3JlIGNvbnRhaW5lciBwb2xpY2llcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IE1lZGlhU3RvcmVDb250YWluZXJTU0xSZXF1ZXN0c09ubHksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ01TNCcsXG4gICAgICBpbmZvOiAnVGhlIE1lZGlhU3RvcmUgY29udGFpbmVyIGRvZXMgbm90IGRlZmluZSBhIG1ldHJpYyBwb2xpY3kgdG8gc2VuZCBtZXRyaWNzIHRvIENsb3VkV2F0Y2guJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVXNpbmcgYSBjb21iaW5hdGlvbiBvZiBNZWRpYVN0b3JlIG1ldHJpY3MgYW5kIENsb3VkV2F0Y2ggYWxhcm1zIGhlbHBzIG9wZXJhdG9ycyBnYWluIGJldHRlciBpbnNpZ2h0cyBpbnRvIGNvbnRhaW5lciBvcGVyYXRpb25zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBNZWRpYVN0b3JlQ2xvdWRXYXRjaE1ldHJpY1BvbGljeSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnTVM3JyxcbiAgICAgIGluZm86ICdUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgZGVmaW5lIGEgY29udGFpbmVyIHBvbGljeS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdVc2luZyBhIGNvbnRhaW5lciBwb2xpY3kgaGVscHMgZm9sbG93IHRoZSBzdGFuZGFyZCBzZWN1cml0eSBhZHZpY2Ugb2YgZ3JhbnRpbmcgbGVhc3QgcHJpdmlsZWdlLCBvciBncmFudGluZyBvbmx5IHRoZSBwZXJtaXNzaW9ucyByZXF1aXJlZCB0byBhbGxvdyBuZWVkZWQgYWNjZXNzIHRvIHRoZSBjb250YWluZXIuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICAgIHJ1bGU6IE1lZGlhU3RvcmVDb250YWluZXJIYXNDb250YWluZXJQb2xpY3ksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ01TOCcsXG4gICAgICBpbmZvOiAnVGhlIE1lZGlhU3RvcmUgY29udGFpbmVyIGRvZXMgbm90IGRlZmluZSBhIENPUlMgcG9saWN5LicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzaW5nIGEgQ09SUyBwb2xpY3kgaGVscHMgZm9sbG93IHRoZSBzdGFuZGFyZCBzZWN1cml0eSBhZHZpY2Ugb2YgZ3JhbnRpbmcgbGVhc3QgcHJpdmlsZWdlLCBvciBncmFudGluZyBvbmx5IHRoZSBwZXJtaXNzaW9ucyByZXF1aXJlZCB0byBhbGxvdyBuZWVkZWQgYWNjZXNzIHRvIHRoZSBjb250YWluZXIuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuV0FSTixcbiAgICAgIHJ1bGU6IE1lZGlhU3RvcmVDb250YWluZXJDT1JTUG9saWN5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBydWxlU3VmZml4T3ZlcnJpZGU6ICdNUzEwJyxcbiAgICAgIGluZm86ICdUaGUgTWVkaWFTdG9yZSBjb250YWluZXIgZG9lcyBub3QgZGVmaW5lIGEgbGlmZWN5Y2xlIHBvbGljeS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdNYW55IHVzZSBjYXNlcyB3YXJyYW50IHRoZSB1c2FnZSBvZiBsaWZlY3ljbGUgY29uZmlndXJhdGlvbnMgdG8gbWFuYWdlIGNvbnRhaW5lciBvYmplY3RzIGR1cmluZyB0aGVpciBsaWZldGltZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogTWVkaWFTdG9yZUNvbnRhaW5lckxpZmVjeWNsZVBvbGljeSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRGV2ZWxvcGVyIFRvb2xzIFNlcnZpY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0RldmVsb3BlclRvb2xzKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ0IzJyxcbiAgICAgIGluZm86ICdUaGUgQ29kZUJ1aWxkIHByb2plY3QgaGFzIHByaXZpbGVnZWQgbW9kZSBlbmFibGVkLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1ByaXZpbGVnZWQgZ3JhbnRzIGVsZXZhdGVkIHJpZ2h0cyB0byB0aGUgc3lzdGVtLCB3aGljaCBpbnRyb2R1Y2VzIGFkZGl0aW9uYWwgcmlzay4gUHJpdmlsZWdlZCBtb2RlIHNob3VsZCBvbmx5IGJlIHNldCB0byB0cnVlIG9ubHkgaWYgdGhlIGJ1aWxkIHByb2plY3QgaXMgdXNlZCB0byBidWlsZCBEb2NrZXIgaW1hZ2VzLiBPdGhlcndpc2UsIGEgYnVpbGQgdGhhdCBhdHRlbXB0cyB0byBpbnRlcmFjdCB3aXRoIHRoZSBEb2NrZXIgZGFlbW9uIGZhaWxzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBDb2RlQnVpbGRQcm9qZWN0UHJpdmlsZWdlZE1vZGVEaXNhYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQ0I0JyxcbiAgICAgIGluZm86ICdUaGUgQ29kZUJ1aWxkIHByb2plY3QgZG9lcyBub3QgdXNlIGFuIEFXUyBLTVMga2V5IGZvciBlbmNyeXB0aW9uLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzaW5nIGFuIEFXUyBLTVMga2V5IGhlbHBzIGZvbGxvdyB0aGUgc3RhbmRhcmQgc2VjdXJpdHkgYWR2aWNlIG9mIGdyYW50aW5nIGxlYXN0IHByaXZpbGVnZSB0byBvYmplY3RzIGdlbmVyYXRlZCBieSB0aGUgcHJvamVjdC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENvZGVCdWlsZFByb2plY3RLTVNFbmNyeXB0ZWRBcnRpZmFjdHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIHJ1bGVTdWZmaXhPdmVycmlkZTogJ0NCNScsXG4gICAgICBpbmZvOiAnVGhlIENvZGVidWlsZCBwcm9qZWN0IGRvZXMgbm90IHVzZSBpbWFnZXMgcHJvdmlkZWQgYnkgdGhlIENvZGVCdWlsZCBzZXJ2aWNlIG9yIGhhdmUgYSBjZGtfbmFnIHN1cHByZXNzaW9uIHJ1bGUgZXhwbGFpbmluZyB0aGUgbmVlZCBmb3IgYSBjdXN0b20gaW1hZ2UuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRXhwbGFpbmluZyBkaWZmZXJlbmNlcy9lZGl0cyB0byBEb2NrZXIgaW1hZ2VzIGhlbHBzIG9wZXJhdG9ycyBiZXR0ZXIgdW5kZXJzdGFuZCBzeXN0ZW0gZGVwZW5kZW5jaWVzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLldBUk4sXG4gICAgICBydWxlOiBDb2RlQnVpbGRQcm9qZWN0TWFuYWdlZEltYWdlcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgcnVsZVN1ZmZpeE92ZXJyaWRlOiAnQzkxJyxcbiAgICAgIGluZm86ICdUaGUgQ2xvdWQ5IGluc3RhbmNlIGRvZXMgbm90IHVzZSBhIG5vLWluZ3Jlc3MgRUMyIGluc3RhbmNlIHdpdGggQVdTIFN5c3RlbXMgTWFuYWdlci4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdTU00gYWRkcyBhbiBhZGRpdGlvbmFsIGxheWVyIG9mIHByb3RlY3Rpb24gYXMgaXQgYWxsb3dzIG9wZXJhdG9ycyB0byBjb250cm9sIGFjY2VzcyB0aHJvdWdoIElBTSBwZXJtaXNzaW9ucyBhbmQgZG9lcyBub3QgcmVxdWlyZSBvcGVuaW5nIGluYm91bmQgcG9ydHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDbG91ZDlJbnN0YW5jZU5vSW5ncmVzc1N5c3RlbXNNYW5hZ2VyLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxufVxuIl19