"""Validation helpers for plugin configuration dataclasses"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/utils/validation.ipynb.

# %% auto 0
__all__ = ['T', 'SCHEMA_TITLE', 'SCHEMA_DESC', 'SCHEMA_MIN', 'SCHEMA_MAX', 'SCHEMA_ENUM', 'SCHEMA_MIN_LEN', 'SCHEMA_MAX_LEN',
           'SCHEMA_PATTERN', 'SCHEMA_FORMAT', 'validate_field_value', 'validate_config', 'config_to_dict',
           'dict_to_config', 'extract_defaults']

# %% ../../nbs/utils/validation.ipynb 3
from dataclasses import fields, is_dataclass, asdict, MISSING
from typing import Dict, Any, Tuple, Optional, Type, TypeVar

T = TypeVar('T')

# %% ../../nbs/utils/validation.ipynb 5
SCHEMA_TITLE = "title"        # Display title for the field
SCHEMA_DESC = "description"   # Help text description
SCHEMA_MIN = "minimum"        # Minimum value for numbers
SCHEMA_MAX = "maximum"        # Maximum value for numbers
SCHEMA_ENUM = "enum"          # Allowed values for dropdowns
SCHEMA_MIN_LEN = "minLength"  # Minimum string length
SCHEMA_MAX_LEN = "maxLength"  # Maximum string length
SCHEMA_PATTERN = "pattern"    # Regex pattern for strings
SCHEMA_FORMAT = "format"      # String format (email, uri, date, etc.)

# %% ../../nbs/utils/validation.ipynb 7
import re

def validate_field_value(
    value:Any, # Value to validate
    metadata:Dict[str, Any], # Field metadata containing constraints
    field_name:str="" # Field name for error messages
) -> Tuple[bool, Optional[str]]: # (is_valid, error_message)
    """Validate a value against field metadata constraints."""
    # Check enum constraint
    if SCHEMA_ENUM in metadata:
        allowed = metadata[SCHEMA_ENUM]
        if value not in allowed:
            return False, f"{field_name}: {value!r} is not one of {allowed}"
    
    # Check numeric constraints
    if isinstance(value, (int, float)) and not isinstance(value, bool):
        if SCHEMA_MIN in metadata and value < metadata[SCHEMA_MIN]:
            return False, f"{field_name}: {value} is less than minimum {metadata[SCHEMA_MIN]}"
        if SCHEMA_MAX in metadata and value > metadata[SCHEMA_MAX]:
            return False, f"{field_name}: {value} is greater than maximum {metadata[SCHEMA_MAX]}"
    
    # Check string constraints
    if isinstance(value, str):
        if SCHEMA_MIN_LEN in metadata and len(value) < metadata[SCHEMA_MIN_LEN]:
            return False, f"{field_name}: string length {len(value)} is less than minimum {metadata[SCHEMA_MIN_LEN]}"
        if SCHEMA_MAX_LEN in metadata and len(value) > metadata[SCHEMA_MAX_LEN]:
            return False, f"{field_name}: string length {len(value)} is greater than maximum {metadata[SCHEMA_MAX_LEN]}"
        if SCHEMA_PATTERN in metadata:
            pattern = metadata[SCHEMA_PATTERN]
            if not re.match(pattern, value):
                return False, f"{field_name}: {value!r} does not match pattern {pattern!r}"
    
    return True, None

# %% ../../nbs/utils/validation.ipynb 8
def validate_config(
    config:Any # Configuration dataclass instance to validate
) -> Tuple[bool, Optional[str]]: # (is_valid, error_message)
    """Validate all fields in a configuration dataclass against their metadata constraints."""
    if not is_dataclass(config) or isinstance(config, type):
        raise TypeError(f"Expected dataclass instance, got {type(config).__name__}")
    
    for f in fields(config):
        value = getattr(config, f.name)
        metadata = f.metadata or {}
        
        is_valid, error = validate_field_value(value, metadata, f.name)
        if not is_valid:
            return False, error
    
    return True, None

# %% ../../nbs/utils/validation.ipynb 10
def config_to_dict(
    config:Any # Configuration dataclass instance
) -> Dict[str, Any]: # Dictionary representation of the configuration
    """Convert a configuration dataclass instance to a dictionary."""
    if is_dataclass(config) and not isinstance(config, type):
        return asdict(config)
    elif isinstance(config, dict):
        return config
    else:
        raise TypeError(f"Expected dataclass instance or dict, got {type(config).__name__}")

# %% ../../nbs/utils/validation.ipynb 12
def dict_to_config(
    config_class:Type[T], # Configuration dataclass type
    data:Optional[Dict[str, Any]]=None, # Dictionary with configuration values
    validate:bool=False # Whether to validate against metadata constraints
) -> T: # Instance of the configuration dataclass
    """Create a configuration dataclass instance from a dictionary."""
    if not is_dataclass(config_class):
        raise TypeError(f"Expected dataclass type, got {type(config_class).__name__}")
    
    if data is None:
        data = {}
    
    # Get valid field names for this dataclass
    valid_fields = {f.name for f in fields(config_class)}
    
    # Filter data to only include valid fields
    filtered_data = {k: v for k, v in data.items() if k in valid_fields}
    
    # Create the config instance
    config = config_class(**filtered_data)
    
    # Optionally validate
    if validate:
        is_valid, error = validate_config(config)
        if not is_valid:
            raise ValueError(error)
    
    return config

# %% ../../nbs/utils/validation.ipynb 13
def extract_defaults(
    config_class:Type # Configuration dataclass type
) -> Dict[str, Any]: # Default values from the dataclass
    """Extract default values from a configuration dataclass type."""
    if not is_dataclass(config_class):
        raise TypeError(f"Expected dataclass type, got {type(config_class).__name__}")
    
    defaults = {}
    for f in fields(config_class):
        if f.default is not MISSING:
            defaults[f.name] = f.default
        elif f.default_factory is not MISSING:
            defaults[f.name] = f.default_factory()
    
    return defaults
