"""Plugin discovery, loading, and lifecycle management system"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/core/manager.ipynb.

# %% auto 0
__all__ = ['PluginManager', 'get_plugin_config_dataclass', 'get_all_plugin_config_dataclasses', 'get_plugin_config',
           'get_plugin_config_class', 'validate_plugin_config', 'update_plugin_config', 'reload_plugin',
           'execute_plugin_stream', 'check_streaming_support', 'get_streaming_plugins']

# %% ../../nbs/core/manager.ipynb 3
import importlib.metadata
import importlib.util
import inspect
import logging
from pathlib import Path
import sys
from typing import Dict, List, Optional, Type, Any, Tuple, Generator
from dataclasses import dataclass

from .interface import PluginInterface
from .metadata import PluginMeta
from ..utils.validation import config_to_dict, validate_config

# %% ../../nbs/core/manager.ipynb 4
# Optional: Import error handling library if available
try:
    from cjm_error_handling.core.base import ErrorContext, ErrorSeverity
    from cjm_error_handling.core.errors import PluginError, ValidationError
    _has_error_handling = True
except ImportError:
    _has_error_handling = False

# %% ../../nbs/core/manager.ipynb 7
class PluginManager:
    """Manages plugin discovery, loading, and lifecycle."""

    def __init__(
        self,
        plugin_interface:Type[PluginInterface]=PluginInterface, # Base class/interface plugins must implement
        entry_point_group:Optional[str]=None # Optional override for entry point group name
    ):
        """Initialize the plugin manager."""
        self.plugin_interface = plugin_interface

        # Use entry_point_group from plugin interface if not explicitly provided
        if entry_point_group is None:
            self.entry_point_group = plugin_interface.entry_point_group
        else:
            self.entry_point_group = entry_point_group

        self.entry_points = []
        self.discovered = []
        self.plugins: Dict[str, PluginMeta] = {}
        self._loaded_modules: Dict[str, Any] = {}
        self.logger = logging.getLogger(f"{__name__}.{type(self).__name__}")
        # Get plugin entry points
        self.get_entry_points()

    def get_entry_points(self) -> importlib.metadata.EntryPoints: # Entry points for the configured group
        """Get plugin entry points from installed packages."""
        self.entry_points = []
        try:
            # For Python 3.10+
            entry_points = importlib.metadata.entry_points()

            # Get entry points for our group
            self.entry_points = entry_points.select(group=self.entry_point_group)
        except Exception as e:
            self.logger.error(f"Error accessing entry points: {e}")
            return self.entry_points

    def discover_plugins(self) -> List[PluginMeta]: # List of discovered plugin metadata objects
        """Discover all installed plugins via entry points."""
        self.discovered = []

        for ep in self.entry_points:
            try:
                # Get the entry point name (plugin name)
                plugin_name = ep.name

                # Get package metadata
                # In Python 3.11+, ep.dist gives us the distribution
                if hasattr(ep, 'dist') and ep.dist:
                    # Get package name and version from the distribution
                    package_name = ep.dist.name  # This is the package name
                    version = ep.dist.version
                else:
                    # Fallback: try to get distribution info from entry point value
                    # The value is like "package.module:ClassName"
                    package_name = plugin_name  # Use plugin name as fallback
                    version = "unknown"

                    # Try to get the actual package info
                    try:
                        # Extract package name from the entry point value
                        ep_value = str(ep.value) if hasattr(ep, 'value') else str(ep)
                        package_part = ep_value.split(':')[0].split('.')[0]

                        # Try to get distribution info
                        dist = importlib.metadata.distribution(package_part)
                        package_name = dist.name
                        version = dist.version
                    except Exception:
                        pass

                meta = PluginMeta(
                    name=plugin_name,
                    version=version,
                    package_name=package_name
                )
                self.discovered.append(meta)
                self.logger.info(f"Discovered plugin: {meta.name} v{meta.version} from package {meta.package_name}")

            except Exception as e:
                self.logger.error(f"Error discovering plugin {ep.name}: {e}")

        return self.discovered

    def load_plugin(
        self,
        plugin_meta:PluginMeta, # Plugin metadata
        config:Optional[Dict[str, Any]]=None # Optional configuration for the plugin
    ) -> bool: # True if successfully loaded, False otherwise
        """Load and initialize a plugin."""
        try:
            # Find the entry point and load it
            plugin_eps = self.entry_points.select(name=plugin_meta.name)

            ep = list(plugin_eps)[0]
            plugin_class = ep.load()

            # Verify it implements the required interface
            if not issubclass(plugin_class, self.plugin_interface):
                error_msg = f"Plugin {plugin_meta.name} does not implement required interface"
                self.logger.error(error_msg)
                if _has_error_handling:
                    raise PluginError(
                        message=f"Plugin does not implement required interface: {plugin_meta.name}",
                        debug_info=f"Expected {self.plugin_interface.__name__}, got {plugin_class.__name__}",
                        context=ErrorContext(
                            operation="load_plugin",
                            extra={"package_name": plugin_meta.package_name}
                        ),
                        plugin_id=plugin_meta.name
                    )
                return False

            # Instantiate and initialize the plugin
            plugin_instance = plugin_class()
            plugin_instance.initialize(config)

            # Store the plugin
            plugin_meta.instance = plugin_instance
            self.plugins[plugin_meta.name] = plugin_meta

            self.logger.info(f"Loaded plugin: {plugin_meta.name}")
            return True

        except PluginError:
            # Re-raise structured errors
            raise
        except Exception as e:
            self.logger.error(f"Error loading plugin {plugin_meta.name}: {e}")
            if _has_error_handling:
                raise PluginError(
                    message=f"Failed to load plugin: {plugin_meta.name}",
                    debug_info=f"Error during plugin loading: {str(e)}",
                    context=ErrorContext(
                        operation="load_plugin",
                        extra={"package_name": plugin_meta.package_name}
                    ),
                    plugin_id=plugin_meta.name,
                    cause=e
                )
            return False

    def load_plugin_from_module(
        self,
        module_path:str, # Path to the Python module
        config:Optional[Dict[str, Any]]=None # Optional configuration for the plugin
    ) -> bool: # True if successfully loaded, False otherwise
        """Load a plugin directly from a Python module file or package."""
        try:
            # Convert to Path object
            path = Path(module_path)

            # Load the module
            if path.is_file():
                # Single file module
                spec = importlib.util.spec_from_file_location(path.stem, path)
                module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(module)
            else:
                # Package directory
                sys.path.insert(0, str(path.parent))
                module = importlib.import_module(path.name)

            # Find plugin classes in the module
            plugin_classes = []
            for name, obj in inspect.getmembers(module):
                if (inspect.isclass(obj) and
                    issubclass(obj, self.plugin_interface) and
                    obj != self.plugin_interface):
                    plugin_classes.append(obj)

            if not plugin_classes:
                error_msg = f"No plugin classes found in {module_path}"
                self.logger.error(error_msg)
                if _has_error_handling:
                    raise PluginError(
                        message=f"No plugin classes found in module",
                        debug_info=f"Module path: {module_path}. Expected subclass of {self.plugin_interface.__name__}",
                        context=ErrorContext(
                            operation="load_plugin_from_module",
                            extra={"module_path": str(module_path)}
                        ),
                        plugin_id=str(module_path)
                    )
                return False

            # Load the first plugin class found
            plugin_class = plugin_classes[0]
            plugin_instance = plugin_class()
            plugin_instance.initialize(config)

            # Create metadata
            meta = PluginMeta(
                name=plugin_instance.name,
                version=plugin_instance.version,
                package_name=str(module_path)
            )
            meta.instance = plugin_instance

            self.plugins[meta.name] = meta
            self._loaded_modules[meta.name] = module

            self.logger.info(f"Loaded plugin from module: {meta.name}")
            return True

        except PluginError:
            # Re-raise structured errors
            raise
        except Exception as e:
            self.logger.error(f"Error loading plugin from {module_path}: {e}")
            if _has_error_handling:
                raise PluginError(
                    message=f"Failed to load plugin from module",
                    debug_info=f"Module path: {module_path}. Error: {str(e)}",
                    context=ErrorContext(
                        operation="load_plugin_from_module",
                        extra={"module_path": str(module_path)}
                    ),
                    plugin_id=str(module_path),
                    cause=e
                )
            return False

    def unload_plugin(
        self,
        plugin_name:str # Name of the plugin to unload
    ) -> bool: # True if successfully unloaded, False otherwise
        """Unload a plugin and call its cleanup method."""
        if plugin_name not in self.plugins:
            error_msg = f"Plugin {plugin_name} not found"
            self.logger.error(error_msg)
            if _has_error_handling:
                raise PluginError(
                    message=f"Cannot unload plugin: not found",
                    debug_info=f"Plugin {plugin_name} is not loaded",
                    context=ErrorContext(operation="unload_plugin"),
                    plugin_id=plugin_name
                )
            return False

        try:
            plugin_meta = self.plugins[plugin_name]
            if plugin_meta.instance:
                plugin_meta.instance.cleanup()

            del self.plugins[plugin_name]

            # Remove from loaded modules if it was loaded that way
            if plugin_name in self._loaded_modules:
                del self._loaded_modules[plugin_name]

            self.logger.info(f"Unloaded plugin: {plugin_name}")
            return True

        except Exception as e:
            self.logger.error(f"Error unloading plugin {plugin_name}: {e}")
            if _has_error_handling:
                raise PluginError(
                    message=f"Failed to unload plugin: {plugin_name}",
                    debug_info=f"Error during cleanup: {str(e)}",
                    context=ErrorContext(operation="unload_plugin"),
                    plugin_id=plugin_name,
                    cause=e
                )
            return False

    def get_plugin(
        self,
        plugin_name:str # Name of the plugin to retrieve
    ) -> Optional[PluginInterface]: # Plugin instance if found, None otherwise
        """Get a loaded plugin instance by name."""
        if plugin_name in self.plugins:
            return self.plugins[plugin_name].instance
        return None

    def list_plugins(self) -> List[PluginMeta]: # List of metadata for all loaded plugins
        """List all loaded plugins."""
        return list(self.plugins.values())

    def execute_plugin(
        self,
        plugin_name:str, # Name of the plugin to execute
        *args, # Arguments to pass to the plugin
        **kwargs # Keyword arguments to pass to the plugin
    ) -> Any: # Result of the plugin execution
        """Execute a plugin's main functionality."""
        plugin = self.get_plugin(plugin_name)
        if not plugin:
            if _has_error_handling:
                raise PluginError(
                    message=f"Cannot execute plugin: not found or not loaded",
                    debug_info=f"Plugin {plugin_name} must be loaded before execution",
                    context=ErrorContext(operation="execute_plugin"),
                    plugin_id=plugin_name
                )
            raise ValueError(f"Plugin {plugin_name} not found or not loaded")

        if not self.plugins[plugin_name].enabled:
            if _has_error_handling:
                raise PluginError(
                    message=f"Cannot execute plugin: disabled",
                    debug_info=f"Plugin {plugin_name} is currently disabled",
                    context=ErrorContext(operation="execute_plugin"),
                    plugin_id=plugin_name
                )
            raise ValueError(f"Plugin {plugin_name} is disabled")

        try:
            return plugin.execute(*args, **kwargs)
        except Exception as e:
            if _has_error_handling:
                raise PluginError(
                    message=f"Plugin execution failed: {plugin_name}",
                    debug_info=f"Error during plugin execution: {str(e)}",
                    context=ErrorContext(operation="execute_plugin"),
                    plugin_id=plugin_name,
                    cause=e
                )
            raise

    def enable_plugin(
        self,
        plugin_name:str # Name of the plugin to enable
    ) -> bool: # True if plugin was enabled, False if not found
        """Enable a plugin."""
        if plugin_name in self.plugins:
            self.plugins[plugin_name].enabled = True
            return True
        return False

    def disable_plugin(
        self,
        plugin_name:str # Name of the plugin to disable
    ) -> bool: # True if plugin was disabled, False if not found
        """Disable a plugin without unloading it."""
        if plugin_name in self.plugins:
            self.plugins[plugin_name].enabled = False
            return True
        return False

# %% ../../nbs/core/manager.ipynb 10
def get_plugin_config_dataclass(
    self,
    plugin_name:str # Name of the plugin
) -> dataclass: # Current configuration dataclass
    """Get the configuration dataclass for a plugin."""

    # Find the entry point
    plugin_eps = self.entry_points.select(name=plugin_name)

    if len(plugin_eps) > 0:
        ep = list(plugin_eps)[0]
        plugin_class = ep.load()
        return type(plugin.get_current_config())
    else:
        return None

def get_all_plugin_config_dataclasses(
    self
) -> Dict[str, dataclass]:
    """Get configuration dataclasses for all loaded plugins."""
    config_dataclasses = {}

    for plugin_name in self.plugins:
        plugin = self.get_plugin(plugin_name)
        if plugin:
            config_dataclasses[plugin_name] = type(plugin.get_current_config())
    return config_dataclasses

def get_plugin_config(
    self,
    plugin_name:str # Name of the plugin
) -> Optional[Any]: # Current configuration dataclass instance or None if plugin not found
    """Get the current configuration of a plugin."""
    plugin = self.get_plugin(plugin_name)
    if plugin:
        return plugin.get_current_config()
    return None

def get_plugin_config_class(
    self,
    plugin_name:str # Name of the plugin
) -> Optional[Type]: # Configuration dataclass type or None if plugin not found
    """Get the configuration dataclass type for a plugin."""
    plugin = self.get_plugin(plugin_name)
    if plugin and hasattr(plugin, 'config_class'):
        return plugin.config_class
    return None

def validate_plugin_config(
    self,
    plugin_name:str, # Name of the plugin
    config:Any # Configuration dataclass instance to validate
) -> Tuple[bool, Optional[str]]: # (is_valid, error_message)
    """Validate a configuration dataclass for a plugin."""
    plugin = self.get_plugin(plugin_name)
    if not plugin:
        return False, f"Plugin {plugin_name} not found"
    
    return validate_config(config)

def update_plugin_config(
    self,
    plugin_name:str, # Name of the plugin
    config:Any, # New configuration (dataclass instance or dict)
    merge:bool=True # Whether to merge with existing config or replace entirely
) -> bool: # True if successful, False otherwise
    """Update a plugin's configuration and reinitialize it."""
    plugin = self.get_plugin(plugin_name)
    if not plugin:
        error_msg = f"Plugin {plugin_name} not found"
        self.logger.error(error_msg)
        if _has_error_handling:
            raise PluginError(
                message=f"Cannot update config: plugin not found",
                debug_info=f"Plugin {plugin_name} is not loaded",
                context=ErrorContext(operation="update_plugin_config"),
                plugin_id=plugin_name
            )
        return False

    try:
        # If merging, convert current config to dict, merge, then pass to initialize
        if merge and isinstance(config, dict):
            current_config = plugin.get_current_config()
            current_dict = config_to_dict(current_config)
            config = {**current_dict, **config}

        # Clean up existing resources
        plugin.cleanup()

        # Reinitialize with new config (plugin's initialize handles validation)
        plugin.initialize(config)

        self.logger.info(f"Updated configuration for plugin: {plugin_name}")
        return True

    except (ValueError, TypeError) as e:
        # Validation or type errors from initialize
        error_msg = f"Invalid configuration for {plugin_name}: {e}"
        self.logger.error(error_msg)
        if _has_error_handling:
            raise ValidationError(
                message=f"Invalid plugin configuration",
                debug_info=f"Validation failed for plugin {plugin_name}: {str(e)}",
                context=ErrorContext(
                    operation="update_plugin_config",
                    extra={"plugin_name": plugin_name}
                ),
                validation_errors={"config": str(e)}
            )
        return False
    except (ValidationError, PluginError):
        # Re-raise structured errors
        raise
    except Exception as e:
        error_msg = f"Error updating plugin {plugin_name} configuration: {e}"
        self.logger.error(error_msg)
        if _has_error_handling:
            raise PluginError(
                message=f"Failed to update plugin configuration",
                debug_info=f"Error reinitializing plugin {plugin_name}: {str(e)}",
                context=ErrorContext(
                    operation="update_plugin_config",
                    extra={"plugin_name": plugin_name}
                ),
                plugin_id=plugin_name,
                cause=e
            )
        return False

def reload_plugin(
    self,
    plugin_name:str, # Name of the plugin to reload
    config:Optional[Any]=None # Optional new configuration (dataclass or dict)
) -> bool: # True if successful, False otherwise
    """Reload a plugin with optional new configuration."""
    if plugin_name not in self.plugins:
        error_msg = f"Plugin {plugin_name} not found"
        self.logger.error(error_msg)
        if _has_error_handling:
            raise PluginError(
                message=f"Cannot reload plugin: not found",
                debug_info=f"Plugin {plugin_name} is not loaded",
                context=ErrorContext(operation="reload_plugin"),
                plugin_id=plugin_name
            )
        return False

    try:
        # Get the plugin metadata
        plugin_meta = self.plugins[plugin_name]

        # Get current config if not provided
        if config is None and plugin_meta.instance:
            config = plugin_meta.instance.get_current_config()

        # Unload the plugin
        if plugin_meta.instance:
            plugin_meta.instance.cleanup()

        # Reload based on how it was originally loaded
        if plugin_name in self._loaded_modules:
            # Was loaded from a module file
            module_path = plugin_meta.package_name
            del self.plugins[plugin_name]
            del self._loaded_modules[plugin_name]
            return self.load_plugin_from_module(module_path, config)
        else:
            # Was loaded from entry points
            plugin_meta.instance = None
            return self.load_plugin(plugin_meta, config)

    except (PluginError, ValidationError):
        # Re-raise structured errors
        raise
    except Exception as e:
        error_msg = f"Error reloading plugin {plugin_name}: {e}"
        self.logger.error(error_msg)
        if _has_error_handling:
            raise PluginError(
                message=f"Failed to reload plugin",
                debug_info=f"Error during plugin reload: {str(e)}",
                context=ErrorContext(
                    operation="reload_plugin",
                    extra={"plugin_name": plugin_name}
                ),
                plugin_id=plugin_name,
                cause=e
            )
        return False

# Add to the PluginManager class
PluginManager.get_plugin_config_dataclass = get_plugin_config_dataclass
PluginManager.get_all_plugin_config_dataclasses = get_all_plugin_config_dataclasses
PluginManager.get_plugin_config = get_plugin_config
PluginManager.get_plugin_config_class = get_plugin_config_class
PluginManager.validate_plugin_config = validate_plugin_config
PluginManager.update_plugin_config = update_plugin_config
PluginManager.reload_plugin = reload_plugin

# %% ../../nbs/core/manager.ipynb 12
def execute_plugin_stream(
    self,
    plugin_name:str, # Name of the plugin to execute
    *args, # Arguments to pass to the plugin
    **kwargs # Keyword arguments to pass to the plugin
) -> Generator[Any, None, Any]: # Generator yielding partial results, returns final result
    """Execute a plugin with streaming support if available."""
    plugin = self.get_plugin(plugin_name)
    if not plugin:
        if _has_error_handling:
            raise PluginError(
                message=f"Cannot execute plugin: not found or not loaded",
                debug_info=f"Plugin {plugin_name} must be loaded before execution",
                context=ErrorContext(operation="execute_plugin_stream"),
                plugin_id=plugin_name
            )
        raise ValueError(f"Plugin {plugin_name} not found or not loaded")

    if not self.plugins[plugin_name].enabled:
        if _has_error_handling:
            raise PluginError(
                message=f"Cannot execute plugin: disabled",
                debug_info=f"Plugin {plugin_name} is currently disabled",
                context=ErrorContext(operation="execute_plugin_stream"),
                plugin_id=plugin_name
            )
        raise ValueError(f"Plugin {plugin_name} is disabled")

    # Check if plugin supports streaming
    if hasattr(plugin, 'supports_streaming') and plugin.supports_streaming():
        self.logger.info(f"Using streaming mode for plugin {plugin_name}")
        try:
            return plugin.execute_stream(*args, **kwargs)
        except Exception as e:
            if _has_error_handling:
                raise PluginError(
                    message=f"Plugin streaming execution failed: {plugin_name}",
                    debug_info=f"Error during plugin streaming: {str(e)}",
                    context=ErrorContext(operation="execute_plugin_stream"),
                    plugin_id=plugin_name,
                    cause=e
                )
            raise
    else:
        self.logger.info(f"Plugin {plugin_name} doesn't support streaming, using regular execution")
        # Fall back to regular execution wrapped in a generator
        def fallback_generator():
            try:
                result = plugin.execute(*args, **kwargs)
                yield result
                return result
            except Exception as e:
                if _has_error_handling:
                    raise PluginError(
                        message=f"Plugin execution failed: {plugin_name}",
                        debug_info=f"Error during plugin execution: {str(e)}",
                        context=ErrorContext(operation="execute_plugin_stream"),
                        plugin_id=plugin_name,
                        cause=e
                    )
                raise
        return fallback_generator()

def check_streaming_support(
    self,
    plugin_name:str # Name of the plugin to check
) -> bool: # True if plugin supports streaming
    """Check if a plugin supports streaming execution."""
    plugin = self.get_plugin(plugin_name)
    if not plugin:
        return False

    # Check if plugin has the supports_streaming method and it returns True
    if hasattr(plugin, 'supports_streaming'):
        return plugin.supports_streaming()

    # Fallback: check if execute_stream is implemented
    if hasattr(plugin, 'execute_stream'):
        # Check if it's overridden from base class
        plugin_class = type(plugin)
        if hasattr(plugin_class, 'execute_stream'):
            # Try to check if it's different from PluginInterface's default
            return True

    return False

def get_streaming_plugins(
    self
) -> List[str]: # List of plugin names that support streaming
    """Get a list of all loaded plugins that support streaming."""
    streaming_plugins = []
    for plugin_name in self.plugins:
        if self.check_streaming_support(plugin_name):
            streaming_plugins.append(plugin_name)
    return streaming_plugins

# Add the methods to the PluginManager class
PluginManager.execute_plugin_stream = execute_plugin_stream
PluginManager.check_streaming_support = check_streaming_support
PluginManager.get_streaming_plugins = get_streaming_plugins
