"""Abstract base class defining the generic plugin interface"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/core/interface.ipynb.

# %% auto 0
__all__ = ['PluginInterface', 'PluginInterface_supports_streaming', 'PluginInterface_execute_stream']

# %% ../../nbs/core/interface.ipynb 3
from abc import ABC, abstractmethod
from dataclasses import dataclass
import logging
from typing import Optional, Dict, Any, Tuple, Generator, Type
from cjm_plugin_system.utils.validation import (
    dict_to_config, config_to_dict, extract_defaults, validate_config,
    SCHEMA_TITLE, SCHEMA_DESC, SCHEMA_MIN, SCHEMA_MAX, SCHEMA_ENUM
)

# %% ../../nbs/core/interface.ipynb 5
class PluginInterface(ABC):
    """Generic plugin interface that all plugins must implement."""

    # Must be overridden by domain-specific interfaces
    entry_point_group: str = None
    
    # Configuration dataclass type - override in concrete plugins
    config_class: Type = None

    def __init_subclass__(cls, **kwargs):
        """Enforce that domain-specific interfaces define entry_point_group."""
        super().__init_subclass__(**kwargs)

        # Check if this is a direct subclass of PluginInterface
        # (i.e., a domain-specific interface like TranscriptionPlugin)
        if PluginInterface in cls.__bases__:
            # Domain-specific interfaces must define entry_point_group
            if not hasattr(cls, 'entry_point_group') or cls.entry_point_group is None:
                raise TypeError(
                    f"Domain-specific interface '{cls.__name__}' must define "
                    f"'entry_point_group' class attribute. Example:\n"
                    f"    class {cls.__name__}(PluginInterface):\n"
                    f"        entry_point_group = 'your.plugins'"
                )

    @property
    @abstractmethod
    def name(self) -> str: # Unique identifier for this plugin
        """Unique plugin identifier."""
        pass

    @property
    @abstractmethod
    def version(self) -> str: # Semantic version string (e.g., "1.0.0")
        """Plugin version."""
        pass

    @abstractmethod
    def initialize(
        self,
        config:Optional[Any]=None # Configuration dataclass instance or dict
    ) -> None:
        """Initialize the plugin with configuration."""
        pass

    @abstractmethod
    def execute(
        self,
        *args,
        **kwargs
    ) -> Any: # Plugin-specific output
        """Execute the plugin's main functionality."""
        pass

    @abstractmethod
    def is_available(self) -> bool: # True if all required dependencies are available
        """Check if the plugin's dependencies are available."""
        pass

    @abstractmethod
    def get_current_config(self) -> Any: # Current configuration dataclass instance
        """Return the current configuration state."""
        pass

    def get_config_defaults(self) -> Dict[str, Any]: # Default values from config_class
        """Extract default values from the configuration dataclass."""
        if self.config_class is None:
            return {}
        return extract_defaults(self.config_class)

    def cleanup(self) -> None:
        """Optional cleanup when plugin is unloaded."""
        pass

# %% ../../nbs/core/interface.ipynb 8
def PluginInterface_supports_streaming(self) -> bool: # True if execute_stream is implemented
    """Check if this plugin supports streaming execution."""
    # Default: check if execute_stream is overridden from the base class
    return type(self).execute_stream != PluginInterface.execute_stream

def PluginInterface_execute_stream(
    self,
    *args, # Arguments for plugin execution
    **kwargs # Keyword arguments for plugin execution
) -> Generator[Any, None, Any]: # Yields partial results, returns final result
    """Stream execution results chunk by chunk."""
    # Default implementation: execute normally and yield complete result at once
    result = self.execute(*args, **kwargs)
    yield result
    return result

# Add the methods to the PluginInterface class
PluginInterface.supports_streaming = PluginInterface_supports_streaming
PluginInterface.execute_stream = PluginInterface_execute_stream
