#!/usr/bin/python

"""
gsuite_sync.gsuite_pull contains the functions used to pull Chrome device data
from a GSuite account.
"""


# Installed Libraries
from __future__ import print_function
from googleapiclient.discovery import build
from google.oauth2.credentials import Credentials
from builtins import input
import argparse

# Built-In Libraries
import pickle
import json
import logging
import os
import datetime


# log (console) is used to output data to the console properly formatted
log = logging.getLogger("console")
# bulklog is used to output structured data without formatting
bulklog = logging.getLogger("bulk")


def get_service(credfile):
    log.debug("gsuite_sync.gsuite_pull.get_service:\
 Pulling credential data from ({})".format(credfile))
    SCOPES = [
        'https://www.googleapis.com/auth/admin.directory.user',
        'https://www.googleapis.com/auth/admin.directory.device.chromeos']
    creds = Credentials.from_authorized_user_file(credfile, SCOPES)
    log.debug("gsuite_sync.gsuite_pull.get_service:\
 Pulled credentials. Connecting to GSuite")
    service = build('admin', 'directory_v1', credentials=creds)
    log.info("gsuite_sync.gsuite_pull.get_service:\
 Successfully connected to Google!")
    return service


def pull_devices(service, cache=True, query_days=None):
    devices = []
    #########################################################
    if cache:
        if os.path.isfile("/opt/gsync_cache/google_cache.json"):
            f = open("/opt/gsync_cache/google_cache.json")
            data = json.loads(f.read())
            f.close()
            return data
    #########################################################
    if query_days:
        today = datetime.datetime.now()
        delta = datetime.timedelta(days=query_days)
        date = today - delta
        query = f'register:{date.strftime("%Y-%m-%d")}..'
    else:
        query = None
    request = service.chromeosdevices().list(customerId='my_customer', query=query)
    response = request.execute()
    bulklog.info("gsuite_sync.gsuite_pull.pull_devices:\
 Response:\n{}".format(json.dumps(response, indent=4)))
    devices += response["chromeosdevices"]
    log.info("gsuite_sync.gsuite_pull.pull_devices:\
 Inventoried ({}) devices so far".format(len(devices)))
    try:
        while "nextPageToken" in response:
            request = service.chromeosdevices().list_next(request, response)
            response = request.execute()
            bulklog.info("gsuite_sync.gsuite_pull.pull_devices:\
 Response:\n{}".format(json.dumps(response, indent=4)))
            devices += response["chromeosdevices"]
            log.info("gsuite_sync.gsuite_pull.pull_devices:\
 Inventoried ({}) devices so far".format(len(devices)))
        bulklog.info("gsuite_sync.gsuite_pull.pull_devices:\
     All devices:\n{}".format(json.dumps(devices, indent=4)))
    except KeyboardInterrupt:
        log.warning("gsuite_sync.gsuite_pull.pull_devices:\
 Stopped, returning the ({}) devices we have so far".format(len(devices)))
    #########################################################
    if cache:
        f = open("/opt/gsync_cache/google_cache.json", "w")
        f.write(json.dumps(devices, indent=4))
        f.close()
    #########################################################
    return devices


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument(
                        '-gc', "--gsuite_credential",
                        help="GSuite Credential File",
                        metavar='CRED_FILE',
                        dest="gsuite_credential")
    args = parser.parse_args()
    print(json.dumps(pull_devices(args.gsuite_credential), indent=4))
