"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCluster = exports.AuroraCapacityUnit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const endpoint_1 = require("./endpoint");
const perms_1 = require("./perms");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * Aurora capacity units (ACUs).
 *
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 * @stability stable
 */
var AuroraCapacityUnit;
(function (AuroraCapacityUnit) {
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_1"] = 1] = "ACU_1";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_2"] = 2] = "ACU_2";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_4"] = 4] = "ACU_4";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_8"] = 8] = "ACU_8";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_16"] = 16] = "ACU_16";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_32"] = 32] = "ACU_32";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_64"] = 64] = "ACU_64";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_128"] = 128] = "ACU_128";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_192"] = 192] = "ACU_192";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_256"] = 256] = "ACU_256";
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_384"] = 384] = "ACU_384";
})(AuroraCapacityUnit = exports.AuroraCapacityUnit || (exports.AuroraCapacityUnit = {}));
/**
 * New or imported Serverless Cluster
 */
class ServerlessClusterBase extends core_1.Resource {
    /**
     * The ARN of the cluster.
     *
     * @stability stable
     */
    get clusterArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            sep: ':',
            resourceName: this.clusterIdentifier,
        });
    }
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantDataApiAccess(grantee) {
        var _b;
        if (this.enableDataApi === false) {
            throw new Error('Cannot grant Data API access when the Data API is disabled');
        }
        this.enableDataApi = true;
        const ret = iam.Grant.addToPrincipal({
            grantee,
            actions: perms_1.DATA_API_ACTIONS,
            resourceArns: ['*'],
            scope: this,
        });
        (_b = this.secret) === null || _b === void 0 ? void 0 : _b.grantRead(grantee);
        return ret;
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create an Aurora Serverless Cluster.
 *
 * @stability stable
 * @resource AWS::RDS::DBCluster
 */
class ServerlessCluster extends ServerlessClusterBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.enableDataApi = props.enableDataApi;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_b = props.subnetGroup) !== null && _b !== void 0 ? _b : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        if (props.backupRetention) {
            const backupRetentionDays = props.backupRetention.toDays();
            if (backupRetentionDays < 1 || backupRetentionDays > 35) {
                throw new Error(`backup retention period must be between 1 and 35 days. received: ${backupRetentionDays}`);
            }
        }
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_c = props.parameterGroup) !== null && _c !== void 0 ? _c : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: this.vpc,
            }),
        ];
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_e = props.clusterIdentifier) === null || _e === void 0 ? void 0 : _e.toLowerCase() : props.clusterIdentifier;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            backupRetentionPeriod: (_f = props.backupRetention) === null || _f === void 0 ? void 0 : _f.toDays(),
            databaseName: props.defaultDatabaseName,
            dbClusterIdentifier: clusterIdentifier,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            engine: props.engine.engineType,
            engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
            engineMode: 'serverless',
            enableHttpEndpoint: core_1.Lazy.anyValue({ produce: () => this.enableDataApi }),
            kmsKeyId: (_h = props.storageEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: (_j = credentials.password) === null || _j === void 0 ? void 0 : _j.toString(),
            scalingConfiguration: props.scaling ? this.renderScalingConfiguration(props.scaling) : undefined,
            storageEncrypted: true,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_k = props.removalPolicy) !== null && _k !== void 0 ? _k : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
    /**
     * Import an existing DatabaseCluster from properties.
     *
     * @stability stable
     */
    static fromServerlessClusterAttributes(scope, id, attrs) {
        return new ImportedServerlessCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @stability stable
     */
    addRotationSingleUser(options = {}) {
        var _b;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     *
     * @stability stable
     */
    addRotationMultiUser(id, options) {
        var _b;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_b = options.excludeCharacters) !== null && _b !== void 0 ? _b : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    renderScalingConfiguration(options) {
        var _b;
        const minCapacity = options.minCapacity;
        const maxCapacity = options.maxCapacity;
        if (minCapacity && maxCapacity && minCapacity > maxCapacity) {
            throw new Error('maximum capacity must be greater than or equal to minimum capacity.');
        }
        const secondsToAutoPause = (_b = options.autoPause) === null || _b === void 0 ? void 0 : _b.toSeconds();
        if (secondsToAutoPause && (secondsToAutoPause < 300 || secondsToAutoPause > 86400)) {
            throw new Error('auto pause time must be between 5 minutes and 1 day.');
        }
        return {
            autoPause: (secondsToAutoPause === 0) ? false : true,
            minCapacity: options.minCapacity,
            maxCapacity: options.maxCapacity,
            secondsUntilAutoPause: (secondsToAutoPause === 0) ? undefined : secondsToAutoPause,
        };
    }
}
exports.ServerlessCluster = ServerlessCluster;
_a = JSII_RTTI_SYMBOL_1;
ServerlessCluster[_a] = { fqn: "@aws-cdk/aws-rds.ServerlessCluster", version: "1.125.0" };
/**
 * Represents an imported database cluster.
 */
class ImportedServerlessCluster extends ServerlessClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.enableDataApi = true;
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.secret = attrs.secret;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
}
//# sourceMappingURL=data:application/json;base64,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