"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 * @stability stable
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter.
     *
     * @stability stable
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
_a = JSII_RTTI_SYMBOL_1;
SessionPinningFilter[_a] = { fqn: "@aws-cdk/aws-rds.SessionPinningFilter", version: "1.125.0" };
/**
 * You can opt out of session pinning for the following kinds of application statements:.
 *
 * - Setting session variables and configuration settings.
 *
 * @stability stable
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster.
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 *
 * @stability stable
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance.
     *
     * @param instance RDS database instance.
     * @stability stable
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster.
     *
     * @param cluster RDS database cluster.
     * @stability stable
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     *
     * @stability stable
     */
    bind(proxy) {
        var _d, _e, _f, _g, _h, _j;
        const engine = (_e = (_d = this.dbInstance) === null || _d === void 0 ? void 0 : _d.engine) !== null && _e !== void 0 ? _e : (_f = this.dbCluster) === null || _f === void 0 ? void 0 : _f.engine;
        if (!engine) {
            const errorResource = (_g = this.dbCluster) !== null && _g !== void 0 ? _g : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        // allow connecting to the Cluster/Instance from the Proxy
        (_h = this.dbCluster) === null || _h === void 0 ? void 0 : _h.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Cluster from the Proxy');
        (_j = this.dbInstance) === null || _j === void 0 ? void 0 : _j.connections.allowDefaultPortFrom(proxy, 'Allow connections to the database Instance from the Proxy');
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
_b = JSII_RTTI_SYMBOL_1;
ProxyTarget[_b] = { fqn: "@aws-cdk/aws-rds.ProxyTarget", version: "1.125.0" };
/**
 * Represents an RDS Database Proxy.
 *
 */
class DatabaseProxyBase extends cdk.Resource {
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            throw new Error('For imported Database Proxies, the dbUser is required in grantConnect()');
        }
        const scopeStack = cdk.Stack.of(this);
        const proxyGeneratedId = scopeStack.parseArn(this.dbProxyArn, ':').resourceName;
        const userArn = scopeStack.formatArn({
            service: 'rds-db',
            resource: 'dbuser',
            resourceName: `${proxyGeneratedId}/${dbUser}`,
            sep: ':',
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [userArn],
        });
    }
}
/**
 * RDS Database Proxy.
 *
 * @stability stable
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends DatabaseProxyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        const securityGroups = (_d = props.securityGroups) !== null && _d !== void 0 ? _d : [
            new ec2.SecurityGroup(this, 'ProxySecurityGroup', {
                description: 'SecurityGroup for Database Proxy',
                vpc: props.vpc,
            }),
        ];
        this.connections = new ec2.Connections({ securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.secrets = props.secrets;
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_e = props.idleClientTimeout) === null || _e === void 0 ? void 0 : _e.toSeconds(),
            requireTls: (_f = props.requireTLS) !== null && _f !== void 0 ? _f : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(_ => _.securityGroupId) }),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        const proxyTargetGroup = new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
        (_g = bindResult.dbClusters) === null || _g === void 0 ? void 0 : _g.forEach((c) => proxyTargetGroup.node.addDependency(c));
    }
    /**
     * Import an existing database proxy.
     *
     * @stability stable
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends DatabaseProxyBase {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
    /**
     * Grant the given identity connection access to the proxy.
     *
     * @stability stable
     */
    grantConnect(grantee, dbUser) {
        if (!dbUser) {
            if (this.secrets.length > 1) {
                throw new Error('When the Proxy contains multiple Secrets, you must pass a dbUser explicitly to grantConnect()');
            }
            // 'username' is the field RDS uses here,
            // see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/rds-proxy.html#rds-proxy-secrets-arns
            dbUser = this.secrets[0].secretValueFromJson('username').toString();
        }
        return super.grantConnect(grantee, dbUser);
    }
}
exports.DatabaseProxy = DatabaseProxy;
_c = JSII_RTTI_SYMBOL_1;
DatabaseProxy[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseProxy", version: "1.125.0" };
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _d, _e;
    return {
        connectionBorrowTimeout: (_d = props.borrowTimeout) === null || _d === void 0 ? void 0 : _d.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_e = props.sessionPinningFilters) === null || _e === void 0 ? void 0 : _e.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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