"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has change, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject", version: "0.12.23" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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