"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const targets = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const rds = require("../lib");
module.exports = {
    'create a DB instance'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            licenseModel: rds.LicenseModel.BRING_YOUR_OWN_LICENSE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MEDIUM),
            multiAz: true,
            storageType: rds.StorageType.IO1,
            masterUsername: 'syscdk',
            vpc,
            databaseName: 'ORCL',
            storageEncrypted: true,
            backupRetention: cdk.Duration.days(7),
            monitoringInterval: cdk.Duration.minutes(1),
            enablePerformanceInsights: true,
            cloudwatchLogsExports: [
                'trace',
                'audit',
                'alert',
                'listener'
            ],
            cloudwatchLogsRetention: logs.RetentionDays.ONE_MONTH,
            autoMinorVersionUpgrade: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            Properties: {
                DBInstanceClass: 'db.t2.medium',
                AllocatedStorage: '100',
                AutoMinorVersionUpgrade: false,
                BackupRetentionPeriod: 7,
                CopyTagsToSnapshot: true,
                DBName: 'ORCL',
                DBSubnetGroupName: {
                    Ref: 'InstanceSubnetGroupF2CBA54F'
                },
                DeletionProtection: true,
                EnableCloudwatchLogsExports: [
                    'trace',
                    'audit',
                    'alert',
                    'listener'
                ],
                EnablePerformanceInsights: true,
                Engine: 'oracle-se1',
                Iops: 1000,
                LicenseModel: 'bring-your-own-license',
                MasterUsername: {
                    'Fn::Join': [
                        '',
                        [
                            '{{resolve:secretsmanager:',
                            {
                                Ref: 'InstanceSecret478E0A47'
                            },
                            ':SecretString:username::}}'
                        ]
                    ]
                },
                MasterUserPassword: {
                    'Fn::Join': [
                        '',
                        [
                            '{{resolve:secretsmanager:',
                            {
                                Ref: 'InstanceSecret478E0A47'
                            },
                            ':SecretString:password::}}'
                        ]
                    ]
                },
                MonitoringInterval: 60,
                MonitoringRoleArn: {
                    'Fn::GetAtt': [
                        'InstanceMonitoringRole3E2B4286',
                        'Arn'
                    ]
                },
                MultiAZ: true,
                PerformanceInsightsRetentionPeriod: 7,
                StorageEncrypted: true,
                StorageType: 'io1',
                VPCSecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'InstanceSecurityGroupB4E5FA83',
                            'GroupId'
                        ]
                    }
                ]
            },
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBSubnetGroup', {
            DBSubnetGroupDescription: 'Subnet group for Instance database',
            SubnetIds: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0'
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A'
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Security group for Instance database',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'monitoring.rds.amazonaws.com'
                        }
                    }
                ],
                Version: '2012-10-17'
            },
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition'
                            },
                            ':iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole'
                        ]
                    ]
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret', {
            Description: {
                'Fn::Join': [
                    '',
                    [
                        'Generated by the CDK for stack: ',
                        {
                            Ref: 'AWS::StackName'
                        }
                    ]
                ]
            },
            GenerateSecretString: {
                ExcludeCharacters: '\"@/\\',
                GenerateStringKey: 'password',
                PasswordLength: 30,
                SecretStringTemplate: '{"username":"syscdk"}'
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::SecretTargetAttachment', {
            SecretId: {
                Ref: 'InstanceSecret478E0A47'
            },
            TargetId: {
                Ref: 'InstanceC1063A87'
            },
            TargetType: 'AWS::RDS::DBInstance'
        }));
        assert_1.expect(stack).to(assert_1.countResources('Custom::LogRetention', 4));
        test.done();
    },
    'instance with option and parameter group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const optionGroup = new rds.OptionGroup(stack, 'OptionGroup', {
            engine: rds.DatabaseInstanceEngine.ORACLE_SE1,
            majorEngineVersion: '11.2',
            configurations: [
                {
                    name: 'XMLDB'
                }
            ]
        });
        const parameterGroup = new rds.ParameterGroup(stack, 'ParameterGroup', {
            family: 'hello',
            description: 'desc',
            parameters: {
                key: 'value'
            }
        });
        // WHEN
        new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            masterUserPassword: cdk.SecretValue.plainText('tooshort'),
            vpc,
            optionGroup,
            parameterGroup
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBParameterGroupName: {
                Ref: 'ParameterGroup5E32DECB'
            },
            OptionGroupName: {
                Ref: 'OptionGroupACA43DC1'
            }
        }));
        test.done();
    },
    'create an instance from snapshot'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.POSTGRES,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBSnapshotIdentifier: 'my-snapshot'
        }));
        test.done();
    },
    'throws when trying to generate a new password from snapshot without username'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            generateMasterUserPassword: true,
        }), '`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
        test.done();
    },
    'throws when specifying user name without asking to generate a new password'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            masterUsername: 'superadmin'
        }), 'Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
        test.done();
    },
    'throws when password and generate password ar both specified'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // THEN
        test.throws(() => new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            masterUserPassword: cdk.SecretValue.plainText('supersecret'),
            generateMasterUserPassword: true,
        }), 'Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
        test.done();
    },
    'create a read replica in the same region - with the subnet group name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const sourceInstance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // WHEN
        new rds.DatabaseInstanceReadReplica(stack, 'ReadReplica', {
            sourceDatabaseInstance: sourceInstance,
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            SourceDBInstanceIdentifier: {
                'Fn::Join': ['', [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':rds:',
                        { Ref: 'AWS::Region' },
                        ':',
                        { Ref: 'AWS::AccountId' },
                        ':db:',
                        { Ref: 'InstanceC1063A87' },
                    ]],
            },
            DBSubnetGroupName: {
                Ref: 'ReadReplicaSubnetGroup680C605C',
            },
        }));
        test.done();
    },
    'on event'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('dummy'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X
        });
        // WHEN
        instance.onEvent('InstanceEvent', { target: new targets.LambdaFunction(fn) });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventPattern: {
                source: [
                    'aws.rds'
                ],
                resources: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                ':rds:',
                                {
                                    Ref: 'AWS::Region'
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':db:',
                                {
                                    Ref: 'InstanceC1063A87'
                                }
                            ]
                        ]
                    }
                ]
            },
            Targets: [
                {
                    Arn: {
                        'Fn::GetAtt': [
                            'Function76856677',
                            'Arn'
                        ],
                    },
                    Id: 'Target0'
                }
            ]
        }));
        test.done();
    },
    'on event without target'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // WHEN
        instance.onEvent('InstanceEvent');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventPattern: {
                source: [
                    'aws.rds'
                ],
                resources: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition'
                                },
                                ':rds:',
                                {
                                    Ref: 'AWS::Region'
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId'
                                },
                                ':db:',
                                {
                                    Ref: 'InstanceC1063A87'
                                }
                            ]
                        ]
                    }
                ]
            }
        }));
        test.done();
    },
    'can use metricCPUUtilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        // THEN
        test.deepEqual(stack.resolve(instance.metricCPUUtilization()), {
            dimensions: { DBInstanceIdentifier: { Ref: 'InstanceC1063A87' } },
            namespace: 'AWS/RDS',
            metricName: 'CPUUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    },
    'can resolve endpoint port and socket address'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc
        });
        test.deepEqual(stack.resolve(instance.instanceEndpoint.port), {
            'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Port']
        });
        test.deepEqual(stack.resolve(instance.instanceEndpoint.socketAddress), {
            'Fn::Join': [
                '',
                [
                    { 'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Address'] },
                    ':',
                    { 'Fn::GetAtt': ['InstanceC1063A87', 'Endpoint.Port'] },
                ]
            ]
        });
        test.done();
    },
    'can deactivate backup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            backupRetention: cdk.Duration.seconds(0),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            BackupRetentionPeriod: 0
        }));
        test.done();
    },
    'imported instance with imported security group with allowAllOutbound set to false'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const instance = rds.DatabaseInstance.fromDatabaseInstanceAttributes(stack, 'Database', {
            instanceEndpointAddress: 'address',
            instanceIdentifier: 'identifier',
            port: 3306,
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false
                })],
        });
        // WHEN
        instance.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-123456789',
        }));
        test.done();
    },
    'create an instance with imported monitoring role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const monitoringRole = new aws_iam_1.Role(stack, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')
            ]
        });
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            monitoringInterval: cdk.Duration.minutes(1),
            monitoringRole
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            MonitoringInterval: 60,
            MonitoringRoleArn: {
                'Fn::GetAtt': ['MonitoringRole90457BF9', 'Arn']
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'create an instance with an existing security group'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const securityGroup = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false
        });
        // WHEN
        const instance = new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            securityGroups: [securityGroup],
        });
        instance.connections.allowDefaultPortFromAnyIpv4();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            VPCSecurityGroups: ['sg-123456789']
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            FromPort: {
                'Fn::GetAtt': [
                    'InstanceC1063A87',
                    'Endpoint.Port'
                ]
            },
            GroupId: 'sg-123456789',
            ToPort: {
                'Fn::GetAtt': [
                    'InstanceC1063A87',
                    'Endpoint.Port'
                ]
            }
        }));
        test.done();
    },
    'throws when trying to add rotation to an instance without secret'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            masterUserPassword: cdk.SecretValue.plainText('tooshort'),
            vpc
        });
        // THEN
        test.throws(() => instance.addRotationSingleUser(), /without secret/);
        test.done();
    },
    'throws when trying to add single user rotation multiple times'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const instance = new rds.DatabaseInstance(stack, 'Database', {
            engine: rds.DatabaseInstanceEngine.SQL_SERVER_EE,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'syscdk',
            vpc
        });
        // WHEN
        instance.addRotationSingleUser();
        // THEN
        test.throws(() => instance.addRotationSingleUser(), /A single user rotation was already added to this instance/);
        test.done();
    },
    'throws when timezone is set for non-sqlserver database engine'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'vpc');
        const tzSupportedEngines = [rds.DatabaseInstanceEngine.SQL_SERVER_EE, rds.DatabaseInstanceEngine.SQL_SERVER_EX,
            rds.DatabaseInstanceEngine.SQL_SERVER_SE, rds.DatabaseInstanceEngine.SQL_SERVER_WEB];
        const tzUnsupportedEngines = [rds.DatabaseInstanceEngine.MYSQL, rds.DatabaseInstanceEngine.POSTGRES,
            rds.DatabaseInstanceEngine.ORACLE_EE, rds.DatabaseInstanceEngine.MARIADB];
        // THEN
        tzSupportedEngines.forEach((engine) => {
            test.ok(new rds.DatabaseInstance(stack, `${engine.name}-db`, {
                engine,
                instanceClass: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.SMALL),
                masterUsername: 'master',
                timezone: 'Europe/Zurich',
                vpc,
            }));
        });
        tzUnsupportedEngines.forEach((engine) => {
            test.throws(() => new rds.DatabaseInstance(stack, `${engine.name}-db`, {
                engine,
                instanceClass: ec2.InstanceType.of(ec2.InstanceClass.C5, ec2.InstanceSize.SMALL),
                masterUsername: 'master',
                timezone: 'Europe/Zurich',
                vpc,
            }), /timezone property can be configured only for Microsoft SQL Server/);
        });
        test.done();
    },
    'create an instance from snapshot with maximum allocated storage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstanceFromSnapshot(stack, 'Instance', {
            snapshotIdentifier: 'my-snapshot',
            engine: rds.DatabaseInstanceEngine.POSTGRES,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.LARGE),
            vpc,
            maxAllocatedStorage: 200
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            DBSnapshotIdentifier: 'my-snapshot',
            MaxAllocatedStorage: 200
        }));
        test.done();
    },
    'create a DB instance with maximum allocated storage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new rds.DatabaseInstance(stack, 'Instance', {
            engine: rds.DatabaseInstanceEngine.MYSQL,
            instanceClass: ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.SMALL),
            masterUsername: 'admin',
            vpc,
            backupRetention: cdk.Duration.seconds(0),
            maxAllocatedStorage: 250
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::RDS::DBInstance', {
            BackupRetentionPeriod: 0,
            MaxAllocatedStorage: 250
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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