"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const version_1 = require("./private/version");
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    // tslint:disable-next-line max-line-length
    constructor(name, singleUserRotationApplication, multiUserRotationApplication, parameterGroupFamilies) {
        this.name = name;
        this.singleUserRotationApplication = singleUserRotationApplication;
        this.multiUserRotationApplication = multiUserRotationApplication;
        this.parameterGroupFamilies = parameterGroupFamilies;
    }
    /**
     * Get the latest parameter group family for this engine. Latest is determined using semver on the engine major version.
     * When `engineVersion` is specified, return the parameter group family corresponding to that engine version.
     * Return undefined if no parameter group family is defined for this engine or for the requested `engineVersion`.
     */
    parameterGroupFamily(engineVersion) {
        if (this.parameterGroupFamilies === undefined) {
            return undefined;
        }
        if (engineVersion) {
            const family = this.parameterGroupFamilies.find(x => engineVersion.startsWith(x.engineMajorVersion));
            if (family) {
                return family.parameterGroupFamily;
            }
        }
        else if (this.parameterGroupFamilies.length > 0) {
            const sorted = this.parameterGroupFamilies.slice().sort((a, b) => {
                return version_1.compare(a.engineMajorVersion, b.engineMajorVersion);
            }).reverse();
            return sorted[0].parameterGroupFamily;
        }
        return undefined;
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/* tslint:disable max-line-length */
DatabaseClusterEngine.AURORA = new DatabaseClusterEngine('aurora', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'aurora5.6' }
]);
DatabaseClusterEngine.AURORA_MYSQL = new DatabaseClusterEngine('aurora-mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.7', parameterGroupFamily: 'aurora-mysql5.7' }
]);
DatabaseClusterEngine.AURORA_POSTGRESQL = new DatabaseClusterEngine('aurora-postgresql', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.6', parameterGroupFamily: 'aurora-postgresql9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'aurora-postgresql10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'aurora-postgresql11' }
]);
//# sourceMappingURL=data:application/json;base64,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