"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * A new cluster or instance parameter group
 */
class ParameterGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Parameters of the parameter group
         */
        this.parameters = {};
    }
    /**
     * Imports a parameter group
     */
    static fromParameterGroupName(scope, id, parameterGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.parameterGroupName = parameterGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a parameter to this parameter group
     *
     * @param key The key of the parameter to be added
     * @param value The value of the parameter to be added
     */
    addParameter(key, value) {
        if (!this.parameters) {
            this.parameters = {};
        }
        this.parameters[key] = value;
    }
}
/**
 * A parameter group
 *
 * @resource AWS::RDS::DBParameterGroup
 */
class ParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBParameterGroup(this, 'Resource', {
            description: props.description || `Parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters })
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ParameterGroup = ParameterGroup;
/**
 * A cluster parameter group
 *
 * @resource AWS::RDS::DBClusterParameterGroup
 */
class ClusterParameterGroup extends ParameterGroupBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.parameters = props.parameters ? props.parameters : {};
        const resource = new rds_generated_1.CfnDBClusterParameterGroup(this, 'Resource', {
            description: props.description || `Cluster parameter group for ${props.family}`,
            family: props.family,
            parameters: core_1.Lazy.anyValue({ produce: () => this.parameters }),
        });
        this.parameterGroupName = resource.ref;
    }
}
exports.ClusterParameterGroup = ClusterParameterGroup;
//# sourceMappingURL=data:application/json;base64,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