"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.name,
            majorEngineVersion: props.majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.name} ${props.majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations)
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port)
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId]
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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