import * as ec2 from '@aws-cdk/aws-ec2';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Construct, Duration, IResource, RemovalPolicy, Resource, SecretValue } from '@aws-cdk/core';
import { Endpoint } from './endpoint';
import { IOptionGroup } from './option-group';
import { IParameterGroup } from './parameter-group';
import { DatabaseClusterEngine, RotationMultiUserOptions } from './props';
import { CfnDBInstanceProps } from './rds.generated';
/**
 * A database instance
 */
export interface IDatabaseInstance extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * The instance identifier.
     */
    readonly instanceIdentifier: string;
    /**
     * The instance arn.
     */
    readonly instanceArn: string;
    /**
     * The instance endpoint address.
     *
     * @attribute EndpointAddress
     */
    readonly dbInstanceEndpointAddress: string;
    /**
     * The instance endpoint port.
     *
     * @attribute EndpointPort
     */
    readonly dbInstanceEndpointPort: string;
    /**
     * The instance endpoint.
     */
    readonly instanceEndpoint: Endpoint;
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
}
/**
 * Properties that describe an existing instance
 */
export interface DatabaseInstanceAttributes {
    /**
     * The instance identifier.
     */
    readonly instanceIdentifier: string;
    /**
     * The endpoint address.
     */
    readonly instanceEndpointAddress: string;
    /**
     * The database port.
     */
    readonly port: number;
    /**
     * The security groups of the instance.
     */
    readonly securityGroups: ec2.ISecurityGroup[];
}
/**
 * A new or imported database instance.
 */
export declare abstract class DatabaseInstanceBase extends Resource implements IDatabaseInstance {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope: Construct, id: string, attrs: DatabaseInstanceAttributes): IDatabaseInstance;
    abstract readonly instanceIdentifier: string;
    abstract readonly dbInstanceEndpointAddress: string;
    abstract readonly dbInstanceEndpointPort: string;
    abstract readonly instanceEndpoint: Endpoint;
    /**
     * Access to network connections.
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * The instance arn.
     */
    get instanceArn(): string;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
export declare class DatabaseInstanceEngine extends DatabaseClusterEngine {
    static readonly MARIADB: DatabaseInstanceEngine;
    static readonly MYSQL: DatabaseInstanceEngine;
    static readonly ORACLE_EE: DatabaseInstanceEngine;
    static readonly ORACLE_SE2: DatabaseInstanceEngine;
    static readonly ORACLE_SE1: DatabaseInstanceEngine;
    static readonly ORACLE_SE: DatabaseInstanceEngine;
    static readonly POSTGRES: DatabaseInstanceEngine;
    static readonly SQL_SERVER_EE: DatabaseInstanceEngine;
    static readonly SQL_SERVER_SE: DatabaseInstanceEngine;
    static readonly SQL_SERVER_EX: DatabaseInstanceEngine;
    static readonly SQL_SERVER_WEB: DatabaseInstanceEngine;
    /** To make it a compile-time error to pass a DatabaseClusterEngine where a DatabaseInstanceEngine is expected. */
    readonly isDatabaseInstanceEngine = true;
}
/**
 * The license model.
 */
export declare enum LicenseModel {
    /**
     * License included.
     */
    LICENSE_INCLUDED = "license-included",
    /**
     * Bring your own licencse.
     */
    BRING_YOUR_OWN_LICENSE = "bring-your-own-license",
    /**
     * General public license.
     */
    GENERAL_PUBLIC_LICENSE = "general-public-license"
}
/**
 * The processor features.
 */
export interface ProcessorFeatures {
    /**
     * The number of CPU core.
     *
     * @default - the default number of CPU cores for the chosen instance class.
     */
    readonly coreCount?: number;
    /**
     * The number of threads per core.
     *
     * @default - the default number of threads per core for the chosen instance class.
     */
    readonly threadsPerCore?: number;
}
/**
 * The type of storage.
 */
export declare enum StorageType {
    /**
     * Standard.
     */
    STANDARD = "standard",
    /**
     * General purpose (SSD).
     */
    GP2 = "gp2",
    /**
     * Provisioned IOPS (SSD).
     */
    IO1 = "io1"
}
/**
 * The retention period for Performance Insight.
 */
export declare enum PerformanceInsightRetention {
    /**
     * Default retention period of 7 days.
     */
    DEFAULT = 7,
    /**
     * Long term retention period of 2 years.
     */
    LONG_TERM = 731
}
/**
 * Construction properties for a DatabaseInstanceNew
 */
export interface DatabaseInstanceNewProps {
    /**
     * The name of the compute and memory capacity classes.
     */
    readonly instanceClass: ec2.InstanceType;
    /**
     * Specifies if the database instance is a multiple Availability Zone deployment.
     *
     * @default false
     */
    readonly multiAz?: boolean;
    /**
     * The name of the Availability Zone where the DB instance will be located.
     *
     * @default - no preference
     */
    readonly availabilityZone?: string;
    /**
     * The storage type.
     *
     * @default GP2
     */
    readonly storageType?: StorageType;
    /**
     * The number of I/O operations per second (IOPS) that the database provisions.
     * The value must be equal to or greater than 1000.
     *
     * @default - no provisioned iops
     */
    readonly iops?: number;
    /**
     * The number of CPU cores and the number of threads per core.
     *
     * @default - the default number of CPU cores and threads per core for the
     * chosen instance class.
     *
     * See https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html#USER_ConfigureProcessor
     */
    readonly processorFeatures?: ProcessorFeatures;
    /**
     * A name for the DB instance. If you specify a name, AWS CloudFormation
     * converts it to lowercase.
     *
     * @default - a CloudFormation generated name
     */
    readonly instanceIdentifier?: string;
    /**
     * The VPC network where the DB subnet group should be created.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The type of subnets to add to the created DB subnet group.
     *
     * @default - private subnets
     */
    readonly vpcPlacement?: ec2.SubnetSelection;
    /**
     * The security groups to assign to the DB instance.
     *
     * @default - a new security group is created
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The port for the instance.
     *
     * @default - the default port for the chosen engine.
     */
    readonly port?: number;
    /**
     * The option group to associate with the instance.
     *
     * @default - no option group
     */
    readonly optionGroup?: IOptionGroup;
    /**
     * Whether to enable mapping of AWS Identity and Access Management (IAM) accounts
     * to database accounts.
     *
     * @default false
     */
    readonly iamAuthentication?: boolean;
    /**
     * The number of days during which automatic DB snapshots are retained. Set
     * to zero to disable backups.
     *
     * @default Duration.days(1)
     */
    readonly backupRetention?: Duration;
    /**
     * The daily time range during which automated backups are performed.
     *
     * Constraints:
     * - Must be in the format `hh24:mi-hh24:mi`.
     * - Must be in Universal Coordinated Time (UTC).
     * - Must not conflict with the preferred maintenance window.
     * - Must be at least 30 minutes.
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region. To see the time blocks available, see
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     */
    readonly preferredBackupWindow?: string;
    /**
     * Indicates whether to copy all of the user-defined tags from the
     * DB instance to snapshots of the DB instance.
     *
     * @default true
     */
    readonly copyTagsToSnapshot?: boolean;
    /**
     * Indicates whether automated backups should be deleted or retained when
     * you delete a DB instance.
     *
     * @default false
     */
    readonly deleteAutomatedBackups?: boolean;
    /**
     * The interval, in seconds, between points when Amazon RDS collects enhanced
     * monitoring metrics for the DB instance.
     *
     * @default - no enhanced monitoring
     */
    readonly monitoringInterval?: Duration;
    /**
     * Role that will be used to manage DB instance monitoring.
     *
     * @default - A role is automatically created for you
     */
    readonly monitoringRole?: iam.IRole;
    /**
     * Whether to enable Performance Insights for the DB instance.
     *
     * @default false
     */
    readonly enablePerformanceInsights?: boolean;
    /**
     * The amount of time, in days, to retain Performance Insights data.
     *
     * @default 7
     */
    readonly performanceInsightRetention?: PerformanceInsightRetention;
    /**
     * The AWS KMS key for encryption of Performance Insights data.
     *
     * @default - default master key
     */
    readonly performanceInsightKmsKey?: kms.IKey;
    /**
     * The list of log types that need to be enabled for exporting to
     * CloudWatch Logs.
     *
     * @default - no log exports
     */
    readonly cloudwatchLogsExports?: string[];
    /**
     * The number of days log events are kept in CloudWatch Logs. When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `Infinity`.
     *
     * @default - logs never expire
     */
    readonly cloudwatchLogsRetention?: logs.RetentionDays;
    /**
     * The IAM role for the Lambda function associated with the custom resource
     * that sets the retention policy.
     *
     * @default - a new role is created.
     */
    readonly cloudwatchLogsRetentionRole?: iam.IRole;
    /**
     * Indicates that minor engine upgrades are applied automatically to the
     * DB instance during the maintenance window.
     *
     * @default true
     */
    readonly autoMinorVersionUpgrade?: boolean;
    /**
     * The weekly time range (in UTC) during which system maintenance can occur.
     *
     * Format: `ddd:hh24:mi-ddd:hh24:mi`
     * Constraint: Minimum 30-minute window
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region, occurring on a random day of the week. To see
     * the time blocks available, see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * Indicates whether the DB instance should have deletion protection enabled.
     *
     * @default true
     */
    readonly deletionProtection?: boolean;
    /**
     * The CloudFormation policy to apply when the instance is removed from the
     * stack or replaced during an update.
     *
     * @default RemovalPolicy.Retain
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Upper limit to which RDS can scale the storage in GiB(Gibibyte).
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling
     * @default - No autoscaling of RDS instance
     */
    readonly maxAllocatedStorage?: number;
}
/**
 * A new database instance.
 */
declare abstract class DatabaseInstanceNew extends DatabaseInstanceBase implements IDatabaseInstance {
    /**
     * The VPC where this database instance is deployed.
     */
    readonly vpc: ec2.IVpc;
    readonly connections: ec2.Connections;
    protected readonly vpcPlacement?: ec2.SubnetSelection;
    protected readonly newCfnProps: CfnDBInstanceProps;
    private readonly cloudwatchLogsExports?;
    private readonly cloudwatchLogsRetention?;
    private readonly cloudwatchLogsRetentionRole?;
    constructor(scope: Construct, id: string, props: DatabaseInstanceNewProps);
    protected setLogRetention(): void;
}
/**
 * Construction properties for a DatabaseInstanceSource
 */
export interface DatabaseInstanceSourceProps extends DatabaseInstanceNewProps {
    /**
     * The database engine.
     */
    readonly engine: DatabaseInstanceEngine;
    /**
     * The license model.
     *
     * @default - RDS default license model
     */
    readonly licenseModel?: LicenseModel;
    /**
     * The engine version. To prevent automatic upgrades, be sure to specify the
     * full version number.
     *
     * @default - RDS default engine version
     */
    readonly engineVersion?: string;
    /**
     * Whether to allow major version upgrades.
     *
     * @default false
     */
    readonly allowMajorVersionUpgrade?: boolean;
    /**
     * The time zone of the instance. This is currently supported only by Microsoft Sql Server.
     *
     * @default - RDS default timezone
     */
    readonly timezone?: string;
    /**
     * The allocated storage size, specified in gigabytes (GB).
     *
     * @default 100
     */
    readonly allocatedStorage?: number;
    /**
     * The master user password.
     *
     * @default - a Secrets Manager generated password
     */
    readonly masterUserPassword?: SecretValue;
    /**
     * The KMS key to use to encrypt the secret for the master user password.
     *
     * @default - default master key
     */
    readonly secretKmsKey?: kms.IKey;
    /**
     * The name of the database.
     *
     * @default - no name
     */
    readonly databaseName?: string;
    /**
     * The DB parameter group to associate with the instance.
     *
     * @default - no parameter group
     */
    readonly parameterGroup?: IParameterGroup;
}
/**
 * A new source database instance (not a read replica)
 */
declare abstract class DatabaseInstanceSource extends DatabaseInstanceNew implements IDatabaseInstance {
    /**
     * The AWS Secrets Manager secret attached to the instance.
     */
    abstract readonly secret?: secretsmanager.ISecret;
    protected readonly sourceCfnProps: CfnDBInstanceProps;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    constructor(scope: Construct, id: string, props: DatabaseInstanceSourceProps);
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter?: Duration): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
/**
 * Construction properties for a DatabaseInstance.
 */
export interface DatabaseInstanceProps extends DatabaseInstanceSourceProps {
    /**
     * The master user name.
     */
    readonly masterUsername: string;
    /**
     * For supported engines, specifies the character set to associate with the
     * DB instance.
     *
     * @default - RDS default character set name
     */
    readonly characterSetName?: string;
    /**
     * Indicates whether the DB instance is encrypted.
     *
     * @default false
     */
    readonly storageEncrypted?: boolean;
    /**
     * The master key that's used to encrypt the DB instance.
     *
     * @default - default master key
     */
    readonly kmsKey?: kms.IKey;
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstance extends DatabaseInstanceSource implements IDatabaseInstance {
    readonly instanceIdentifier: string;
    readonly dbInstanceEndpointAddress: string;
    readonly dbInstanceEndpointPort: string;
    readonly instanceEndpoint: Endpoint;
    readonly secret?: secretsmanager.ISecret;
    constructor(scope: Construct, id: string, props: DatabaseInstanceProps);
}
/**
 * Construction properties for a DatabaseInstanceFromSnapshot.
 */
export interface DatabaseInstanceFromSnapshotProps extends DatabaseInstanceSourceProps {
    /**
     * The name or Amazon Resource Name (ARN) of the DB snapshot that's used to
     * restore the DB instance. If you're restoring from a shared manual DB
     * snapshot, you must specify the ARN of the snapshot.
     */
    readonly snapshotIdentifier: string;
    /**
     * The master user name.
     *
     * Specify this prop with the **current** master user name of the snapshot
     * only when generating a new master user password with `generateMasterUserPassword`.
     * The value will be set in the generated secret attached to the instance.
     *
     * It is not possible to change the master user name of a RDS instance.
     *
     * @default - inherited from the snapshot
     */
    readonly masterUsername?: string;
    /**
     * Whether to generate a new master user password and store it in
     * Secrets Manager. `masterUsername` must be specified with the **current**
     * master user name of the snapshot when this property is set to true.
     *
     * @default false
     */
    readonly generateMasterUserPassword?: boolean;
}
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource implements IDatabaseInstance {
    readonly instanceIdentifier: string;
    readonly dbInstanceEndpointAddress: string;
    readonly dbInstanceEndpointPort: string;
    readonly instanceEndpoint: Endpoint;
    readonly secret?: secretsmanager.ISecret;
    constructor(scope: Construct, id: string, props: DatabaseInstanceFromSnapshotProps);
}
/**
 * Construction properties for a DatabaseInstanceReadReplica.
 */
export interface DatabaseInstanceReadReplicaProps extends DatabaseInstanceSourceProps {
    /**
     * The source database instance.
     *
     * Each DB instance can have a limited number of read replicas. For more
     * information, see https://docs.aws.amazon.com/AmazonRDS/latest/DeveloperGuide/USER_ReadRepl.html.
     *
     */
    readonly sourceDatabaseInstance: IDatabaseInstance;
    /**
     * Indicates whether the DB instance is encrypted.
     *
     * @default false
     */
    readonly storageEncrypted?: boolean;
    /**
     * The master key that's used to encrypt the DB instance.
     *
     * @default - default master key
     */
    readonly kmsKey?: kms.IKey;
}
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseInstanceReadReplica extends DatabaseInstanceNew implements IDatabaseInstance {
    readonly instanceIdentifier: string;
    readonly dbInstanceEndpointAddress: string;
    readonly dbInstanceEndpointPort: string;
    readonly instanceEndpoint: Endpoint;
    constructor(scope: Construct, id: string, props: DatabaseInstanceReadReplicaProps);
}
export {};
