"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        const securityGroup = props.instanceProps.securityGroup !== undefined ?
            props.instanceProps.securityGroup : new ec2.SecurityGroup(this, 'SecurityGroup', {
            description: 'RDS security group',
            vpc: props.instanceProps.vpc
        });
        this.securityGroupId = securityGroup.securityGroupId;
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey
            });
        }
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let s3ImportRole = props.s3ImportRole;
        if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
            if (props.s3ImportRole) {
                throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
            }
            s3ImportRole = new aws_iam_1.Role(this, 'S3ImportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com')
            });
            for (const bucket of props.s3ImportBuckets) {
                bucket.grantRead(s3ImportRole);
            }
        }
        let s3ExportRole = props.s3ExportRole;
        if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
            if (props.s3ExportRole) {
                throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
            }
            s3ExportRole = new aws_iam_1.Role(this, 'S3ExportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ExportBuckets) {
                bucket.grantReadWrite(s3ExportRole);
            }
        }
        let clusterParameterGroup = props.parameterGroup;
        const clusterAssociatedRoles = [];
        if (s3ImportRole || s3ExportRole) {
            if (s3ImportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn });
            }
            if (s3ExportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn });
            }
            // MySQL requires the associated roles to be specified as cluster parameters as well, PostgreSQL does not
            if (props.engine === props_1.DatabaseClusterEngine.AURORA || props.engine === props_1.DatabaseClusterEngine.AURORA_MYSQL) {
                if (!clusterParameterGroup) {
                    const parameterGroupFamily = props.engine.parameterGroupFamily(props.engineVersion);
                    if (!parameterGroupFamily) {
                        throw new Error(`No parameter group family found for database engine ${props.engine.name} with version ${props.engineVersion}.` +
                            'Failed to set the correct cluster parameters for s3 import and export roles.');
                    }
                    clusterParameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ClusterParameterGroup', {
                        family: parameterGroupFamily,
                    });
                }
                if (clusterParameterGroup instanceof parameter_group_1.ClusterParameterGroup) { // ignore imported ClusterParameterGroup
                    if (s3ImportRole) {
                        clusterParameterGroup.addParameter('aurora_load_from_s3_role', s3ImportRole.roleArn);
                    }
                    if (s3ExportRole) {
                        clusterParameterGroup.addParameter('aurora_select_into_s3_role', s3ExportRole.roleArn);
                    }
                }
            }
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: [this.securityGroupId],
            port: props.port,
            dbClusterParameterGroupName: clusterParameterGroup && clusterParameterGroup.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            backupRetentionPeriod: props.backup && props.backup.retention && props.backup.retention.toDays(),
            preferredBackupWindow: props.backup && props.backup.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            // Encryption
            kmsKeyId: props.kmsKey && props.kmsKey.keyArn,
            storageEncrypted: props.kmsKey ? true : props.storageEncrypted
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const instanceCount = props.instances != null ? props.instances : 2;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        // Get the actual subnet objects so we can depend on internet connectivity.
        const internetConnected = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets).internetConnectivityEstablished;
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new aws_iam_1.Role(this, 'MonitoringRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')
                ]
            });
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
                props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                    undefined;
            const publiclyAccessible = props.instanceProps.vpcSubnets && props.instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
            const instance = new rds_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                engine: props.engine.name,
                engineVersion: props.engineVersion,
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceProps.instanceType),
                publiclyAccessible,
                // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
                dbSubnetGroupName: subnetGroup.ref,
                dbParameterGroupName: props.instanceProps.parameterGroup && props.instanceProps.parameterGroup.parameterGroupName,
                monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
                monitoringRoleArn: monitoringRole && monitoringRole.roleArn
            });
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnected);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups: [securityGroup], defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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