"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Deployer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const codedeploy = require("@aws-cdk/aws-codedeploy");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const customresources = require("@aws-cdk/custom-resources");
;
/**
 * Represents a Deployer resource for deploying an artifact to EC2 using CodeDeploy.
 *
 */
class Ec2Deployer extends cdk.Construct {
    constructor(scope, id, props) {
        // Validate that props.deploymentTimeout is less than 2 hours, per maximum value accepted by downstream customresources.Provider.totalTimeout
        if (props.deploymentTimeout && props.deploymentTimeout.toMilliseconds() > Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT.toMilliseconds()) { // have to convert to milliseconds in case the cdk.Duration is passed in milliseconds
            throw new Error(`Invalid prop: deploymentTimeout must be less than ${Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT.toHumanString()}.`);
        }
        // Validate that at least one instanceRole is supplied if we cannot get them from deploymentGroup.autoScalingGroups
        if (!props.deploymentGroup.autoScalingGroups && (!props.instanceRoles || props.instanceRoles.length === 0)) {
            throw new Error('If deploymentGroup is of type IServerDeploymentGroup, you must supply at least one role in instanceRoles.');
        }
        super(scope, id);
        // Set defaults for any missing props
        this.code = props.code.bind(this);
        this.deploymentGroup = props.deploymentGroup;
        this.waitToComplete = props.waitToComplete !== undefined ? props.waitToComplete : true;
        this.deploymentTimeout = this.waitToComplete ? props.deploymentTimeout || cdk.Duration.minutes(5) : undefined; // can only be defined if waitToComplete=true because of downstream customresources.Provider.totalTimeout
        // Create OnEventHandler Lambda function for custom resource
        // Can't use SingletonFunction because permissions are dependent on props passed into each Ec2Deployer instance
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            // const onEvent = new lambda.SingletonFunction(this, 'OnEventHandler', {
            //   uuid: '3a9c56a9-1dd5-42dc-af2f-10b76edde830',
            code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/ec2-deployer')),
            runtime: lambda.Runtime.PYTHON_3_8,
            handler: 'index.on_event',
            initialPolicy: [
                new iam.PolicyStatement({
                    actions: ['codedeploy:GetDeploymentConfig'],
                    resources: [codedeploy.ServerDeploymentConfig.ONE_AT_A_TIME.deploymentConfigArn],
                }),
                new iam.PolicyStatement({
                    actions: ['codedeploy:CreateDeployment'],
                    resources: [this.deploymentGroup.deploymentGroupArn],
                }),
                new iam.PolicyStatement({
                    actions: ['codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
                    resources: [this.deploymentGroup.application.applicationArn],
                }),
            ],
        });
        // Create IsCompleteHandler Lambda function for custom resource, only if waitToComplete=true
        // Can't use SingletonFunction because permissions are dependent on props passed into each Ec2Deployer instance
        let isComplete = undefined;
        if (this.waitToComplete) {
            // isComplete = new lambda.SingletonFunction(this, 'IsCompleteHandler', {
            //   uuid: 'f58e4e2e-8b7e-4bd0-b33b-c5c9f19f5546',
            isComplete = new lambda.Function(this, 'IsCompleteHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/ec2-deployer')),
                runtime: lambda.Runtime.PYTHON_3_8,
                handler: 'index.is_complete',
                initialPolicy: [
                    new iam.PolicyStatement({
                        resources: [this.deploymentGroup.deploymentGroupArn],
                        actions: ['codedeploy:GetDeployment'],
                    }),
                ],
            });
        }
        // Create provider for custom resource
        const deployerProvider = new customresources.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            totalTimeout: this.deploymentTimeout,
            isCompleteHandler: isComplete,
        });
        // Ensure ASGs have read access to code S3 object for deployment
        const policyStatement = new iam.PolicyStatement({
            actions: ['s3:GetObject*'],
            resources: [`arn:${cdk.Stack.of(this).partition}:s3:::${this.code.s3Location.bucketName}/${this.code.s3Location.objectKey}`],
        });
        if (props.instanceRoles) {
            for (let role of props.instanceRoles) {
                role.addToPrincipalPolicy(policyStatement);
            }
        }
        else {
            for (let asg of this.deploymentGroup.autoScalingGroups) {
                asg.role.addToPrincipalPolicy(policyStatement);
            }
        }
        // Create custom resource that triggers a deployment
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: deployerProvider.serviceToken,
            properties: {
                applicationName: this.deploymentGroup.application.applicationName,
                deploymentGroupName: this.deploymentGroup.deploymentGroupName,
                codeS3BucketName: this.code.s3Location.bucketName,
                codeS3ObjectKey: this.code.s3Location.objectKey,
                codeS3ObjectVersion: this.code.s3Location.objectVersion,
            },
        });
    }
}
exports.Ec2Deployer = Ec2Deployer;
_a = JSII_RTTI_SYMBOL_1;
Ec2Deployer[_a] = { fqn: "cdk-deployer.Ec2Deployer", version: "1.0.12" };
/**
 * Maximum allowed value for deploymentTimeout prop.
 *
 */
Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT = cdk.Duration.hours(2);
//# sourceMappingURL=data:application/json;base64,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