"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const fs = require("fs");
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ifw = require(".");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_INTEG_REGION || process.env.CDK_DEFAULT_REGION || 'us-east-1',
            account: process.env.CDK_INTEG_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-stack', { env });
        const databaseName = 'FleetWise';
        const tableName = 'FleetWise';
        const database = new aws_cdk_lib_1.aws_timestream.CfnDatabase(stack, 'Database', {
            databaseName,
        });
        const table = new aws_cdk_lib_1.aws_timestream.CfnTable(stack, 'Table', {
            databaseName,
            tableName,
        });
        table.node.addDependency(database);
        const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'Role', {
            roleName: 'iotfleetwise-role',
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('iotfleetwise.amazonaws.com'),
        });
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:WriteRecords',
                'timestream:Select',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
            actions: [
                'timestream:DescribeEndpoints',
            ],
            resources: ['*'],
        }));
        const canDbc = fs.readFileSync(path.join(__dirname, '/../hscan.dbc'), 'utf8');
        const nodes = [new ifw.SignalCatalogBranch('Vehicle', 'Vehicle')];
        canDbc.split('\n').filter(line => /^\s+SG_\s+\w+/.test(line)).map(line => {
            const signal_name = line.split(/\s+/)[2];
            nodes.push(new ifw.SignalCatalogSensor(signal_name, `Vehicle.${signal_name}`, 'DOUBLE'));
        });
        const signalCatalog = new ifw.SignalCatalog(stack, 'SignalCatalog', {
            database,
            table,
            role,
            description: 'my signal catalog',
            nodes,
        });
        const signalsMap = {};
        canDbc.split('\n').filter(line => /^\s+SG_\s+\w+/.test(line)).map(line => {
            const signal_name = line.split(/\s+/)[2];
            signalsMap[signal_name] = `Vehicle.${signal_name}`;
        });
        const model_a = new ifw.VehicleModel(stack, 'ModelA', {
            signalCatalog,
            name: 'modelA',
            description: 'Model A vehicle',
            networkInterfaces: [new ifw.CanVehicleInterface('1', 'vcan0')],
            networkFileDefinitions: [new ifw.CanDefinition('1', signalsMap, [canDbc])],
        });
        const vin100 = new ifw.Vehicle(stack, 'vin100', {
            vehicleId: 'vin100',
            vehicleModel: model_a,
            createIotThing: true,
        });
        new cdk.CfnOutput(stack, 'certificateId', { value: vin100.certificateId });
        new ifw.Fleet(stack, 'fleet1', {
            fleetId: 'fleet1',
            signalCatalog: signalCatalog,
            description: 'my fleet1',
            vehicles: [vin100],
        });
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZGJjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2ludGVnLmRiYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG1DQUFtQztBQUNuQyw2Q0FHcUI7QUFDckIseUJBQXlCO0FBRXpCLE1BQWEsWUFBWTtJQUV2QjtRQUNFLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRTFCLE1BQU0sR0FBRyxHQUFHO1lBQ1YsTUFBTSxFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsSUFBSSxXQUFXO1lBQ3JGLE9BQU8sRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLGlCQUFpQixJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUJBQW1CO1NBQzFFLENBQUM7UUFFRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFFekQsTUFBTSxZQUFZLEdBQUcsV0FBVyxDQUFDO1FBQ2pDLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQztRQUU5QixNQUFNLFFBQVEsR0FBRyxJQUFJLDRCQUFFLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckQsWUFBWTtTQUNiLENBQUMsQ0FBQztRQUVILE1BQU0sS0FBSyxHQUFHLElBQUksNEJBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtZQUM1QyxZQUFZO1lBQ1osU0FBUztTQUNWLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRW5DLE1BQU0sSUFBSSxHQUFHLElBQUkscUJBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN2QyxRQUFRLEVBQUUsbUJBQW1CO1lBQzdCLFNBQVMsRUFBRSxJQUFJLHFCQUFHLENBQUMsZ0JBQWdCLENBQUMsNEJBQTRCLENBQUM7U0FDbEUsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3ZDLE1BQU0sRUFBRSxxQkFBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCx5QkFBeUI7Z0JBQ3pCLG1CQUFtQjthQUNwQjtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUN2QyxNQUFNLEVBQUUscUJBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsOEJBQThCO2FBQy9CO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxlQUFlLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU5RSxNQUFNLEtBQUssR0FBaUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUNoRyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDdkUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN6QyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLG1CQUFtQixDQUFDLFdBQVcsRUFBRSxXQUFXLFdBQVcsRUFBRSxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDM0YsQ0FBQyxDQUFDLENBQUM7UUFFSCxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtZQUNsRSxRQUFRO1lBQ1IsS0FBSztZQUNMLElBQUk7WUFDSixXQUFXLEVBQUUsbUJBQW1CO1lBQ2hDLEtBQUs7U0FDTixDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBMkIsRUFBRSxDQUFDO1FBQzlDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN2RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pDLFVBQVUsQ0FBQyxXQUFXLENBQUMsR0FBRSxXQUFXLFdBQVcsRUFBRSxDQUFDO1FBQ3BELENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDcEQsYUFBYTtZQUNiLElBQUksRUFBRSxRQUFRO1lBQ2QsV0FBVyxFQUFFLGlCQUFpQjtZQUM5QixpQkFBaUIsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUM5RCxzQkFBc0IsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FDNUMsR0FBRyxFQUNILFVBQVUsRUFDVixDQUFDLE1BQU0sQ0FBQyxDQUNULENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM5QyxTQUFTLEVBQUUsUUFBUTtZQUNuQixZQUFZLEVBQUUsT0FBTztZQUNyQixjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUM7UUFDSCxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsYUFBYyxFQUFFLENBQUMsQ0FBQztRQUU1RSxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QixPQUFPLEVBQUUsUUFBUTtZQUNqQixhQUFhLEVBQUUsYUFBYTtZQUM1QixXQUFXLEVBQUUsV0FBVztZQUN4QixRQUFRLEVBQUUsQ0FBQyxNQUFNLENBQUM7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZCLENBQUM7Q0FDRjtBQWxHRCxvQ0FrR0M7QUFFRCxJQUFJLFlBQVksRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQge1xuICBhd3NfdGltZXN0cmVhbSBhcyB0cyxcbiAgYXdzX2lhbSBhcyBpYW0sXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGlmdyBmcm9tICcuJztcblxuZXhwb3J0IGNsYXNzIEludGVnVGVzdGluZyB7XG4gIHJlYWRvbmx5IHN0YWNrOiBjZGsuU3RhY2tbXTtcbiAgY29uc3RydWN0b3IoKSB7XG4gICAgY29uc3QgYXBwID0gbmV3IGNkay5BcHAoKTtcblxuICAgIGNvbnN0IGVudiA9IHtcbiAgICAgIHJlZ2lvbjogcHJvY2Vzcy5lbnYuQ0RLX0lOVEVHX1JFR0lPTiB8fCBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9SRUdJT04gfHwgJ3VzLWVhc3QtMScsXG4gICAgICBhY2NvdW50OiBwcm9jZXNzLmVudi5DREtfSU5URUdfQUNDT1VOVCB8fCBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5ULFxuICAgIH07XG5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnaW50ZWctc3RhY2snLCB7IGVudiB9KTtcblxuICAgIGNvbnN0IGRhdGFiYXNlTmFtZSA9ICdGbGVldFdpc2UnO1xuICAgIGNvbnN0IHRhYmxlTmFtZSA9ICdGbGVldFdpc2UnO1xuXG4gICAgY29uc3QgZGF0YWJhc2UgPSBuZXcgdHMuQ2ZuRGF0YWJhc2Uoc3RhY2ssICdEYXRhYmFzZScsIHtcbiAgICAgIGRhdGFiYXNlTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHRhYmxlID0gbmV3IHRzLkNmblRhYmxlKHN0YWNrLCAnVGFibGUnLCB7XG4gICAgICBkYXRhYmFzZU5hbWUsXG4gICAgICB0YWJsZU5hbWUsXG4gICAgfSk7XG5cbiAgICB0YWJsZS5ub2RlLmFkZERlcGVuZGVuY3koZGF0YWJhc2UpO1xuXG4gICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICByb2xlTmFtZTogJ2lvdGZsZWV0d2lzZS1yb2xlJyxcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdpb3RmbGVldHdpc2UuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgcm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICd0aW1lc3RyZWFtOldyaXRlUmVjb3JkcycsXG4gICAgICAgICd0aW1lc3RyZWFtOlNlbGVjdCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICByb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3RpbWVzdHJlYW06RGVzY3JpYmVFbmRwb2ludHMnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgfSkpO1xuXG4gICAgY29uc3QgY2FuRGJjID0gZnMucmVhZEZpbGVTeW5jKHBhdGguam9pbihfX2Rpcm5hbWUsICcvLi4vaHNjYW4uZGJjJyksICd1dGY4Jyk7XG5cbiAgICBjb25zdCBub2RlczogQXJyYXk8aWZ3LlNpZ25hbENhdGFsb2dOb2RlPiA9IFtuZXcgaWZ3LlNpZ25hbENhdGFsb2dCcmFuY2goJ1ZlaGljbGUnLCAnVmVoaWNsZScpXTtcbiAgICBjYW5EYmMuc3BsaXQoJ1xcbicpLmZpbHRlcihsaW5lID0+IC9eXFxzK1NHX1xccytcXHcrLy50ZXN0KGxpbmUpKS5tYXAobGluZSA9PiB7XG4gICAgICBjb25zdCBzaWduYWxfbmFtZSA9IGxpbmUuc3BsaXQoL1xccysvKVsyXTtcbiAgICAgIG5vZGVzLnB1c2gobmV3IGlmdy5TaWduYWxDYXRhbG9nU2Vuc29yKHNpZ25hbF9uYW1lLCBgVmVoaWNsZS4ke3NpZ25hbF9uYW1lfWAsICdET1VCTEUnKSk7XG4gICAgfSk7XG5cbiAgICBjb25zdCBzaWduYWxDYXRhbG9nID0gbmV3IGlmdy5TaWduYWxDYXRhbG9nKHN0YWNrLCAnU2lnbmFsQ2F0YWxvZycsIHtcbiAgICAgIGRhdGFiYXNlLFxuICAgICAgdGFibGUsXG4gICAgICByb2xlLFxuICAgICAgZGVzY3JpcHRpb246ICdteSBzaWduYWwgY2F0YWxvZycsXG4gICAgICBub2RlcyxcbiAgICB9KTtcblxuICAgIGNvbnN0IHNpZ25hbHNNYXA6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPSB7fTtcbiAgICBjYW5EYmMuc3BsaXQoJ1xcbicpLmZpbHRlcihsaW5lID0+IC9eXFxzK1NHX1xccytcXHcrLy50ZXN0KGxpbmUpKS5tYXAobGluZSA9PiB7XG4gICAgICBjb25zdCBzaWduYWxfbmFtZSA9IGxpbmUuc3BsaXQoL1xccysvKVsyXTtcbiAgICAgIHNpZ25hbHNNYXBbc2lnbmFsX25hbWVdID1gVmVoaWNsZS4ke3NpZ25hbF9uYW1lfWA7XG4gICAgfSk7XG5cbiAgICBjb25zdCBtb2RlbF9hID0gbmV3IGlmdy5WZWhpY2xlTW9kZWwoc3RhY2ssICdNb2RlbEEnLCB7XG4gICAgICBzaWduYWxDYXRhbG9nLFxuICAgICAgbmFtZTogJ21vZGVsQScsXG4gICAgICBkZXNjcmlwdGlvbjogJ01vZGVsIEEgdmVoaWNsZScsXG4gICAgICBuZXR3b3JrSW50ZXJmYWNlczogW25ldyBpZncuQ2FuVmVoaWNsZUludGVyZmFjZSgnMScsICd2Y2FuMCcpXSxcbiAgICAgIG5ldHdvcmtGaWxlRGVmaW5pdGlvbnM6IFtuZXcgaWZ3LkNhbkRlZmluaXRpb24oXG4gICAgICAgICcxJyxcbiAgICAgICAgc2lnbmFsc01hcCxcbiAgICAgICAgW2NhbkRiY10sXG4gICAgICApXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHZpbjEwMCA9IG5ldyBpZncuVmVoaWNsZShzdGFjaywgJ3ZpbjEwMCcsIHtcbiAgICAgIHZlaGljbGVJZDogJ3ZpbjEwMCcsXG4gICAgICB2ZWhpY2xlTW9kZWw6IG1vZGVsX2EsXG4gICAgICBjcmVhdGVJb3RUaGluZzogdHJ1ZSxcbiAgICB9KTtcbiAgICBuZXcgY2RrLkNmbk91dHB1dChzdGFjaywgJ2NlcnRpZmljYXRlSWQnLCB7IHZhbHVlOiB2aW4xMDAuY2VydGlmaWNhdGVJZCEgfSk7XG5cbiAgICBuZXcgaWZ3LkZsZWV0KHN0YWNrLCAnZmxlZXQxJywge1xuICAgICAgZmxlZXRJZDogJ2ZsZWV0MScsXG4gICAgICBzaWduYWxDYXRhbG9nOiBzaWduYWxDYXRhbG9nLFxuICAgICAgZGVzY3JpcHRpb246ICdteSBmbGVldDEnLFxuICAgICAgdmVoaWNsZXM6IFt2aW4xMDBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5zdGFjayA9IFtzdGFja107XG4gIH1cbn1cblxubmV3IEludGVnVGVzdGluZygpOyJdfQ==