/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.validate;

import java.util.Iterator;
import org.openscience.cdk.config.AtomTypeFactory;
import org.openscience.cdk.config.IsotopeFactory;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IAtomType;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IElement;
import org.openscience.cdk.interfaces.IIsotope;
import org.openscience.cdk.interfaces.IMolecule;
import org.openscience.cdk.interfaces.IMoleculeSet;
import org.openscience.cdk.interfaces.IPseudoAtom;
import org.openscience.cdk.interfaces.IReaction;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.openscience.cdk.tools.manipulator.BondManipulator;
import org.openscience.cdk.validate.AbstractValidator;
import org.openscience.cdk.validate.ValidationReport;
import org.openscience.cdk.validate.ValidationTest;

public class BasicValidator
extends AbstractValidator {
    private static ILoggingTool logger = LoggingToolFactory.createLoggingTool(BasicValidator.class);

    public ValidationReport validateAtom(IAtom subject) {
        ValidationReport report = new ValidationReport();
        report.addReport(this.validateCharge(subject));
        report.addReport(this.validateHydrogenCount(subject));
        report.addReport(this.validatePseudoAtom(subject));
        return report;
    }

    public ValidationReport validateBond(IBond subject) {
        ValidationReport report = new ValidationReport();
        report.addReport(this.validateStereoChemistry(subject));
        report.addReport(this.validateMaxBondOrder(subject));
        return report;
    }

    public ValidationReport validateIsotope(IIsotope subject) {
        return this.validateIsotopeExistence(subject);
    }

    public ValidationReport validateMolecule(IMolecule subject) {
        ValidationReport report = new ValidationReport();
        ValidationTest emptyMolecule = new ValidationTest(subject, "Molecule does not contain any atom");
        if (subject.getAtomCount() == 0) {
            report.addError(emptyMolecule);
        } else {
            report.addOK(emptyMolecule);
            ValidationTest massCalcProblem = new ValidationTest(subject, "Molecule contains PseudoAtom's. Won't be able to calculate some properties, like molecular mass.");
            boolean foundMassCalcProblem = false;
            for (int i = 0; i < subject.getAtomCount(); ++i) {
                if (subject.getAtom(i) instanceof IPseudoAtom) {
                    foundMassCalcProblem = true;
                    continue;
                }
                report.addReport(this.validateBondOrderSum(subject.getAtom(i), subject));
            }
            if (foundMassCalcProblem) {
                report.addWarning(massCalcProblem);
            } else {
                report.addOK(massCalcProblem);
            }
        }
        return report;
    }

    public ValidationReport validateReaction(IReaction subject) {
        ValidationReport report = new ValidationReport();
        IAtomContainer container1 = subject.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        IMoleculeSet reactants = subject.getReactants();
        for (int i = 0; i < reactants.getAtomContainerCount(); ++i) {
            container1.add(reactants.getMolecule(i));
        }
        IAtomContainer container2 = subject.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        IMoleculeSet products = subject.getProducts();
        for (int i = 0; i < products.getAtomContainerCount(); ++i) {
            container2.add(products.getMolecule(i));
        }
        report.addReport(this.validateAtomCountConservation(subject, container1, container2));
        report.addReport(this.validateChargeConservation(subject, container1, container2));
        return report;
    }

    private ValidationReport validateCharge(IAtom atom) {
        ValidationReport report = new ValidationReport();
        ValidationTest tooCharged = new ValidationTest(atom, "Atom has an unlikely large positive or negative charge");
        if (atom.getSymbol().equals("O") || atom.getSymbol().equals("N") || atom.getSymbol().equals("C") || atom.getSymbol().equals("H")) {
            if (atom.getFormalCharge() == 0) {
                report.addOK(tooCharged);
            } else {
                tooCharged.setDetails("Atom " + atom.getSymbol() + " has charge " + atom.getFormalCharge());
                if (atom.getFormalCharge() < -3) {
                    report.addError(tooCharged);
                } else if (atom.getFormalCharge() < -1) {
                    report.addWarning(tooCharged);
                } else if (atom.getFormalCharge() > 3) {
                    report.addError(tooCharged);
                } else if (atom.getFormalCharge() > 1) {
                    report.addWarning(tooCharged);
                }
            }
        } else if (atom.getFormalCharge() == 0) {
            report.addOK(tooCharged);
        } else {
            tooCharged.setDetails("Atom " + atom.getSymbol() + " has charge " + atom.getFormalCharge());
            if (atom.getFormalCharge() < -4) {
                report.addError(tooCharged);
            } else if (atom.getFormalCharge() < -3) {
                report.addWarning(tooCharged);
            } else if (atom.getFormalCharge() > 4) {
                report.addError(tooCharged);
            } else if (atom.getFormalCharge() > 3) {
                report.addWarning(tooCharged);
            }
        }
        return report;
    }

    private ValidationReport validateHydrogenCount(IAtom atom) {
        ValidationReport report = new ValidationReport();
        ValidationTest negativeHydrogenCount = new ValidationTest(atom, "An Atom cannot have a negative number of hydrogens attached.");
        if (atom.getImplicitHydrogenCount() < 0) {
            negativeHydrogenCount.setDetails("Atom has " + atom.getImplicitHydrogenCount() + " hydrogens.");
            report.addError(negativeHydrogenCount);
        } else {
            report.addOK(negativeHydrogenCount);
        }
        return report;
    }

    private ValidationReport validatePseudoAtom(IAtom atom) {
        ValidationReport report = new ValidationReport();
        ValidationTest isElementOrPseudo = new ValidationTest(atom, "Non-element atom must be of class PseudoAtom.");
        if (atom instanceof IPseudoAtom) {
            report.addOK(isElementOrPseudo);
        } else {
            try {
                IsotopeFactory isotopeFactory = IsotopeFactory.getInstance(atom.getBuilder());
                IElement element = isotopeFactory.getElement(atom.getSymbol());
                if (element == null) {
                    isElementOrPseudo.setDetails("Element " + atom.getSymbol() + " does not exist.");
                    report.addError(isElementOrPseudo);
                } else {
                    report.addOK(isElementOrPseudo);
                }
            }
            catch (Exception exception) {
                isElementOrPseudo.setDetails(exception.toString());
                report.addCDKError(isElementOrPseudo);
            }
        }
        return report;
    }

    private ValidationReport validateStereoChemistry(IBond bond) {
        ValidationReport report = new ValidationReport();
        ValidationTest bondStereo = new ValidationTest(bond, "Defining stereochemistry on bonds is not safe.", "Use atom based stereochemistry.");
        if (bond.getStereo() != IBond.Stereo.NONE) {
            report.addWarning(bondStereo);
        } else {
            report.addOK(bondStereo);
        }
        return report;
    }

    private ValidationReport validateMaxBondOrder(IBond bond) {
        ValidationReport report = new ValidationReport();
        ValidationTest maxBO = new ValidationTest(bond, "Bond order exceeds the maximum for one of its atoms.");
        try {
            IAtom atom;
            AtomTypeFactory structgenATF = AtomTypeFactory.getInstance("org/openscience/cdk/dict/data/cdk-atom-types.owl", bond.getBuilder());
            for (int i = 0; i < bond.getAtomCount() && !((atom = bond.getAtom(i)) instanceof IPseudoAtom); ++i) {
                IAtomType[] atomTypes = structgenATF.getAtomTypes(atom.getSymbol());
                IAtomType failedOn = null;
                boolean foundMatchingAtomType = false;
                for (int j = 0; j < atomTypes.length; ++j) {
                    if (!BondManipulator.isHigherOrder(bond.getOrder(), atomTypes[j].getMaxBondOrder())) {
                        foundMatchingAtomType = true;
                        continue;
                    }
                    failedOn = atomTypes[j];
                }
                if (foundMatchingAtomType) {
                    report.addOK(maxBO);
                    continue;
                }
                if (failedOn != null) {
                    maxBO.setDetails("Bond order exceeds the one allowed for atom " + atom.getSymbol() + " for which the maximum bond order is " + (Object)((Object)failedOn.getMaxBondOrder()));
                }
                report.addError(maxBO);
            }
        }
        catch (Exception exception) {
            logger.error("Error while performing atom bos validation");
            logger.debug(exception);
            maxBO.setDetails("Error while performing atom bos validation: " + exception.toString());
            report.addCDKError(maxBO);
        }
        return report;
    }

    public ValidationReport validateIsotopeExistence(IIsotope isotope) {
        ValidationReport report = new ValidationReport();
        ValidationTest isotopeExists = new ValidationTest(isotope, "Isotope with this mass number is not known for this element.");
        try {
            IsotopeFactory isotopeFac = IsotopeFactory.getInstance(isotope.getBuilder());
            IIsotope[] isotopes = isotopeFac.getIsotopes(isotope.getSymbol());
            if (isotope.getMassNumber() != 0) {
                boolean foundKnownIsotope = false;
                for (int i = 0; i < isotopes.length; ++i) {
                    if (isotopes[i].getMassNumber() != isotope.getMassNumber()) continue;
                    foundKnownIsotope = true;
                }
                if (!foundKnownIsotope) {
                    report.addError(isotopeExists);
                } else {
                    report.addOK(isotopeExists);
                }
            } else {
                report.addOK(isotopeExists);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return report;
    }

    private ValidationReport validateBondOrderSum(IAtom atom, IMolecule molecule) {
        ValidationReport report = new ValidationReport();
        ValidationTest checkBondSum = new ValidationTest(atom, "The atom's total bond order is too high.");
        try {
            AtomTypeFactory structgenATF = AtomTypeFactory.getInstance("org/openscience/cdk/dict/data/cdk-atom-types.owl", atom.getBuilder());
            int bos = (int)molecule.getBondOrderSum(atom);
            IAtomType[] atomTypes = structgenATF.getAtomTypes(atom.getSymbol());
            if (atomTypes.length == 0) {
                checkBondSum.setDetails("Cannot validate bond order sum for atom not in valency atom type list: " + atom.getSymbol());
                report.addWarning(checkBondSum);
            } else {
                IAtomType failedOn = null;
                boolean foundMatchingAtomType = false;
                for (int j = 0; j < atomTypes.length; ++j) {
                    IAtomType type = atomTypes[j];
                    if (atom.getFormalCharge() != type.getFormalCharge()) continue;
                    foundMatchingAtomType = true;
                    if ((double)bos == type.getBondOrderSum()) continue;
                    failedOn = atomTypes[j];
                }
                if (foundMatchingAtomType) {
                    report.addOK(checkBondSum);
                } else {
                    if (failedOn != null) {
                        checkBondSum.setDetails("Bond order exceeds the one allowed for atom " + atom.getSymbol() + " for which the total bond order is " + failedOn.getBondOrderSum());
                    }
                    report.addError(checkBondSum);
                }
            }
        }
        catch (Exception exception) {
            logger.error("Error while performing atom bos validation: ", exception.getMessage());
            logger.debug(exception);
        }
        return report;
    }

    private ValidationReport validateAtomCountConservation(IReaction reaction, IAtomContainer reactants, IAtomContainer products) {
        ValidationReport report = new ValidationReport();
        ValidationTest atomCount = new ValidationTest(reaction, "Atom count mismatch for reaction: the product side has a different atom count than the reactant side.");
        if (reactants.getAtomCount() != products.getAtomCount()) {
            report.addError(atomCount);
        } else {
            report.addOK(atomCount);
        }
        return report;
    }

    private ValidationReport validateChargeConservation(IReaction reaction, IAtomContainer reactants, IAtomContainer products) {
        ValidationReport report = new ValidationReport();
        ValidationTest chargeConservation = new ValidationTest(reaction, "Total formal charge is not preserved during the reaction");
        Iterator<IAtom> atoms1 = reactants.atoms().iterator();
        int totalCharge1 = 0;
        while (atoms1.hasNext()) {
            totalCharge1 = atoms1.next().getFormalCharge();
        }
        Iterator<IAtom> atoms2 = products.atoms().iterator();
        int totalCharge2 = 0;
        while (atoms2.hasNext()) {
            totalCharge2 = atoms2.next().getFormalCharge();
        }
        if (totalCharge1 != totalCharge2) {
            report.addError(chargeConservation);
        } else {
            report.addOK(chargeConservation);
        }
        return report;
    }
}

