"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const rds_generated_1 = require("./rds.generated");
/**
 * An option group.
 */
class OptionGroup extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        this.configurations = [];
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        props.configurations.forEach(config => this.addConfiguration(config));
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: core_1.Lazy.any({ produce: () => this.renderConfigurations(this.configurations) }),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
            addConfiguration(_) { return false; }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a configuration to this OptionGroup.
     *
     * This method is a no-op for an imported OptionGroup.
     */
    addConfiguration(configuration) {
        this.configurations.push(configuration);
        if (configuration.port) {
            if (!configuration.vpc) {
                throw new Error('`port` and `vpc` must be specified together.');
            }
            const securityGroups = configuration.securityGroups && configuration.securityGroups.length > 0
                ? configuration.securityGroups
                : [new ec2.SecurityGroup(this, `SecurityGroup${configuration.name}`, {
                        description: `Security group for ${configuration.name} option`,
                        vpc: configuration.vpc,
                    })];
            this.optionConnections[configuration.name] = new ec2.Connections({
                securityGroups: securityGroups,
                defaultPort: ec2.Port.tcp(configuration.port),
            });
        }
        return true;
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            const securityGroups = config.vpc
                ? this.optionConnections[config.name].securityGroups.map(sg => sg.securityGroupId)
                : undefined;
            configs.push({
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
                port: config.port,
                vpcSecurityGroupMemberships: securityGroups,
            });
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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