"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        /**
         * Identifiers of the replicas.
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_a = props.subnetGroup) !== null && _a !== void 0 ? _a : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        this.securityGroups = (_b = props.instanceProps.securityGroups) !== null && _b !== void 0 ? _b : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: (_c = clusterEngineBindConfig.features) === null || _c === void 0 ? void 0 : _c.s3Import });
        }
        if (s3ExportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: (_d = clusterEngineBindConfig.features) === null || _d === void 0 ? void 0 : _d.s3Export });
        }
        const clusterParameterGroup = (_e = props.parameterGroup) !== null && _e !== void 0 ? _e : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        this.engine = props.engine;
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_f = props.engine.engineVersion) === null || _f === void 0 ? void 0 : _f.fullVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: (_g = props.port) !== null && _g !== void 0 ? _g : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            // Admin
            backupRetentionPeriod: (_j = (_h = props.backup) === null || _h === void 0 ? void 0 : _h.retention) === null || _j === void 0 ? void 0 : _j.toDays(),
            preferredBackupWindow: (_k = props.backup) === null || _k === void 0 ? void 0 : _k.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
        };
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.engine = attrs.engine;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: credentials.username,
            masterUserPassword: (_a = credentials.password) === null || _a === void 0 ? void 0 : _a.toString(),
            // Encryption
            kmsKeyId: (_b = props.storageEncryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        util_1.applyRemovalPolicy(cluster, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        setLogRetention(this, props);
        createInstances(this, props, this.subnetGroup);
    }
    /**
     * Import an existing DatabaseCluster from properties.
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options = {}) {
        var _a;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_a = options.excludeCharacters) !== null && _a !== void 0 ? _a : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        var _a;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_a = options.excludeCharacters) !== null && _a !== void 0 ? _a : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * A database cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        util_1.applyRemovalPolicy(cluster, props.removalPolicy);
        setLogRetention(this, props);
        createInstances(this, props, this.subnetGroup);
    }
}
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    var _a, _b, _c, _d;
    const instanceCount = props.instances != null ? props.instances : 2;
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = (_a = instanceProps.instanceType) !== null && _a !== void 0 ? _a : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    const instanceParameterGroupConfig = (_b = instanceProps.parameterGroup) === null || _b === void 0 ? void 0 : _b.bindToInstance({});
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const publiclyAccessible = instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible,
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: (_d = instanceProps.performanceInsightEncryptionKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            autoMinorVersionUpgrade: props.instanceProps.autoMinorVersionUpgrade,
            allowMajorVersionUpgrade: props.instanceProps.allowMajorVersionUpgrade,
            deleteAutomatedBackups: props.instanceProps.deleteAutomatedBackups,
        });
        // If removalPolicy isn't explicitly set,
        // it's Snapshot for Cluster.
        // Because of that, in this case,
        // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
        if (props.removalPolicy) {
            util_1.applyRemovalPolicy(instance, props.removalPolicy);
        }
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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