import * as ec2 from '@aws-cdk/aws-ec2';
import { IRole } from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as s3 from '@aws-cdk/aws-s3';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Duration, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IClusterEngine } from './cluster-engine';
import { DatabaseClusterAttributes, IDatabaseCluster } from './cluster-ref';
import { Endpoint } from './endpoint';
import { IParameterGroup } from './parameter-group';
import { BackupProps, Credentials, InstanceProps, RotationSingleUserOptions, RotationMultiUserOptions } from './props';
import { DatabaseProxy, DatabaseProxyOptions } from './proxy';
import { CfnDBClusterProps } from './rds.generated';
import { ISubnetGroup } from './subnet-group';
/**
 * Common properties for a new database cluster or cluster from snapshot.
 */
interface DatabaseClusterBaseProps {
    /**
     * What kind of database to start.
     */
    readonly engine: IClusterEngine;
    /**
     * How many replicas/instances to create.
     *
     * Has to be at least 1.
     *
     * @default 2
     */
    readonly instances?: number;
    /**
     * Settings for the individual instances that are launched.
     */
    readonly instanceProps: InstanceProps;
    /**
     * Backup settings.
     *
     * @default - Backup retention period for automated backups is 1 day.
     * Backup preferred window is set to a 30-minute window selected at random from an
     * 8-hour block of time for each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     */
    readonly backup?: BackupProps;
    /**
     * What port to listen on.
     *
     * @default - The default for the engine is used.
     */
    readonly port?: number;
    /**
     * An optional identifier for the cluster.
     *
     * @default - A name is automatically generated.
     */
    readonly clusterIdentifier?: string;
    /**
     * Base identifier for instances.
     *
     * Every replica is named by appending the replica number to this string, 1-based.
     *
     * @default - clusterIdentifier is used with the word "Instance" appended.
     * If clusterIdentifier is not provided, the identifier is automatically generated.
     */
    readonly instanceIdentifierBase?: string;
    /**
     * Name of a database which is automatically created inside the cluster.
     *
     * @default - Database is not created in cluster.
     */
    readonly defaultDatabaseName?: string;
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if ``removalPolicy`` is RETAIN, false otherwise
     */
    readonly deletionProtection?: boolean;
    /**
     * A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).
     *
     * Example: 'Sun:23:45-Mon:00:15'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * Additional parameters to pass to the database engine.
     *
     * @default - No parameter group.
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * The list of log types that need to be enabled for exporting to CloudWatch Logs.
     *
     * @default - no log exports
     */
    readonly cloudwatchLogsExports?: string[];
    /**
     * The number of days log events are kept in CloudWatch Logs.
     *
     * When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `Infinity`.
     *
     * @default - logs never expire
     */
    readonly cloudwatchLogsRetention?: logs.RetentionDays;
    /**
     * The IAM role for the Lambda function associated with the custom resource that sets the retention policy.
     *
     * @default - a new role is created.
     */
    readonly cloudwatchLogsRetentionRole?: IRole;
    /**
     * The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances.
     *
     * @default no enhanced monitoring
     */
    readonly monitoringInterval?: Duration;
    /**
     * Role that will be used to manage DB instances monitoring.
     *
     * @default - A role is automatically created for you
     */
    readonly monitoringRole?: IRole;
    /**
     * Role that will be associated with this DB cluster to enable S3 import.
     *
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportBuckets` is used.
     *
     * For MySQL:
     *
     * @default - New role is created if `s3ImportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     */
    readonly s3ImportRole?: IRole;
    /**
     * S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportRole` is used.
     *
     * For MySQL:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     */
    readonly s3ImportBuckets?: s3.IBucket[];
    /**
     * Role that will be associated with this DB cluster to enable S3 export.
     *
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportBuckets` is used.
     *
     * For MySQL:
     *
     * @default - New role is created if `s3ExportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     */
    readonly s3ExportRole?: IRole;
    /**
     * S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportRole` is used.
     *
     * For MySQL:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     */
    readonly s3ExportBuckets?: s3.IBucket[];
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     */
    readonly subnetGroup?: ISubnetGroup;
}
/**
 * A new or imported clustered database.
 */
export declare abstract class DatabaseClusterBase extends Resource implements IDatabaseCluster {
    /**
     * The engine of this Cluster.
     *
     * May be not known for imported Clusters if it wasn't provided explicitly.
     */
    abstract readonly engine?: IClusterEngine;
    /**
     * Identifier of the cluster.
     */
    abstract readonly clusterIdentifier: string;
    /**
     * Identifiers of the replicas.
     */
    abstract readonly instanceIdentifiers: string[];
    /**
     * The endpoint to use for read/write operations.
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * Endpoints which address each individual replica.
     */
    abstract readonly instanceEndpoints: Endpoint[];
    /**
     * Access to the network connections.
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Add a new db proxy to this cluster.
     */
    addProxy(id: string, options: DatabaseProxyOptions): DatabaseProxy;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
declare abstract class DatabaseClusterNew extends DatabaseClusterBase {
    /**
     * The engine for this Cluster.
     *
     * Never undefined.
     */
    readonly engine?: IClusterEngine;
    /**
     * Identifiers of the replicas.
     */
    readonly instanceIdentifiers: string[];
    /**
     * Endpoints which address each individual replica.
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     *
     */
    protected readonly newCfnProps: CfnDBClusterProps;
    /**
     *
     */
    protected readonly securityGroups: ec2.ISecurityGroup[];
    /**
     *
     */
    protected readonly subnetGroup: ISubnetGroup;
    constructor(scope: Construct, id: string, props: DatabaseClusterBaseProps);
}
/**
 * Properties for a new database cluster.
 */
export interface DatabaseClusterProps extends DatabaseClusterBaseProps {
    /**
     * Credentials for the administrative user.
     *
     * @default - A username of 'admin' (or 'postgres' for PostgreSQL) and SecretsManager-generated password
     */
    readonly credentials?: Credentials;
    /**
     * Whether to enable storage encryption.
     *
     * @default - true if storageEncryptionKey is provided, false otherwise
     */
    readonly storageEncrypted?: boolean;
    /**
     * The KMS key for storage encryption.
     *
     * If specified, {@link storageEncrypted} will be set to `true`.
     *
     * @default - if storageEncrypted is true then the default master key, no key otherwise
     */
    readonly storageEncryptionKey?: kms.IKey;
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
export declare class DatabaseCluster extends DatabaseClusterNew {
    /**
     * Import an existing DatabaseCluster from properties.
     */
    static fromDatabaseClusterAttributes(scope: Construct, id: string, attrs: DatabaseClusterAttributes): IDatabaseCluster;
    /**
     * Identifier of the cluster.
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations.
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections.
     */
    readonly connections: ec2.Connections;
    /**
     * The secret attached to this cluster.
     */
    readonly secret?: secretsmanager.ISecret;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     *
     */
    constructor(scope: Construct, id: string, props: DatabaseClusterProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
/**
 * Properties for ``DatabaseClusterFromSnapshot``.
 */
export interface DatabaseClusterFromSnapshotProps extends DatabaseClusterBaseProps {
    /**
     * The identifier for the DB instance snapshot or DB cluster snapshot to restore from.
     *
     * You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot.
     * However, you can use only the ARN to specify a DB instance snapshot.
     */
    readonly snapshotIdentifier: string;
}
/**
 * A database cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    /**
     * Identifier of the cluster.
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations.
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections.
     */
    readonly connections: ec2.Connections;
    /**
     *
     */
    constructor(scope: Construct, id: string, props: DatabaseClusterFromSnapshotProps);
}
export {};
