"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string, for example "5.6.mysql_aurora.1.78.3.6".
     * @param auroraMajorVersion the major version of the engine, defaults to "5.6".
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/**
 * Version "5.6.10a".
 */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/**
 * Version "5.6.mysql_aurora.1.17.9".
 */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/**
 * Version "5.6.mysql_aurora.1.19.0".
 */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/**
 * Version "5.6.mysql_aurora.1.19.1".
 */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/**
 * Version "5.6.mysql_aurora.1.19.2".
 */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/**
 * Version "5.6.mysql_aurora.1.19.5".
 */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/**
 * Version "5.6.mysql_aurora.1.19.6".
 */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/**
 * Version "5.6.mysql_aurora.1.20.0".
 */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/**
 * Version "5.6.mysql_aurora.1.20.1".
 */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/**
 * Version "5.6.mysql_aurora.1.21.0".
 */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/**
 * Version "5.6.mysql_aurora.1.22.0".
 */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/**
 * Version "5.6.mysql_aurora.1.22.1".
 */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/**
 * Version "5.6.mysql_aurora.1.22.1.3".
 */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/**
 * Version "5.6.mysql_aurora.1.22.2".
 */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string, for example "5.7.mysql_aurora.2.78.3.6".
     * @param auroraMysqlMajorVersion the major version of the engine, defaults to "5.7".
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/**
 * Version "5.7.12".
 */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/**
 * Version "5.7.mysql_aurora.2.03.2".
 */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/**
 * Version "5.7.mysql_aurora.2.03.3".
 */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/**
 * Version "5.7.mysql_aurora.2.03.4".
 */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/**
 * Version "5.7.mysql_aurora.2.04.0".
 */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/**
 * Version "5.7.mysql_aurora.2.04.1".
 */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/**
 * Version "5.7.mysql_aurora.2.04.2".
 */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/**
 * Version "5.7.mysql_aurora.2.04.3".
 */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/**
 * Version "5.7.mysql_aurora.2.04.4".
 */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/**
 * Version "5.7.mysql_aurora.2.04.5".
 */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/**
 * Version "5.7.mysql_aurora.2.04.6".
 */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/**
 * Version "5.7.mysql_aurora.2.04.7".
 */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/**
 * Version "5.7.mysql_aurora.2.04.8".
 */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/**
 * Version "5.7.mysql_aurora.2.05.0".
 */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/**
 * Version "5.7.mysql_aurora.2.06.0".
 */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/**
 * Version "5.7.mysql_aurora.2.07.0".
 */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/**
 * Version "5.7.mysql_aurora.2.07.1".
 */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/**
 * Version "5.7.mysql_aurora.2.07.2".
 */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/**
 * Version "5.7.mysql_aurora.2.08.0".
 */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/**
 * Version "5.7.mysql_aurora.2.08.1".
 */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/**
 * Version "5.7.mysql_aurora.2.08.2".
 */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/**
 * Version "5.7.mysql_aurora.2.09.0".
 */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string, for example "9.6.25.1".
     * @param auroraPostgresMajorVersion the major version of the engine, for example "9.6".
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/**
 * Version "9.6.8".
 */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/**
 * Version "9.6.9".
 */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/**
 * Version "9.6.11".
 */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/**
 * Version "9.6.12".
 */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/**
 * Version "9.6.16".
 */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/**
 * Version "9.6.17".
 */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/**
 * Version "10.4".
 */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/**
 * Version "10.5".
 */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/**
 * Version "10.6".
 */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/**
 * Version "10.7".
 */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * Version "10.11".
 */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.12".
 */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/**
 * Version "11.4".
 */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * Version "11.6".
 */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.7".
 */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.8".
 */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _a, _b, _c;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_a = this.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) {
            if (options.s3ImportRole && !((_b = config.features) === null || _b === void 0 ? void 0 : _b.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_c = config.features) === null || _c === void 0 ? void 0 : _c.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine.
 *
 * Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /**
     * Creates a new plain Aurora database cluster engine.
     */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora MySQL database cluster engine.
     */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora PostgreSQL database cluster engine.
     */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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