# coding: utf-8
#
#  This file is part of Sequana software
#
#  Copyright (c) 2020 - Sequana Development Team
#
#  Distributed under the terms of the 3-clause BSD license.
#  The full license is in the LICENSE file, distributed with this software.
#
#  website: https://github.com/sequana/sequana
#  documentation: http://sequana.readthedocs.io
#
##############################################################################
"""Module to write variant calling report"""
import ast

import pandas as pd

from sequana.modules_report.base_module import SequanaBaseModule
from sequana.utils.datatables_js import DataTable


class Enrichment(SequanaBaseModule):
    """ Write HTML report of variant calling. This class takes a csv file
    generated by sequana_variant_filter.
    """
    def __init__(self, rnadiff_folder, taxon,
                kegg_organism=None,
                 enrichment_params={
                        "log2_fc": 3,
                        "mapper": "biomart.csv",
                        "preload_directory": "kegg_pathways"
                        }
                ):
        """.. rubric:: constructor

        """
        super().__init__()
        self.title = "Enrichment"

        self.rnadiff_folder = rnadiff_folder
        self.enrichment_params = enrichment_params
        self.organism = kegg_organism
        self.taxon = taxon
        if taxon == 10090:
            self.organism = "mmu"

        from sequana.rnadiff import RNADiffResults
        self.rnadiff = RNADiffResults(self.rnadiff_folder)

        self.create_report_content()
        self.create_html("enrichment.html")

    def create_report_content(self):
        self.sections = list()
        self.summary()
        self.add_go()
        self.add_kegg()

    def summary(self):
        """ Add information of filter."""

        S = self.rnadiff.summary()
        Sup = S.loc['up'][0]
        Sdown = S.loc['down'][0]
        Stotal = Sup + Sdown
        link_rnadiff = self.copy_file(self.rnadiff.filename, ".")
        log2fc = self.enrichment_params["log2_fc"]

        self.sections.append({
            'name': "1 - Summary",
            'anchor': 'filters_option',
            'content':
                f"""
<p>The final Differententially Gene Expression (DGE) analysis
led to {Sup} up and {Sdown} down genes (total {Stotal})</p>

<p>In the following sections, you will find the KEGG Pathway enrichment and GO
terms enrichment. The input data for those analyis filters out all
adjusted p-values below 0.05 and log2 fold change (absolute) greater than
{log2fc}. As for the fold change, the numbers above have no
threshold. This may explain that numbers may be large. In the KEGG and
GO enrichment we may add a threshold to filter out genes with low fold change. 
</p>

<p>Input file from the RNADiff analysis downloadable here and in the table below <a href="{link_rnadiff}">here</a></p>
"""
        })

    def add_go(self):
        style="width:85%"
        from sequana import logger
        level = logger.level
        logger.level = "INFO"
        from sequana.enrichment import PantherEnrichment
        self.pe = PantherEnrichment(self.rnadiff_folder,
            log2_fc_threshold=self.enrichment_params['log2_fc'])

        # First, the down regulated GO tersm
        # FIXME. useless to run twice but we want the df. may can just create
        # the image and use png_embedded_png
        self.pe.compute_enrichment(self.pe.mygenes_down, self.taxon)
        df_down = self.pe.plot_go_terms(['GO:0003674', 'GO:0008150', 'GO:0005575'])
        def plot_go_terms_down(filename):
            import pylab
            self.pe.compute_enrichment(self.pe.mygenes_down, self.taxon)
            df = self.pe.plot_go_terms(['GO:0003674', 'GO:0008150', 'GO:0005575'])
            pylab.savefig(filename)
        img_down = self.create_embedded_png(plot_go_terms_down, "filename", style=style)

        self.pe.compute_enrichment(self.pe.mygenes_up, self.taxon)
        df_up = self.pe.plot_go_terms(['GO:0003674', 'GO:0008150', 'GO:0005575'])

        def plot_go_terms_up(filename):
            import pylab
            self.pe.compute_enrichment(self.pe.mygenes_up, self.taxon)
            df = self.pe.plot_go_terms(['GO:0003674', 'GO:0008150', 'GO:0005575'])
            pylab.savefig(filename)
        img_up = self.create_embedded_png(plot_go_terms_up, "filename", style=style)
        html = f"{img_down} <hr>{img_up}"
        self.sections.append({"name": "GO", "anchor": "go", "content": html})
        logger.level = level
        #pe.compute_enrichment(pe.mygenes_up, self.taxon)
        #pe.plot_go_terms((['GO:0003674', 'GO:0008150', 'GO:0005575'])



    def add_kegg(self):
        style="width:45%"
        from sequana.enrichment import KeggPathwayEnrichment
        ke = KeggPathwayEnrichment(self.rnadiff,
            self.organism,
            mapper=self.enrichment_params["mapper"],
            log2_fc=self.enrichment_params['log2_fc'],
            preload_directory=self.enrichment_params["preload_directory"] + "/" + self.organism)


        # Image kegg pathways down
        def plot_barplot_down(filename):
            import pylab
            ke.barplot(ke.enrichment['down'])
            pylab.savefig(filename)
        img_barplot_down = self.create_embedded_png(plot_barplot_down, "filename", style=style)
        def plot_scatter_down(filename):
            import pylab
            ke.scatterplot(ke.enrichment['down'])
            pylab.savefig(filename)
        img_scatter_down = self.create_embedded_png(plot_scatter_down, "filename", style=style)

        # Image kegg pathways up
        def plot_barplot_up(filename):
            import pylab
            ke.barplot(ke.enrichment['up'])
            pylab.savefig(filename)
        img_barplot_up = self.create_embedded_png(plot_barplot_up, "filename", style=style)
        def plot_scatter_up(filename):
            import pylab
            ke.scatterplot(ke.enrichment['up'])
            pylab.savefig(filename)
        img_scatter_up = self.create_embedded_png(plot_scatter_up, "filename", style=style)



        # Results down (pathway info)
        html_before_table = """<p>Enrichment pathways summary</p>"""
        df_down = ke.barplot(ke.enrichment['down'])
        links = ["https://www.genome.jp/dbget-bin/www_bget?path:{}".format(x) for x in df_down["pathway_id"]]
        df_down['links'] = links
        df_down = df_down[["pathway_id", "name", "size", "Overlap", "P-value", "Adjusted P-value", "Genes", "links"]]


        # save pathways and add fotorama
        from sequana import logger
        level = logger.level
        logger.level = "WARNING"
        from easydev import Progress
        pb = Progress(len(df_down))
        files = []
        for i, ID in enumerate(df_down['pathway_id']):
            df = ke.save_pathway(ID)
            files.append(ID + ".png")
            pb.animate(i+1)
        fotorama_down = self.add_fotorama(files)


        datatable = DataTable(df_down, 'kegg_down')
        datatable.datatable.set_links_to_column("links", "pathway_id")
        datatable.datatable.datatable_options = {
             'scrollX': 'true',
             'pageLength': 20,
             'scrollCollapse': 'true',
             'dom': 'Bfrtip',
             'buttons': ['copy', 'csv']
        }
        js_table_down = datatable.create_javascript_function()
        html_table_down = datatable.create_datatable(float_format='%E')


        # Results up (pathway info)
        df_up = ke.barplot(ke.enrichment['up'])
        links = ["https://www.genome.jp/dbget-bin/www_bget?path:{}".format(x) for x in df_up["pathway_id"]]
        df_up['links'] = links
        df_up = df_up[["pathway_id", "name", "size", "Overlap", "P-value", "Adjusted P-value", "Genes", "links"]]
        datatable = DataTable(df_up, 'kegg_up')
        datatable.datatable.set_links_to_column("links", "pathway_id")
        datatable.datatable.datatable_options = {
             'scrollX': 'true',
             'pageLength': 20,
             'scrollCollapse': 'true',
             'dom': 'Bfrtip',
             'buttons': ['copy', 'csv']
        }
        js_table_up = datatable.create_javascript_function()
        html_table_up = datatable.create_datatable(float_format='%E')
        pb = Progress(len(df_up))
        files = []
        for i, ID in enumerate(df_up['pathway_id']):
            df = ke.save_pathway(ID)
            files.append(ID + ".png")
            pb.animate(i+1)
        fotorama_up = self.add_fotorama(files)
        logger.level = level

        Ndown = len(df_down)
        Nup = len(df_up)

        html = f"""
<h3>2.1 - KEGG pathways down regulated</h3>
<p>{Ndown} KEGG pathways are found to be down regulated</p>
<br>
{img_barplot_down}
{img_scatter_down}
<hr>
{js_table_down} {html_table_down}
<hr>
{fotorama_down}


<h3>2.1 - KEGG pathways up regulated</h3>
<p>{Nup} KEGG pathways are found to be up regulated</p>
<br>
{img_barplot_up}
{img_scatter_up}
<hr>
{js_table_up} {html_table_up}
<hr>
{fotorama_up}
"""
        self.sections.append({"name": "2 - KEGG", "anchor": "kegg", "content": html})



    
