from .mobileattckobject import MobileAttckObject


class MobileAttckMalware(MobileAttckObject):
    '''A child class of MobileAttckObject
    
    Creates objects which have been categorized as malware used in attacks
    
    Example:
        You can iterate over a `malwares` list and access specific properties and relationship properties.

        The following relationship properties are accessible:
                1. actors
                2. techniques
        
            1. To iterate over an `malwares` list, do the following:

            .. code-block:: python
               
               from pyattck import Attck

               attck = Attck()

               for malware in attck.mobile.malwares:
                   print(malware.id)
                   print(malware.name)
                   print(malware.aliases)
                   print(malware.description)
                   # etc.

            2. To access relationship properties, do the following:

            .. code-block:: python

               from pyattck import Attck

               attck = Attck()

               for malware in attck.mobile.malwares:
                   print(malware.id)
                   print(malware.name)
                   print(malware.aliases)
                   print(malware.description)

                   for technique in malware.techniques:
                       print(technique.name)
                       print(technique.description)
                       # etc.
    '''

    def __init__(self, mobile_attck_obj = None, **kwargs):
        """
        Creates an MobileAttckMalware object.  
        The MobileAttckMalware object is based on malware which have been categorized as software used in attacks
        
        Arguments:
            mobile_attck_obj (json) -- Takes the raw MITRE Mobile ATT&CK Json object
            AttckObject (dict) -- Takes the MITRE Mobile ATT&CK Json object as a kwargs values

        """
        super(MobileAttckMalware, self).__init__(**kwargs)
        self.__mobile_attck_obj = mobile_attck_obj

        self.old_attack_id = self._set_attribute(kwargs, 'x_mitre_old_attack_id')
        self.platforms = self._set_list_items(kwargs, 'x_mitre_platforms')
        self.aliases = self._set_list_items(kwargs, 'x_mitre_aliases')
        self.version = self._set_attribute(kwargs, 'x_mitre_version')
        self.created_by_ref = self._set_attribute(kwargs, 'created_by_ref')
        self.labels = self._set_list_items(kwargs, 'labels')
        self.external_references = self._set_reference(kwargs)
        self.contributor = self._set_list_items(kwargs, 'x_mitre_contributors')
        self.stix = self._set_attribute(kwargs, 'id')

        self.set_relationships(self.__mobile_attck_obj)

    @property
    def actors(self):
        """Accessing known actors who have used a piece of malware as part of the MITRE Mobile ATT&CK Framework

        Returns:
            list: Returns all actor objects as a list that are documented using a piece of malware
        """
        from .actor import MobileAttckActor
        return_list = []
        item_dict = {}
        for item in self.__mobile_attck_obj['objects']:
            if 'type' in item:
                if item['type'] == 'intrusion-set':
                    item_dict[item['id']] = item
        
        for item in self._RELATIONSHIPS[self.stix]:
            if item in item_dict:
                return_list.append(MobileAttckActor(**item_dict[item]))
        return return_list

    @property
    def techniques(self):
        """Accessing known techniques a piece of malware has been used in as part of the MITRE Mobile ATT&CK Framework

        Returns:
            list: Returns all technique objects as a list that are documented as using a piece of malware
        """
        from .technique import MobileAttckTechnique
        return_list = []
        item_dict = {}
        for item in self.__mobile_attck_obj['objects']:
            if 'type' in item:
                if item['type'] == 'intrusion-set':
                    item_dict[item['id']] = item
        
        for item in self._RELATIONSHIPS[self.stix]:
            if item in item_dict:
                return_list.append(MobileAttckTechnique(**item_dict[item]))
        return return_list