from .attckobject import AttckObject


class AttckMalware(AttckObject):
    '''A child class of AttckObject
    
    Creates objects which have been categorized as malware used in attacks
    
    Example:
        You can iterate over a `malwares` list and access specific properties and relationship properties.

        The following relationship properties are accessible:
                1. actors
                2. techniques
        
            1. To iterate over an `malwares` list, do the following:

            .. code-block:: python
               
               from pyattck import Attck

               attck = Attck()

               for malware in attck.enterprise.malwares:
                   print(malware.id)
                   print(malware.name)
                   print(malware.aliases)
                   print(malware.description)
                   # etc.

            2. To access relationship properties, do the following:

            .. code-block:: python

               from pyattck import Attck

               attck = Attck()

               for malware in attck.enterprise.malwares:
                   print(malware.id)
                   print(malware.name)
                   print(malware.aliases)
                   print(malware.description)

                   for technique in malware.techniques:
                       print(technique.name)
                       print(technique.description)
                       # etc.

    Arguments:
        attck_obj (json) -- Takes the raw MITRE ATT&CK Json object
        AttckObject (dict) -- Takes the MITRE ATT&CK Json object as a kwargs values
    '''

    def __init__(self, attck_obj = None, **kwargs):
        """This class represents malware as defined with the Enterprise MITRE ATT&CK framework.

        Keyword Arguments:
            attck_obj {json} -- A Enterprise MITRE ATT&CK Framework json object (default: {None})
        """
        super(AttckMalware, self).__init__(**kwargs)
        self.__attck_obj = attck_obj

        self.id = self._set_id(kwargs)
        self.created_by_ref = self._set_attribute(kwargs, 'created_by_ref')
        self.name = self._set_attribute(kwargs, 'name')
        self.aliases = self._set_list_items(kwargs, 'x_mitre_aliases')
        self.platforms = self._set_list_items(kwargs, 'x_mitre_platforms')
        self.labels = self._set_list_items(kwargs, 'labels')
        self.description = self._set_attribute(kwargs, 'description')
        self.external_references = self._set_reference(kwargs)
        self.created = self._set_attribute(kwargs, 'created')
        self.modified = self._set_attribute(kwargs, 'modified')
        self.stix = self._set_attribute(kwargs, 'id')
        self.type = self._set_attribute(kwargs, 'type')
        self.revoked = self._set_attribute(kwargs, 'revoked')

        self.set_relationships(self.__attck_obj)

    @property
    def actors(self):
        """Returns all actor objects as a list that are documented as using this malware

        Returns:
            [list] -- A list of related actor objects defined within the Enterprise MITRE ATT&CK Framework for a malware object
        """
        from .actor import AttckActor
        return_list = []
        item_dict = {}
        for item in self.__attck_obj['objects']:
            if 'type' in item:
                if item['type'] == 'intrusion-set':
                    item_dict[item['id']] = item
        
        for item in self._RELATIONSHIPS[self.stix]:
            if item in item_dict:
                return_list.append(AttckActor(**item_dict[item]))
        return return_list

    @property
    def techniques(self):
        """Returns all technique objects as a list that are documented as using this malware

        Returns:
            [list] -- A list of related technique objects defined within the Enterprise MITRE ATT&CK Framework for a malware object
        """
        from .technique import AttckTechnique
        return_list = []
        item_dict = {}
        for item in self.__attck_obj['objects']:
            if 'type' in item:
                if item['type'] == 'intrusion-set':
                    item_dict[item['id']] = item
        
        for item in self._RELATIONSHIPS[self.stix]:
            if item in item_dict:
                return_list.append(AttckTechnique(**item_dict[item]))
        return return_list