import asyncio
import time
import logging

import pytest

from src.consensus.cost_calculator import calculate_cost_of_program, CostResult
from src.full_node.bundle_tools import best_solution_program
from src.full_node.mempool_check_conditions import (
    get_name_puzzle_conditions,
    get_puzzle_and_solution_for_coin,
)
from src.types.program import SerializedProgram
from src.util.byte_types import hexstr_to_bytes
from tests.setup_nodes import test_constants, bt
from clvm_tools import binutils

BURN_PUZZLE_HASH = b"0" * 32

log = logging.getLogger(__name__)


@pytest.fixture(scope="module")
def event_loop():
    loop = asyncio.get_event_loop()
    yield loop


class TestCostCalculation:
    @pytest.mark.asyncio
    async def test_basics(self):
        wallet_tool = bt.get_pool_wallet_tool()
        ph = wallet_tool.get_new_puzzlehash()
        num_blocks = 3
        blocks = bt.get_consecutive_blocks(
            num_blocks, [], guarantee_block=True, pool_reward_puzzle_hash=ph, farmer_reward_puzzle_hash=ph
        )
        coinbase = None
        for coin in blocks[2].get_included_reward_coins():
            if coin.puzzle_hash == ph:
                coinbase = coin
                break
        assert coinbase is not None
        spend_bundle = wallet_tool.generate_signed_transaction(
            coinbase.amount,
            BURN_PUZZLE_HASH,
            coinbase,
        )
        assert spend_bundle is not None
        program = best_solution_program(spend_bundle)

        ratio = test_constants.CLVM_COST_RATIO_CONSTANT

        result: CostResult = calculate_cost_of_program(program, ratio)
        clvm_cost = result.cost

        error, npc_list, cost = get_name_puzzle_conditions(program, False)
        assert error is None
        coin_name = npc_list[0].coin_name
        error, puzzle, solution = get_puzzle_and_solution_for_coin(program, coin_name)
        assert error is None

        # Create condition + agg_sig_condition + length + cpu_cost
        assert clvm_cost == 200 * ratio + 20 * ratio + len(bytes(program)) * ratio + cost

    @pytest.mark.asyncio
    async def test_strict_mode(self):
        wallet_tool = bt.get_pool_wallet_tool()
        ph = wallet_tool.get_new_puzzlehash()

        num_blocks = 3
        blocks = bt.get_consecutive_blocks(
            num_blocks, [], guarantee_block=True, pool_reward_puzzle_hash=ph, farmer_reward_puzzle_hash=ph
        )

        coinbase = None
        for coin in blocks[2].get_included_reward_coins():
            if coin.puzzle_hash == ph:
                coinbase = coin
                break
        assert coinbase is not None
        spend_bundle = wallet_tool.generate_signed_transaction(
            coinbase.amount,
            BURN_PUZZLE_HASH,
            coinbase,
        )
        assert spend_bundle is not None
        program = SerializedProgram.from_bytes(
            binutils.assemble(
                "(q ((0x3d2331635a58c0d49912bc1427d7db51afe3f20a7b4bcaffa17ee250dcbcbfaa"
                " (((c (q ((c (q ((c (i 11 (q ((c (i (= 5 (point_add 11"
                " (pubkey_for_exp (sha256 11 ((c 6 (c 2 (c 23 (q ())))))))))"
                " (q ((c 23 47))) (q (x))) 1))) (q (c (c 4 (c 5 (c ((c 6 (c 2"
                " (c 23 (q ()))))) (q ())))) ((c 23 47))))) 1))) (c (q (57 (c"
                " (i (l 5) (q (sha256 (q 2) ((c 6 (c 2 (c 9 (q ()))))) ((c 6 (c"
                " 2 (c 13 (q ()))))))) (q (sha256 (q 1) 5))) 1))) 1)))) (c"
                " (q 0x88bc9360319e7c54ab42e19e974288a2d7a817976f7633f4b43"
                "f36ce72074e59c4ab8ddac362202f3e366f0aebbb6280)"
                ' 1))) (() (q ((65 "00000000000000000000000000000000" 0x0cbba106e000))) ())))))'
            ).as_bin()
        )
        error, npc_list, cost = get_name_puzzle_conditions(program, True)
        assert error is not None
        error, npc_list, cost = get_name_puzzle_conditions(program, False)
        assert error is None

        coin_name = npc_list[0].coin_name
        error, puzzle, solution = get_puzzle_and_solution_for_coin(program, coin_name)
        assert error is None

    async def test_clvm_strict_mode(self):
        program = SerializedProgram.from_bytes(
            binutils.assemble(
                "(i (a (q 0xfe) (q ())) (q ()) (q ((51 "
                "0xe3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855 200))))"
            ).as_bin()
        )
        error, npc_list, cost = get_name_puzzle_conditions(program, True)
        assert error is not None
        error, npc_list, cost = get_name_puzzle_conditions(program, False)
        assert error is None

    @pytest.mark.asyncio
    async def test_tx_generator_speed(self):
        hex_str = "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"  # noqa: E501
        generator = hexstr_to_bytes(hex_str)
        program = SerializedProgram.from_bytes(generator)

        start_time = time.time()
        err, npc, cost = get_name_puzzle_conditions(program, False)
        end_time = time.time()
        duration = end_time - start_time
        assert err is None
        assert len(npc) == 687
        log.info(f"Time spent: {duration}")

        assert duration < 3

    @pytest.mark.asyncio
    async def test_standard_tx(self):

        puzzle = "((c (q ((c (q ((c (i 11 (q ((c (i (= 5 (point_add 11 (pubkey_for_exp (sha256 11 ((c 6 (c 2 (c 23 (q ()))))))))) (q ((c 23 47))) (q (x))) 1))) (q (c (c 4 (c 5 (c ((c 6 (c 2 (c 23 (q ()))))) (q ())))) ((c 23 47))))) 1))) (c (q (57 (c (i (l 5) (q (sha256 (q 2) ((c 6 (c 2 (c 9 (q ()))))) ((c 6 (c 2 (c 13 (q ()))))))) (q (sha256 (q 1) 5))) 1))) 1)))) (c (q 0xaf949b78fa6a957602c3593a3d6cb7711e08720415dad831ab18adacaa9b27ec3dda508ee32e24bc811c0abc5781ae21) 1)))"  # noqa: E501

        solution = "(() (q ((51 0x699eca24f2b6f4b25b16f7a418d0dc4fc5fce3b9145aecdda184158927738e3e 10) (51 0x847bb2385534070c39a39cc5dfdc7b35e2db472dc0ab10ab4dec157a2178adbf 0x00cbba106df6))) ())"  # noqa: E501
        time_start = time.time()
        total_cost = 0
        puzzle_program = SerializedProgram.from_bytes(binutils.assemble(puzzle).as_bin())
        solution_program = SerializedProgram.from_bytes(binutils.assemble(solution).as_bin())
        for i in range(0, 1000):
            cost, result = puzzle_program.run_with_cost(solution_program)
            total_cost += cost

        time_end = time.time()
        duration = time_end - time_start

        log.info(f"Time spent: {duration}")
        assert duration < 3
