# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ormar',
 'ormar.decorators',
 'ormar.fields',
 'ormar.models',
 'ormar.models.descriptors',
 'ormar.models.helpers',
 'ormar.models.mixins',
 'ormar.protocols',
 'ormar.queryset',
 'ormar.queryset.actions',
 'ormar.relations',
 'ormar.signals']

package_data = \
{'': ['*']}

install_requires = \
['SQLAlchemy>=1.3.18,<=1.4.29',
 'databases>=0.3.2,!=0.5.0,!=0.5.1,!=0.5.2,!=0.5.3,<0.5.5',
 'pydantic>=1.6.1,!=1.7,!=1.7.1,!=1.7.2,!=1.7.3,!=1.8,!=1.8.1,<=1.9.1']

extras_require = \
{':python_version < "3.8"': ['typing-extensions>=3.7,<4.0',
                             'importlib-metadata>=3.1'],
 'crypto': ['cryptography>=35,<37'],
 'dev': ['asyncpg>=0.24,<0.26',
         'psycopg2-binary>=2.9.1,<3.0.0',
         'aiomysql>=0.0.21,<0.0.23',
         'cryptography>=35,<37'],
 'dev:python_version >= "3.7"': ['orjson>=3.6.4,<4.0.0'],
 'mysql': ['aiomysql>=0.0.21,<0.0.23'],
 'orjson:python_version >= "3.7"': ['orjson>=3.6.4,<4.0.0'],
 'postgres': ['asyncpg>=0.24,<0.26', 'psycopg2-binary>=2.9.1,<3.0.0'],
 'postgresql': ['asyncpg>=0.24,<0.26', 'psycopg2-binary>=2.9.1,<3.0.0']}

setup_kwargs = {
    'name': 'ormar',
    'version': '0.10.24',
    'description': 'A simple async ORM with fastapi in mind and pydantic validation.',
    'long_description': '# ormar\n<p>\n<a href="https://pypi.org/project/ormar">\n    <img src="https://img.shields.io/pypi/v/ormar.svg" alt="Pypi version">\n</a>\n<a href="https://pypi.org/project/ormar">\n    <img src="https://img.shields.io/pypi/pyversions/ormar.svg" alt="Pypi version">\n</a>\n<img src="https://github.com/collerek/ormar/workflows/build/badge.svg" alt="Build Status">\n<a href="https://codecov.io/gh/collerek/ormar">\n    <img src="https://codecov.io/gh/collerek/ormar/branch/master/graph/badge.svg" alt="Coverage">\n</a>\n<a href="https://www.codefactor.io/repository/github/collerek/ormar">\n<img src="https://www.codefactor.io/repository/github/collerek/ormar/badge" alt="CodeFactor" />\n</a>\n<a href="https://codeclimate.com/github/collerek/ormar/maintainability">\n<img src="https://api.codeclimate.com/v1/badges/186bc79245724864a7aa/maintainability" /></a>\n<a href="https://pepy.tech/project/ormar">\n<img src="https://pepy.tech/badge/ormar"></a>\n</p>\n\n### Overview\n\nThe `ormar` package is an async mini ORM for Python, with support for **Postgres,\nMySQL**, and **SQLite**.\n\nThe main benefits of using `ormar` are:\n\n*  getting an **async ORM that can be used with async frameworks** (fastapi, starlette etc.)\n*  getting just **one model to maintain** - you don\'t have to maintain pydantic and other orm models (sqlalchemy, peewee, gino etc.)\n\nThe goal was to create a simple ORM that can be **used directly (as request and response models) with [`fastapi`][fastapi]** that bases it\'s data validation on pydantic.\n\nOrmar - apart from the obvious "ORM" in name - gets its name from _ormar_ in Swedish which means _snakes_, and _ormar_ in Croatian which means _cabinet_.\n\nAnd what\'s a better name for python ORM than snakes cabinet :)\n\n**If you like ormar remember to star the repository in [github](https://github.com/collerek/ormar)!**\n\nThe bigger community we build, the easier it will be to catch bugs and attract contributors ;)\n\n### Documentation\n\nCheck out the [documentation][documentation] for details.\n\n**Note that for brevity most of the documentation snippets omit the creation of the database\nand scheduling the execution of functions for asynchronous run.**\n\nIf you want more real life examples than in the documentation you can see the [tests][tests] folder,\nsince they actually have to create and connect to a database in most of the tests.\n\nYet remember that those are - well - tests and not all solutions are suitable to be used in real life applications.\n\n### Part of the `fastapi` ecosystem\n\nAs part of the fastapi ecosystem `ormar` is supported in libraries that somehow work with databases.\n\nAs of now `ormar` is supported by:\n\n*  [`fastapi-users`](https://github.com/frankie567/fastapi-users)\n*  [`fastapi-crudrouter`](https://github.com/awtkns/fastapi-crudrouter)\n*  [`fastapi-pagination`](https://github.com/uriyyo/fastapi-pagination)\n\nIf you maintain or use a different library and would like it to support `ormar` let us know how we can help.\n\n### Dependencies\n\nOrmar is built with:\n\n* [`sqlalchemy core`][sqlalchemy-core] for query building.\n* [`databases`][databases] for cross-database async support.\n* [`pydantic`][pydantic] for data validation.\n* `typing_extensions` for python 3.6 - 3.7\n\n### License\n\n`ormar` is built as open-sorce software and will remain completely free (MIT license).\n\nAs I write open-source code to solve everyday problems in my work or to promote and build strong python\ncommunity you can say thank you and buy me a coffee or sponsor me with a monthly amount to help ensure my work remains free and maintained.\n\n<a aria-label="Sponsor collerek" href="https://github.com/sponsors/collerek" style="text-decoration: none; color: #c9d1d9 !important;">\n<div style="\n    background-color: #21262d;\n    border-color: #30363d;\n    box-shadow:  0 0 transparent, 0 0 transparent;\n    color: #c9d1d9 !important;\n    border: 1px solid;\n    border-radius: 6px;\n    cursor: pointer;\n    display: inline-block;\n    font-size: 14px;\n    padding: 10px;\n    line-height: 0px;\n    height: 40px;\n">\n<span style="color: #c9d1d9 !important;">Sponsor - Github Sponsors</span>\n</div>\n</a>\n\n### Migrating from `sqlalchemy` and existing databases\n\nIf you currently use `sqlalchemy` and would like to switch to `ormar` check out the auto-translation\ntool that can help you with translating existing sqlalchemy orm models so you do not have to do it manually.\n\n**Beta** versions available at github: [`sqlalchemy-to-ormar`](https://github.com/collerek/sqlalchemy-to-ormar)\nor simply `pip install sqlalchemy-to-ormar`\n\n`sqlalchemy-to-ormar` can be used in pair with `sqlacodegen` to auto-map/ generate `ormar` models from existing database, even if you don\'t use `sqlalchemy` for your project.\n\n### Migrations & Database creation\n\nBecause ormar is built on SQLAlchemy core, you can use [`alembic`][alembic] to provide\ndatabase migrations (and you really should for production code).\n\nFor tests and basic applications the `sqlalchemy` is more than enough:\n```python\n# note this is just a partial snippet full working example below\n# 1. Imports\nimport sqlalchemy\nimport databases\n\n# 2. Initialization\nDATABASE_URL = "sqlite:///db.sqlite"\ndatabase = databases.Database(DATABASE_URL)\nmetadata = sqlalchemy.MetaData()\n\n# Define models here\n\n# 3. Database creation and tables creation\nengine = sqlalchemy.create_engine(DATABASE_URL)\nmetadata.create_all(engine)\n```\n\nFor a sample configuration of alembic and more information regarding migrations and\ndatabase creation visit [migrations][migrations] documentation section.\n\n### Package versions\n**ormar is still under development:**\nWe recommend pinning any dependencies (with i.e. `ormar~=0.9.1`)\n\n`ormar` also follows the release numeration that breaking changes bump the major number,\nwhile other changes and fixes bump minor number, so with the latter you should be safe to\nupdate, yet always read the [releases][releases] docs before.\n`example: (0.5.2 -> 0.6.0 - breaking, 0.5.2 -> 0.5.3 - non breaking)`.\n\n### Asynchronous Python\n\nNote that `ormar` is an asynchronous ORM, which means that you have to `await` the calls to\nthe methods, that are scheduled for execution in an event loop. Python has a builtin module\n[`asyncio`][asyncio] that allows you to do just that.\n\nNote that most "normal" python interpreters do not allow execution of `await`\noutside of a function (because you actually schedule this function for delayed execution\nand don\'t get the result immediately).\n\nIn a modern web framework (like `fastapi`), the framework will handle this for you, but if\nyou plan to do this on your own you need to perform this manually like described in the\nquick start below.\n\n### Quick Start\n\nNote that you can find the same script in examples folder on github.\n\n```python\nfrom typing import Optional\n\nimport databases\nimport pydantic\n\nimport ormar\nimport sqlalchemy\n\nDATABASE_URL = "sqlite:///db.sqlite"\ndatabase = databases.Database(DATABASE_URL)\nmetadata = sqlalchemy.MetaData()\n\n\n# note that this step is optional -> all ormar cares is a internal\n# class with name Meta and proper parameters, but this way you do not\n# have to repeat the same parameters if you use only one database\nclass BaseMeta(ormar.ModelMeta):\n    metadata = metadata\n    database = database\n\n\n# Note that all type hints are optional\n# below is a perfectly valid model declaration\n# class Author(ormar.Model):\n#     class Meta(BaseMeta):\n#         tablename = "authors"\n#\n#     id = ormar.Integer(primary_key=True) # <= notice no field types\n#     name = ormar.String(max_length=100)\n\n\nclass Author(ormar.Model):\n    class Meta(BaseMeta):\n        tablename = "authors"\n\n    id: int = ormar.Integer(primary_key=True)\n    name: str = ormar.String(max_length=100)\n\n\nclass Book(ormar.Model):\n    class Meta(BaseMeta):\n        tablename = "books"\n\n    id: int = ormar.Integer(primary_key=True)\n    author: Optional[Author] = ormar.ForeignKey(Author)\n    title: str = ormar.String(max_length=100)\n    year: int = ormar.Integer(nullable=True)\n\n\n# create the database\n# note that in production you should use migrations\n# note that this is not required if you connect to existing database\nengine = sqlalchemy.create_engine(DATABASE_URL)\n# just to be sure we clear the db before\nmetadata.drop_all(engine)\nmetadata.create_all(engine)\n\n\n# all functions below are divided into functionality categories\n# note how all functions are defined with async - hence can use await AND needs to\n# be awaited on their own\nasync def create():\n    # Create some records to work with through QuerySet.create method.\n    # Note that queryset is exposed on each Model\'s class as objects\n    tolkien = await Author.objects.create(name="J.R.R. Tolkien")\n    await Book.objects.create(author=tolkien, title="The Hobbit", year=1937)\n    await Book.objects.create(author=tolkien, title="The Lord of the Rings", year=1955)\n    await Book.objects.create(author=tolkien, title="The Silmarillion", year=1977)\n\n    # alternative creation of object divided into 2 steps\n    sapkowski = Author(name="Andrzej Sapkowski")\n    # do some stuff\n    await sapkowski.save()\n\n    # or save() after initialization\n    await Book(author=sapkowski, title="The Witcher", year=1990).save()\n    await Book(author=sapkowski, title="The Tower of Fools", year=2002).save()\n\n    # to read more about inserting data into the database\n    # visit: https://collerek.github.io/ormar/queries/create/\n\n\nasync def read():\n    # Fetch an instance, without loading a foreign key relationship on it.\n    # Django style\n    book = await Book.objects.get(title="The Hobbit")\n    # or python style\n    book = await Book.objects.get(Book.title == "The Hobbit")\n    book2 = await Book.objects.first()\n\n    # first() fetch the instance with lower primary key value\n    assert book == book2\n\n    # you can access all fields on loaded model\n    assert book.title == "The Hobbit"\n    assert book.year == 1937\n\n    # when no condition is passed to get()\n    # it behaves as last() based on primary key column\n    book3 = await Book.objects.get()\n    assert book3.title == "The Tower of Fools"\n\n    # When you have a relation, ormar always defines a related model for you\n    # even when all you loaded is a foreign key value like in this example\n    assert isinstance(book.author, Author)\n    # primary key is populated from foreign key stored in books table\n    assert book.author.pk == 1\n    # since the related model was not loaded all other fields are None\n    assert book.author.name is None\n\n    # Load the relationship from the database when you already have the related model\n    # alternatively see joins section below\n    await book.author.load()\n    assert book.author.name == "J.R.R. Tolkien"\n\n    # get all rows for given model\n    authors = await Author.objects.all()\n    assert len(authors) == 2\n\n    # to read more about reading data from the database\n    # visit: https://collerek.github.io/ormar/queries/read/\n\n\nasync def update():\n    # read existing row from db\n    tolkien = await Author.objects.get(name="J.R.R. Tolkien")\n    assert tolkien.name == "J.R.R. Tolkien"\n    tolkien_id = tolkien.id\n\n    # change the selected property\n    tolkien.name = "John Ronald Reuel Tolkien"\n    # call update on a model instance\n    await tolkien.update()\n\n    # confirm that object was updated\n    tolkien = await Author.objects.get(name="John Ronald Reuel Tolkien")\n    assert tolkien.name == "John Ronald Reuel Tolkien"\n    assert tolkien.id == tolkien_id\n\n    # alternatively update data without loading\n    await Author.objects.filter(name__contains="Tolkien").update(name="J.R.R. Tolkien")\n\n    # to read more about updating data in the database\n    # visit: https://collerek.github.io/ormar/queries/update/\n\n\nasync def delete():\n    silmarillion = await Book.objects.get(year=1977)\n    # call delete() on instance\n    await silmarillion.delete()\n\n    # alternatively delete without loading\n    await Book.objects.delete(title="The Tower of Fools")\n\n    # note that when there is no record ormar raises NoMatch exception\n    try:\n        await Book.objects.get(year=1977)\n    except ormar.NoMatch:\n        print("No book from 1977!")\n\n    # to read more about deleting data from the database\n    # visit: https://collerek.github.io/ormar/queries/delete/\n\n    # note that despite the fact that record no longer exists in database\n    # the object above is still accessible and you can use it (and i.e. save()) again.\n    tolkien = silmarillion.author\n    await Book.objects.create(author=tolkien, title="The Silmarillion", year=1977)\n\n\nasync def joins():\n    # Tho join two models use select_related\n    \n    # Django style\n    book = await Book.objects.select_related("author").get(title="The Hobbit")\n    # Python style\n    book = await Book.objects.select_related(Book.author).get(\n        Book.title == "The Hobbit"\n    )\n\n    # now the author is already prefetched\n    assert book.author.name == "J.R.R. Tolkien"\n\n    # By default you also get a second side of the relation\n    # constructed as lowercase source model name +\'s\' (books in this case)\n    # you can also provide custom name with parameter related_name\n    \n    # Django style\n    author = await Author.objects.select_related("books").all(name="J.R.R. Tolkien")\n    # Python style\n    author = await Author.objects.select_related(Author.books).all(\n        Author.name == "J.R.R. Tolkien"\n    )\n    assert len(author[0].books) == 3\n\n    # for reverse and many to many relations you can also prefetch_related\n    # that executes a separate query for each of related models\n\n    # Django style\n    author = await Author.objects.prefetch_related("books").get(name="J.R.R. Tolkien")\n    # Python style\n    author = await Author.objects.prefetch_related(Author.books).get(\n        Author.name == "J.R.R. Tolkien"\n    )\n    assert len(author.books) == 3\n\n    # to read more about relations\n    # visit: https://collerek.github.io/ormar/relations/\n\n    # to read more about joins and subqueries\n    # visit: https://collerek.github.io/ormar/queries/joins-and-subqueries/\n\n\nasync def filter_and_sort():\n    # to filter the query you can use filter() or pass key-value pars to\n    # get(), all() etc.\n    # to use special methods or access related model fields use double\n    # underscore like to filter by the name of the author use author__name\n    # Django style\n    books = await Book.objects.all(author__name="J.R.R. Tolkien")\n    # python style\n    books = await Book.objects.all(Book.author.name == "J.R.R. Tolkien")\n    assert len(books) == 3\n\n    # filter can accept special methods also separated with double underscore\n    # to issue sql query ` where authors.name like "%tolkien%"` that is not\n    # case sensitive (hence small t in Tolkien)\n    # Django style\n    books = await Book.objects.filter(author__name__icontains="tolkien").all()\n    # python style\n    books = await Book.objects.filter(Book.author.name.icontains("tolkien")).all()\n    assert len(books) == 3\n\n    # to sort use order_by() function of queryset\n    # to sort decreasing use hyphen before the field name\n    # same as with filter you can use double underscores to access related fields\n    # Django style\n    books = (\n        await Book.objects.filter(author__name__icontains="tolkien")\n        .order_by("-year")\n        .all()\n    )\n    # python style\n    books = (\n        await Book.objects.filter(Book.author.name.icontains("tolkien"))\n        .order_by(Book.year.desc())\n        .all()\n    )\n    assert len(books) == 3\n    assert books[0].title == "The Silmarillion"\n    assert books[2].title == "The Hobbit"\n\n    # to read more about filtering and ordering\n    # visit: https://collerek.github.io/ormar/queries/filter-and-sort/\n\n\nasync def subset_of_columns():\n    # to exclude some columns from loading when querying the database\n    # you can use fileds() method\n    hobbit = await Book.objects.fields(["title"]).get(title="The Hobbit")\n    # note that fields not included in fields are empty (set to None)\n    assert hobbit.year is None\n    assert hobbit.author is None\n\n    # selected field is there\n    assert hobbit.title == "The Hobbit"\n\n    # alternatively you can provide columns you want to exclude\n    hobbit = await Book.objects.exclude_fields(["year"]).get(title="The Hobbit")\n    # year is still not set\n    assert hobbit.year is None\n    # but author is back\n    assert hobbit.author is not None\n\n    # also you cannot exclude primary key column - it\'s always there\n    # even if you EXPLICITLY exclude it it will be there\n\n    # note that each model have a shortcut for primary_key column which is pk\n    # and you can filter/access/set the values by this alias like below\n    assert hobbit.pk is not None\n\n    # note that you cannot exclude fields that are not nullable\n    # (required) in model definition\n    try:\n        await Book.objects.exclude_fields(["title"]).get(title="The Hobbit")\n    except pydantic.ValidationError:\n        print("Cannot exclude non nullable field title")\n\n    # to read more about selecting subset of columns\n    # visit: https://collerek.github.io/ormar/queries/select-columns/\n\n\nasync def pagination():\n    # to limit number of returned rows use limit()\n    books = await Book.objects.limit(1).all()\n    assert len(books) == 1\n    assert books[0].title == "The Hobbit"\n\n    # to offset number of returned rows use offset()\n    books = await Book.objects.limit(1).offset(1).all()\n    assert len(books) == 1\n    assert books[0].title == "The Lord of the Rings"\n\n    # alternatively use paginate that combines both\n    books = await Book.objects.paginate(page=2, page_size=2).all()\n    assert len(books) == 2\n    # note that we removed one book of Sapkowski in delete()\n    # and recreated The Silmarillion - by default when no order_by is set\n    # ordering sorts by primary_key column\n    assert books[0].title == "The Witcher"\n    assert books[1].title == "The Silmarillion"\n\n    # to read more about pagination and number of rows\n    # visit: https://collerek.github.io/ormar/queries/pagination-and-rows-number/\n\n\nasync def aggregations():\n    # count:\n    assert 2 == await Author.objects.count()\n\n    # exists\n    assert await Book.objects.filter(title="The Hobbit").exists()\n\n    # maximum\n    assert 1990 == await Book.objects.max(columns=["year"])\n\n    # minimum\n    assert 1937 == await Book.objects.min(columns=["year"])\n\n    # average\n    assert 1964.75 == await Book.objects.avg(columns=["year"])\n\n    # sum\n    assert 7859 == await Book.objects.sum(columns=["year"])\n\n    # to read more about aggregated functions\n    # visit: https://collerek.github.io/ormar/queries/aggregations/\n\n\nasync def raw_data():\n    # extract raw data in a form of dicts or tuples\n    # note that this skips the validation(!) as models are\n    # not created from parsed data\n\n    # get list of objects as dicts\n    assert await Book.objects.values() == [\n        {"id": 1, "author": 1, "title": "The Hobbit", "year": 1937},\n        {"id": 2, "author": 1, "title": "The Lord of the Rings", "year": 1955},\n        {"id": 4, "author": 2, "title": "The Witcher", "year": 1990},\n        {"id": 5, "author": 1, "title": "The Silmarillion", "year": 1977},\n    ]\n\n    # get list of objects as tuples\n    assert await Book.objects.values_list() == [\n        (1, 1, "The Hobbit", 1937),\n        (2, 1, "The Lord of the Rings", 1955),\n        (4, 2, "The Witcher", 1990),\n        (5, 1, "The Silmarillion", 1977),\n    ]\n\n    # filter data - note how you always get a list\n    assert await Book.objects.filter(title="The Hobbit").values() == [\n        {"id": 1, "author": 1, "title": "The Hobbit", "year": 1937}\n    ]\n\n    # select only wanted fields\n    assert await Book.objects.filter(title="The Hobbit").values(["id", "title"]) == [\n        {"id": 1, "title": "The Hobbit"}\n    ]\n\n    # if you select only one column you could flatten it with values_list\n    assert await Book.objects.values_list("title", flatten=True) == [\n        "The Hobbit",\n        "The Lord of the Rings",\n        "The Witcher",\n        "The Silmarillion",\n    ]\n\n    # to read more about extracting raw values\n    # visit: https://collerek.github.io/ormar/queries/aggregations/\n\n\nasync def with_connect(function):\n    # note that for any other backend than sqlite you actually need to\n    # connect to the database to perform db operations\n    async with database:\n        await function()\n\n    # note that if you use framework like `fastapi` you shouldn\'t connect\n    # in your endpoints but have a global connection pool\n    # check https://collerek.github.io/ormar/fastapi/ and section with db connection\n\n\n# gather and execute all functions\n# note - normally import should be at the beginning of the file\nimport asyncio\n\n# note that normally you use gather() function to run several functions\n# concurrently but we actually modify the data and we rely on the order of functions\nfor func in [\n    create,\n    read,\n    update,\n    delete,\n    joins,\n    filter_and_sort,\n    subset_of_columns,\n    pagination,\n    aggregations,\n    raw_data,\n]:\n    print(f"Executing: {func.__name__}")\n    asyncio.run(with_connect(func))\n\n# drop the database tables\nmetadata.drop_all(engine)\n```\n\n## Ormar Specification\n\n### QuerySet methods\n\n*  `create(**kwargs): -> Model`\n*  `get(*args, **kwargs): -> Model`\n*  `get_or_none(*args, **kwargs): -> Optional[Model]`\n*  `get_or_create(*args, **kwargs) -> Model`\n*  `first(*args, **kwargs): -> Model`\n*  `update(each: bool = False, **kwargs) -> int`\n*  `update_or_create(**kwargs) -> Model`\n*  `bulk_create(objects: List[Model]) -> None`\n*  `bulk_update(objects: List[Model], columns: List[str] = None) -> None`\n*  `delete(*args, each: bool = False, **kwargs) -> int`\n*  `all(*args, **kwargs) -> List[Optional[Model]]`\n*  `filter(*args, **kwargs) -> QuerySet`\n*  `exclude(*args, **kwargs) -> QuerySet`\n*  `select_related(related: Union[List, str]) -> QuerySet`\n*  `prefetch_related(related: Union[List, str]) -> QuerySet`\n*  `limit(limit_count: int) -> QuerySet`\n*  `offset(offset: int) -> QuerySet`\n*  `count() -> int`\n*  `exists() -> bool`\n*  `max(columns: List[str]) -> Any`\n*  `min(columns: List[str]) -> Any`\n*  `avg(columns: List[str]) -> Any`\n*  `sum(columns: List[str]) -> Any`\n*  `fields(columns: Union[List, str, set, dict]) -> QuerySet`\n*  `exclude_fields(columns: Union[List, str, set, dict]) -> QuerySet`\n*  `order_by(columns:Union[List, str]) -> QuerySet`\n*  `values(fields: Union[List, str, Set, Dict])`\n*  `values_list(fields: Union[List, str, Set, Dict])`\n\n\n#### Relation types\n\n*  One to many  - with `ForeignKey(to: Model)`\n*  Many to many - with `ManyToMany(to: Model, Optional[through]: Model)`\n\n#### Model fields types\n\nAvailable Model Fields (with required args - optional ones in docs):\n\n* `String(max_length)`\n* `Text()`\n* `Boolean()`\n* `Integer()`\n* `Float()`\n* `Date()`\n* `Time()`\n* `DateTime()`\n* `JSON()`\n* `BigInteger()`\n* `SmallInteger()`\n* `Decimal(scale, precision)`\n* `UUID()`\n* `LargeBinary(max_length)`\n* `EnumField` - by passing `choices` to any other Field type\n* `EncryptedString` - by passing `encrypt_secret` and `encrypt_backend`\n* `ForeignKey(to)`\n* `ManyToMany(to, through)`\n\n### Available fields options\nThe following keyword arguments are supported on all field types.\n\n* `primary_key: bool`\n* `nullable: bool`\n* `default: Any`\n* `server_default: Any`\n* `index: bool`\n* `unique: bool`\n* `choices: typing.Sequence`\n* `name: str`\n* `pydantic_only: bool`\n\nAll fields are required unless one of the following is set:\n\n* `nullable` - Creates a nullable column. Sets the default to `False`. Read the fields common parameters for details.\n* `sql_nullable` - Used to set different setting for pydantic and the database. Sets the default to `nullable` value. Read the fields common parameters for details.\n* `default` - Set a default value for the field. **Not available for relation fields**\n* `server_default` - Set a default value for the field on server side (like sqlalchemy\'s `func.now()`). **Not available for relation fields**\n* `primary key` with `autoincrement` - When a column is set to primary key and autoincrement is set on this column.\n  Autoincrement is set by default on int primary keys.\n* `pydantic_only` - Field is available only as normal pydantic field, not stored in the database.\n\n### Available signals\n\nSignals allow to trigger your function for a given event on a given Model.\n\n* `pre_save`\n* `post_save`\n* `pre_update`\n* `post_update`\n* `pre_delete`\n* `post_delete`\n* `pre_relation_add`\n* `post_relation_add`\n* `pre_relation_remove`\n* `post_relation_remove`\n* `post_bulk_update`\n\n\n[sqlalchemy-core]: https://docs.sqlalchemy.org/en/latest/core/\n[databases]: https://github.com/encode/databases\n[pydantic]: https://pydantic-docs.helpmanual.io/\n[encode/orm]: https://github.com/encode/orm/\n[alembic]: https://alembic.sqlalchemy.org/en/latest/\n[fastapi]: https://fastapi.tiangolo.com/\n[documentation]: https://collerek.github.io/ormar/\n[migrations]: https://collerek.github.io/ormar/models/migrations/\n[asyncio]: https://docs.python.org/3/library/asyncio.html\n[releases]: https://collerek.github.io/ormar/releases/\n[tests]: https://github.com/collerek/ormar/tree/master/tests\n',
    'author': 'Radosław Drążkiewicz',
    'author_email': 'collerek@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/collerek/ormar',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
