"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancingv2_1 = require("monocdk/aws-elasticloadbalancingv2");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * ALBs are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const type = nag_rules_1.NagRules.resolveIfPrimitive(node, node.type);
        if (type === undefined || type === 'application') {
            const loadBalancerLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            let found = false;
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                    if (isMatchingWebACLAssociation(child, loadBalancerLogicalId)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Load Balancer
 * @param node the CfnWebACLAssociation to check
 * @param loadBalancerLogicalId the Cfn Logical ID of the Load Balancer
 * returns whether the CfnWebACLAssociation is associates with the given Load Balancer
 */
function isMatchingWebACLAssociation(node, loadBalancerLogicalId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${loadBalancerLogicalId}(?![\\w])`, 'gm').test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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