"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053R5Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const dms_1 = require("../rules/dms");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for NIST 800-53 rev 5 compliance.
 * Based on the NIST 800-53 rev 5 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_5.html
 */
class NIST80053R5Checks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'NIST.800.53.R5';
    }
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkElasticBeanstalk(node);
            this.checkELB(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'AWS WAF enables you to configure a set of rules (called a web access control list (web ACL)) that allow, block, or count web requests based on customizable web security rules and conditions that you define. Ensure your Amazon API Gateway stage is associated with a WAF Web ACL to protect it from malicious attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, CM-6a, CM-9b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: AC-3, AC-4(21), CM-6a, SC-7(3)).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-4(1), AU-6(1), AU-6(3), AU-6(4), AU-6(5), AU-6(6), AU-6(9), AU-7(1), AU-8b, AU-9(7), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), AU-16, CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have encryption enabled - (Control IDs: AU-9(3), CM-6a, CM-9b, CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control IDs: AU-9a, CM-6a, CM-9b, PM-11b, PM-17b, SA-1(1), SA-10(1), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-4d, SI-7a, SI-7(1), SI-7(3), SI-7(7)).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AU-6(1), AU-6(5), AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a, SI-4(12), SI-5b, SI-5(1)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: AC-16b, AT-4b, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-10, AU-11(1), AU-11, AU-12(1), AU-12(2), AU-12(3), AU-14a, AU-14b, CA-7b, PM-14a.1, PM-14b, PM-21b, PM-31, SC-28(2), SI-4(17), SI-12).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            info: "The provisioned capacity DynamoDB table does not have Auto Scaling enabled on it's indexes - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).",
            explanation: 'Amazon DynamoDB auto scaling uses the AWS Application Auto Scaling service to adjust provisioned throughput capacity that automatically responds to actual traffic patterns. This enables a table or a global secondary index to increase its provisioned read/write capacity to handle sudden increases in traffic, without throttling.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBAutoScalingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon DynamoDB tables are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-1(2), CP-2(5), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EBS volume is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic Block Store (Amazon EBS) volumes are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: "The EC2 instance type 'supports' EBS optimization and does not have EBS optimization enabled - (Control IDs: CP-2(5), CP-9a, CP-9b, CP-9c, CP-10, SC-5(2)).",
            explanation: 'An optimized instance in Amazon Elastic Block Store (Amazon EBS) provides additional, dedicated capacity for Amazon EBS I/O operations. This optimization provides the most efficient performance for your EBS volumes by minimizing contention between Amazon EBS I/O operations and other traffic from your instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSOptimizedInstance,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: AC-3, CM-5(1)(a), CM-6a).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-2(6), AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-2a, CM-2(2), CM-6a, CM-7b, CM-8(6), CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-9b, SC-7a, SC-7c, SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: AC-3, AC-5b, CM-5(1)(a)).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic File System (Amazon EFS) file systems are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node) {
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6, SC-36(1)(a), SI-2a).',
            explanation: 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkEnhancedHealthReportingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Elastic Beanstalk environment does not have managed updates enabled - (Control IDs: SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control ID: AC-4(21)).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SC-23(5), SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: "Enable cross-zone load balancing for your Classic Load Balancers (CLBs) to help maintain adequate capacity and availability. The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone. It also improves your application's ability to handle the loss of one or more instances.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-2a, CM-2(2), CM-3a, CM-8(6), CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22).',
            explanation: 'This rule ensures that Elastic Load Balancing has deletion protection enabled. Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, IA-5(1)(c), PM-17b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1a.2, SI-1a.2, SI-1c.2, SI-1c.2).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: SC-8(1), SC-23(5)).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-5b, AC-6, AC-6(2), AC-6(3), AC-6(10), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants full access - (Control IDs: AC-3, AC-5b, AC-6(2), AC-6(10), CM-5(1)(a)).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2i.2, AC-2(1), AC-2(6), AC-3, AC-3(3)(a), AC-3(3)(b)(1), AC-3(3)(b)(2), AC-3(3)(b)(3), AC-3(3)(b)(4), AC-3(3)(b)(5), AC-3(3)(c), AC-3(3), AC-3(4)(a), AC-3(4)(b), AC-3(4)(c), AC-3(4)(d), AC-3(4)(e), AC-3(4), AC-3(7), AC-3(8), AC-3(12)(a), AC-3(13), AC-3(15)(a), AC-3(15)(b), AC-4(28), AC-6, AC-6(3), AC-24, CM-5(1)(a), CM-6a, CM-9b, MP-2, SC-23(3), SC-25).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control IDs: CM-6a, CM-9b, SA-9(6), SC-12, SC-12(2), SC-12(6)).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not configured with function-level concurrent execution limits - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-6).',
            explanation: "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaConcurrency,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not configured with a dead-letter configuration - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaDLQ,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-25).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control ID: AU-10).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchErrorLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: 'The RDS DB Instance does not enhanced monitoring enabled - (Control IDs: AU-12(3), AU-14a, AU-14b, CA-2(2), CA-7, CA-7b, PM-14a.1, PM-14b, PM-31, SC-36(1)(a), SI-2a).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance is not in an AWS Backup plan - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Relational Database Service (Amazon RDS) instances are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have backup enabled - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have deletion protection enabled - (Control IDs: CA-7(4)(c), CM-3a, CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), SA-15a.4, SC-5(2), SC-22, SI-13(5)).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have multi-AZ support - (Control IDs: CP-1a.1(b), CP-1a.2, CP-2a, CP-2a.6, CP-2a.7, CP-2d, CP-2e, CP-2(5), CP-2(6), CP-6(2), CP-10, SC-5(2), SC-6, SC-22, SC-36, SI-13(5)).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-9(3), AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CP-9d, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c), SI-19(4)).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: CM-2b, CM-2b.1, CM-2b.2, CM-2b.3, CM-3(3), CP-9a, CP-9b, CP-9c, SC-5(2), SI-2c, SI-2d, SI-2(2), SI-2(5)).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                            ',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: AC-4(21), SC-7b).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: AC-2(4), AC-3(1), AC-3(10), AC-4(26), AC-6(9), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-1(1)(c), SI-3(8)(b), SI-4(2), SI-4(17), SI-4(20), SI-7(8), SI-10(1)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CM-6a, CM-9b, CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), SC-5(2), SI-13(5)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: AU-9(3), CM-6a, CM-9b, CP-9d, CP-9(8), PM-11b, SC-8(3), SC-8(4), SC-13a, SC-16(1), SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not require requests to use SSL - (Control IDs: AC-4, AC-4(22), AC-17(2), AC-24(1), AU-9(3), CA-9b, CM-6a, CM-9b, IA-5(1)(c), PM-11b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-16(1), SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'To help protect data in transit, ensure that your Amazon Simple Storage Service (Amazon S3) buckets require requests to use Secure Socket Layer (SSL). Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: AU-9(2), CP-1(2), CP-2(5), CP-6a, CP-6(1), CP-6(2), CP-9a, CP-9b, CP-9c, CP-10, CP-10(2), PM-11b, PM-17b, SC-5(2), SC-16(1), SI-1a.2, SI-1a.2, SI-1c.2, SI-13(5)).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: AU-9(3), CP-9d, CP-9(8), SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            info: 'The secret does not have automatic rotation scheduled - (Control IDs: AC-4, AC-4(22), AC-24(1), AU-9(3), CA-9b, PM-17b, SC-7(4)(b), SC-7(4)(g), SC-8, SC-8(1), SC-8(2), SC-8(3), SC-8(4), SC-8(5), SC-13a, SC-23, SI-1a.2, SI-1a.2, SI-1c.2).',
            explanation: 'Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1), SI-19(4)).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: AU-9(3), CP-9d, SC-8(3), SC-8(4), SC-13a, SC-28(1)).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4(21), AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), CM-6a, CM-9b, SC-7a, SC-7c, SC-7(5), SC-7(7), SC-7(11), SC-7(12), SC-7(16), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28)).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), CM-6a, CM-9b, IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SI-4(17), SI-7(8)).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: AC-4(21), CM-7b).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: AC-2(6), AC-3, AC-3(7), AC-4(21), AC-6, AC-17b, AC-17(1), AC-17(1), AC-17(4)(a), AC-17(9), AC-17(10), MP-2, SC-7a, SC-7b, SC-7c, SC-7(2), SC-7(3), SC-7(7), SC-7(9)(a), SC-7(11), SC-7(12), SC-7(16), SC-7(20), SC-7(21), SC-7(24)(b), SC-7(25), SC-7(26), SC-7(27), SC-7(28), SC-25).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: AC-4(26), AU-2b, AU-3a, AU-3b, AU-3c, AU-3d, AU-3e, AU-3f, AU-6(3), AU-6(4), AU-6(6), AU-6(9), AU-8b, AU-10, AU-12a, AU-12c, AU-12(1), AU-12(2), AU-12(3), AU-12(4), AU-14a, AU-14b, AU-14b, AU-14(3), CA-7b, CM-5(1)(b), IA-3(3)(b), MA-4(1)(a), PM-14a.1, PM-14b, PM-31, SC-7(9)(b), SI-4(17), SI-7(8)).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.NIST80053R5Checks = NIST80053R5Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053R5Checks[_a] = { fqn: "monocdk-nag.NIST80053R5Checks", version: "1.9.4" };
//# sourceMappingURL=data:application/json;base64,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