"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagPack = exports.NagMessageLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const fs_1 = require("fs");
const path_1 = require("path");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("./nag-rules");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
const VALIDATION_FAILURE_ID = 'CdkNagValidationFailure';
const SUPPRESSION_ID = 'CdkNagSuppression';
/**
 * The level of the message that the rule applies.
 */
var NagMessageLevel;
(function (NagMessageLevel) {
    NagMessageLevel["WARN"] = "Warning";
    NagMessageLevel["ERROR"] = "Error";
})(NagMessageLevel = exports.NagMessageLevel || (exports.NagMessageLevel = {}));
/**
 * Base class for all rule packs.
 */
class NagPack {
    constructor(props) {
        this.reportStacks = new Array();
        this.packName = '';
        this.verbose =
            props == undefined || props.verbose == undefined ? false : props.verbose;
        this.logIgnores =
            props == undefined || props.logIgnores == undefined
                ? false
                : props.logIgnores;
        this.reports =
            props == undefined || props.reports == undefined ? true : props.reports;
    }
    get readPackName() {
        return this.packName;
    }
    get readReportStacks() {
        return this.reportStacks;
    }
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    applyRule(params) {
        if (this.packName === '') {
            throw Error('The NagPack does not have a pack name, therefore the rule could not be applied. Set a packName in the NagPack constructor.');
        }
        const allIgnores = nag_suppression_helper_1.NagSuppressionHelper.getSuppressions(params.node);
        const ruleSuffix = params.ruleSuffixOverride
            ? params.ruleSuffixOverride
            : params.rule.name;
        const ruleId = `${this.packName}-${ruleSuffix}`;
        try {
            const ruleCompliance = params.rule(params.node);
            if (this.reports === true &&
                ruleCompliance === nag_rules_1.NagRuleCompliance.COMPLIANT) {
                this.writeToStackComplianceReport(params, ruleId, ruleCompliance);
            }
            else if (this.isNonCompliant(ruleCompliance)) {
                const findings = this.asFindings(ruleCompliance);
                for (const findingId of findings) {
                    const suppressionReason = this.ignoreRule(allIgnores, ruleId, findingId);
                    if (this.reports === true) {
                        this.writeToStackComplianceReport(params, ruleId, nag_rules_1.NagRuleCompliance.NON_COMPLIANT, suppressionReason);
                    }
                    if (suppressionReason) {
                        if (this.logIgnores === true) {
                            const message = this.createMessage(SUPPRESSION_ID, findingId, `${ruleId} was triggered but suppressed.`, `Provided reason: "${suppressionReason}"`);
                            monocdk_1.Annotations.of(params.node).addInfo(message);
                        }
                    }
                    else {
                        const message = this.createMessage(ruleId, findingId, params.info, params.explanation);
                        if (params.level == NagMessageLevel.ERROR) {
                            monocdk_1.Annotations.of(params.node).addError(message);
                        }
                        else if (params.level == NagMessageLevel.WARN) {
                            monocdk_1.Annotations.of(params.node).addWarning(message);
                        }
                    }
                }
            }
        }
        catch (error) {
            const reason = this.ignoreRule(allIgnores, VALIDATION_FAILURE_ID, '');
            if (this.reports === true) {
                this.writeToStackComplianceReport(params, ruleId, 'UNKNOWN', reason);
            }
            if (reason) {
                if (this.logIgnores === true) {
                    const message = this.createMessage(SUPPRESSION_ID, '', `${VALIDATION_FAILURE_ID} was triggered but suppressed.`, reason);
                    monocdk_1.Annotations.of(params.node).addInfo(message);
                }
            }
            else {
                const information = `'${ruleId}' threw an error during validation. This is generally caused by a parameter referencing an intrinsic function. For more details enable verbose logging.'`;
                const message = this.createMessage(VALIDATION_FAILURE_ID, '', information, error.message);
                monocdk_1.Annotations.of(params.node).addWarning(message);
            }
        }
    }
    /**
     * Check whether a specific rule should be ignored.
     * @param ignores The ignores listed in cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param findingId The id of the finding that is being checked.
     * @returns The reason the rule was ignored, or an empty string.
     */
    ignoreRule(ignores, ruleId, findingId) {
        for (let ignore of ignores) {
            if (nag_suppression_helper_1.NagSuppressionHelper.doesApply(ignore, ruleId, findingId)) {
                if (!ignore.appliesTo) {
                    // the rule is not granular so it always applies
                    return ignore.reason;
                }
                else {
                    return `[${findingId}] ${ignore.reason}`;
                }
            }
        }
        return '';
    }
    /**
     * The message to output to the console when a rule is triggered.
     * @param ruleId The id of the rule.
     * @param findingId The id of the finding.
     * @param info Why the rule was triggered.
     * @param explanation Why the rule exists.
     * @returns The formatted message string.
     */
    createMessage(ruleId, findingId, info, explanation) {
        let message = findingId
            ? `${ruleId}[${findingId}]: ${info}`
            : `${ruleId}: ${info}`;
        return this.verbose ? `${message} ${explanation}\n` : `${message}\n`;
    }
    /**
     * Write a line to the rule packs compliance report for the resource's Stack
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    writeToStackComplianceReport(params, ruleId, compliance, explanation = '') {
        var _b;
        const line = this.createComplianceReportLine(params, ruleId, compliance, explanation);
        let outDir = (_b = monocdk_1.App.of(params.node)) === null || _b === void 0 ? void 0 : _b.outdir;
        const stackName = params.node.stack.nested
            ? monocdk_1.Names.uniqueId(params.node.stack)
            : params.node.stack.stackName;
        const fileName = `${this.packName}-${stackName}-NagReport.csv`;
        const filePath = path_1.join(outDir ? outDir : '', fileName);
        if (!this.reportStacks.includes(fileName)) {
            this.reportStacks.push(fileName);
            fs_1.writeFileSync(filePath, 'Rule ID,Resource ID,Compliance,Exception Reason,Rule Level,Rule Info\n');
        }
        fs_1.appendFileSync(filePath, line);
    }
    /**
     * Helper function to create a line for the compliance report
     * @param params The @IApplyRule interface with rule details.
     * @param ruleId The id of the rule.
     * @param compliance The compliance status of the rule.
     * @param explanation The explanation for suppressed rules.
     */
    createComplianceReportLine(params, ruleId, compliance, explanation = '') {
        //| Rule ID | Resource ID | Compliance | Exception Reason | Rule Level | Rule Info
        const line = Array();
        line.push(ruleId);
        line.push(params.node.node.path);
        if ((compliance === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            compliance === 'UNKNOWN') &&
            explanation !== '') {
            line.push('Suppressed');
            line.push(explanation);
        }
        else {
            line.push(compliance);
            line.push('N/A');
        }
        line.push(params.level);
        line.push(params.info);
        return line.map((i) => '"' + i.replace(/"/g, '""') + '"').join(',') + '\n';
    }
    isNonCompliant(ruleResult) {
        return (ruleResult === nag_rules_1.NagRuleCompliance.NON_COMPLIANT ||
            Array.isArray(ruleResult));
    }
    asFindings(ruleResult) {
        if (Array.isArray(ruleResult)) {
            return ruleResult;
        }
        else {
            return [''];
        }
    }
}
exports.NagPack = NagPack;
_a = JSII_RTTI_SYMBOL_1;
NagPack[_a] = { fqn: "monocdk-nag.NagPack", version: "1.9.4" };
//# sourceMappingURL=data:application/json;base64,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