"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.IotToKinesisFirehoseToS3 = void 0;
const iot = require("@aws-cdk/aws-iot");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-solutions-constructs/core");
const aws_kinesisfirehose_s3_1 = require("@aws-solutions-constructs/aws-kinesisfirehose-s3");
class IotToKinesisFirehoseToS3 extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the IotToKinesisFirehoseToS3 class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        const firehoseToS3 = new aws_kinesisfirehose_s3_1.KinesisFirehoseToS3(this, 'KinesisFirehoseToS3', {
            kinesisFirehoseProps: props.kinesisFirehoseProps,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        this.kinesisFirehose = firehoseToS3.kinesisFirehose;
        this.s3Bucket = firehoseToS3.s3Bucket;
        // Setup the IAM Role for IoT Actions
        this.iotActionsRole = new iam.Role(this, 'IotActionsRole', {
            assumedBy: new iam.ServicePrincipal('iot.amazonaws.com'),
        });
        // Setup the IAM policy for IoT Actions
        const iotActionsPolicy = new iam.Policy(this, 'IotActionsPolicy', {
            statements: [new iam.PolicyStatement({
                    actions: [
                        'firehose:PutRecord'
                    ],
                    resources: [this.kinesisFirehose.attrArn]
                })
            ]
        });
        // Attach policy to role
        iotActionsPolicy.attachToRole(this.iotActionsRole);
        const defaultIotTopicProps = defaults.DefaultCfnTopicRuleProps([{
                firehose: {
                    deliveryStreamName: this.kinesisFirehose.ref,
                    roleArn: this.iotActionsRole.roleArn
                }
            }]);
        const iotTopicProps = core_2.overrideProps(defaultIotTopicProps, props.iotTopicRuleProps, true);
        // Create the IoT topic rule
        this.iotTopicRule = new iot.CfnTopicRule(this, 'IotTopic', iotTopicProps);
        this.kinesisFirehoseRole = firehoseToS3.kinesisFirehoseRole;
        this.s3LoggingBucket = firehoseToS3.s3LoggingBucket;
        this.kinesisFirehoseLogGroup = firehoseToS3.kinesisFirehoseLogGroup;
    }
}
exports.IotToKinesisFirehoseToS3 = IotToKinesisFirehoseToS3;
//# sourceMappingURL=data:application/json;base64,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