"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscalingManager = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
const DEFAULT_SSH_KEY_PATH = "/etc/gitlab-runner/ssh";
/**
 * Settings for the manager (coordinator)
 *
 *  Manager coordinates the placement of runner (job executor) instances
 */
class GitlabRunnerAutoscalingManager extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.globalConfiguration =
            props.globalConfiguration ||
                {
                    concurrent: 10,
                    checkInterval: 0,
                    logFormat: "runner",
                    logLevel: "info",
                };
        this.machineImage =
            props.machineImage ??
                aws_ec2_1.MachineImage.latestAmazonLinux({
                    generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                    edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                    virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                    storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                    cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
                });
        this.instanceType = props.instanceType ?? aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        this.keyPairName = props.keyPairName;
        this.runners = props.runners;
        this.network = props.network;
        this.cacheBucket = props.cacheBucket;
        this.runnersSecurityGroupName = props.runnersSecurityGroup.securityGroupName;
        this.role =
            props.role ||
                new aws_iam_1.Role(scope, "ManagerRole", {
                    assumedBy: new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {}),
                    managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore")],
                    inlinePolicies: {
                        Cache: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                                    Resource: [`${this.cacheBucket.bucketArn}/*`],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["s3:ListBucket"],
                                    Resource: [`${this.cacheBucket.bucketArn}`],
                                },
                            ],
                        }),
                        Runners: aws_iam_1.PolicyDocument.fromJson({
                            Version: "2012-10-17",
                            Statement: [
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                                    Resource: ["*"],
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "aws:RequestTag/Name": "*gitlab-runner-*",
                                        },
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["Name"],
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringEqualsIfExists: {
                                            "ec2:Region": `${scope.region}`,
                                        },
                                        ArnEqualsIfExists: {
                                            "ec2:Vpc": `arn:aws:ec2:${scope.region}:${scope.account}:vpc/${this.network.vpc.vpcId}`,
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:RunInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["InstanceProfile"],
                                            "ec2:InstanceType": (this.runners || []).map((runner) => {
                                                const runnersInstanceType = (this.runners && runner.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
                                                return runnersInstanceType.toString();
                                            }),
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                                    Resource: ["*"],
                                    Condition: {
                                        StringLike: {
                                            "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                        },
                                    },
                                },
                                {
                                    Effect: "Allow",
                                    Action: ["iam:PassRole"],
                                    Resource: ["*"],
                                    Condition: {
                                        "ForAllValues:StringEquals": {
                                            "aws:TagKeys": ["RunnersRole"],
                                        },
                                    },
                                },
                            ],
                        }),
                    },
                });
        this.userData = aws_ec2_1.UserData.forLinux({});
        this.userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        this.initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitPackage.yum("jq"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: this.globalConfiguration,
                        runnersConfiguration: this.runners.map((runner) => {
                            const configuration = runner.configuration;
                            return {
                                ...configuration,
                                machine: {
                                    ...configuration.machine,
                                    machineOptions: {
                                        sshKeypath: runner.keyPair ? DEFAULT_SSH_KEY_PATH : "",
                                        ...configuration.machine?.machineOptions,
                                        instanceType: runner.instanceType.toString(),
                                        ami: runner.machineImage.getImage(scope).imageId,
                                        region: scope.region,
                                        vpcId: this.network.vpc.vpcId,
                                        zone: this.network.availabilityZone.slice(-1),
                                        subnetId: this.network.subnet.subnetId,
                                        securityGroup: this.runnersSecurityGroupName,
                                        privateAddressOnly: configuration.machine?.machineOptions?.privateAddressOnly ?? this.network.hasPrivateSubnets(),
                                        usePrivateAddress: configuration.machine?.machineOptions?.usePrivateAddress ?? true,
                                        iamInstanceProfile: runner.instanceProfile.ref,
                                    },
                                },
                                cache: {
                                    s3: {
                                        serverAddress: `s3.${scope.urlSuffix}`,
                                        bucketName: `${this.cacheBucket.bucketName}`,
                                        bucketLocation: `${scope.region}`,
                                        authenticationType: "iam",
                                    },
                                },
                            };
                        }),
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                    aws_ec2_1.InitCommand.shellCommand(
                    // Download custom EC2 key pair for manager <> runner ssh connect
                    this.runners
                        .map((runner) => {
                        if (!runner.keyPair) {
                            return "";
                        }
                        runner.keyPair.grantRead(this.role);
                        const region = aws_cdk_lib_1.Stack.of(this).region;
                        const secretArn = runner.keyPair.secretArn;
                        const keyPairName = runner.configuration.machine.machineOptions.keypairName;
                        const sshKeyPath = runner.configuration.machine.machineOptions.sshKeypath ?? DEFAULT_SSH_KEY_PATH;
                        return [
                            `mkdir -p ${sshKeyPath};`,
                            `$(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text | jq -r ."${keyPairName}") > ${sshKeyPath}/${keyPairName};`,
                            `$(aws secretsmanager get-secret-value --region ${region} --secret-id ${secretArn} --query SecretString --output text | jq -r ."${keyPairName}.pub") > ${sshKeyPath}/${keyPairName}.pub;`,
                        ].join("\n");
                    })
                        .filter((s) => s.length > 0)
                        .join("\n"), {
                        key: "999-retrieve-ec2-key-pair",
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
    }
}
exports.GitlabRunnerAutoscalingManager = GitlabRunnerAutoscalingManager;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscalingManager[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscalingManager", version: "0.2.128" };
//# sourceMappingURL=data:application/json;base64,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