# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ariadne_token_auth',
 'ariadne_token_auth.api',
 'ariadne_token_auth.migrations']

package_data = \
{'': ['*']}

install_requires = \
['Django>=3.0,<4.0', 'ariadne>=0.12.0']

setup_kwargs = {
    'name': 'ariadne-token-auth',
    'version': '0.1.2',
    'description': 'A django app to implement token based authentication in Ariadne GraphQL',
    'long_description': '# ariadne-token-auth\n\nA django app to implement token based authentication in projects which use\n[ariadne](https://ariadnegraphql.org/).\n\n## Summary\n\n- [ariadne-token-auth](#ariadne-token-auth)\n  - [Summary](#summary)\n    - [Installing](#installing)\n  - [Using the package](#using-the-package)\n    - [Example Project](#example-project)\n    - [Setup](#setup)\n    - [Migrations](#migrations)\n    - [Schema](#schema)\n    - [Protecting Views](#protecting-views)\n    - [Configuration](#configuration)\n    - [Bonus](#bonus)\n  - [Contributing](#contributing)\n  - [Versioning](#versioning)\n  - [Authors](#authors)\n  - [License](#license)\n  - [Acknowledgements](#acknowledgements)\n  - [Others](#others)\n\n### Installing\n\nThe package can be downloaded from its\n[github](https://github.com/IgnisDa/ariadne-token-auth) repository.\n\n```bash\npip install git+https://github.com/IgnisDa/ariadne-token-auth.git\n```\n\nOr using [poetry](https://python-poetry.org/)\n\n```bash\npoetry add git+https://github.com/IgnisDa/ariadne-token-auth.git#main\n```\n\n## Using the package\n\n### Example Project\n\nYou can have a look at the [example project](./example_project) for a fully\nworking project. [Habitrac](https://github.com/IgnisDa/habitrac) is also a production\nwebsite which uses this package to implement authentication.\n\n### Setup\n\nInclude the `AuthTokenMiddleware` in your `MIDDLEWARE` settings.\n\n```python\nMIDDLEWARE = [\n    # Other middleware\n    "ariadne_token_auth.middleware.AuthTokenMiddleware",\n]\n```\n\nInclude the `AuthTokenBackend` in your `AUTHENTICATION_BACKENDS` settings.\n\n```python\nAUTHENTICATION_BACKENDS = (\n    "django.contrib.auth.backends.ModelBackend",\n    "ariadne_token_auth.backends.AuthTokenBackend",\n)\n```\n\nFinally add `ariadne_token_auth` to your `INSTALLED_APPS`.\n\n```python\nINSTALLED_APPS = [\n    # other apps\n    \'ariadne_token_auth\',\n]\n```\n\n### Migrations\n\nNext, run `python manage.py migrate` to commit the auth-token model to your database.\n\n### Schema\n\nAdd the relevant mutations to your GraphQL schema.\n\n```python\nfrom ariadne import MutationType, make_executable_schema\nfrom ariadne_token_auth.api import resolvers\n\nauth_mutation = MutationType()\n\nauth_mutation.set_field("getAuthToken", resolvers.get_auth_token)\nauth_mutation.set_field("deleteAuthToken", resolvers.delete_auth_token)\ntype_defs = """\n  type Mutation {\n    getAuthToken(identifier: String!, password: String!): AuthTokenPayload!\n    deleteAuthToken(token: String!): DeleteTokenPayload!\n}\n"""\n\nschema = make_executable_schema([type_defs, resolvers.type_defs], auth_mutation)\n```\n\n- `getAuthToken` to authenticate an existing user and obtain a corresponding token. The\n  resolver uses the user model\'s `USERNAME_FIELD` which by default is `username`. However\n  it will work with other `USERNAME_FIELD`s just fine, for example when the default user\n  identifier is `email` instead of `username`. The [example project](#example-project) does\n  this by defining a\n  [custom user model](https://docs.djangoproject.com/en/3.1/topics/auth/customizing/#specifying-a-custom-user-model).\n\n  ```graphql\n  mutation getAuthToken($identifier: String!, $password: String!) {\n    getAuthToken(identifier: $identifier, password: $password) {\n      error\n      auth {\n        token\n      }\n    }\n  }\n  ```\n\n  If authentication is successful, you can obtain the auth-token from\n  `response.data.getAuthToken.auth.token`, and if it is unsuccessful, errors will\n  be present in `response.data.getAuthToken.error`.\n\n- `deleteAuthToken` to delete a logged in user using the above token.\n\n  ```graphql\n  mutation deleteAuthToken($token: String!) {\n    deleteAuthToken(token: $token) {\n      status\n      error\n    }\n  }\n  ```\n\n  If the token was correct and deletion was successful, the value of\n  `response.data.deleteAuthToken.status` will be set to `true` (or it\'s equivalent in your\n  frontend language). Otherwise, the error will be present in\n  `response.data.deleteAuthToken.error` and `response.data.deleteAuthToken.status` will be\n  set to `false`.\n\n### Protecting Views\n\nYou can use the `login_required` decorator to protect your graphql queries from\nnon-authenticated users.\n\n```python\nfrom ariadne import QueryType\nfrom ariadne_token_auth.decorators import login_required\n\n@query.field("testQuery")\n@login_required\ndef test_query(self, info, *args, **kwargs):\n    return {"user": info.context.get("request").user}\n```\n\n### Configuration\n\nSettings specific to ariadne-token-auth are stored in the `ARIADNE_TOKEN_AUTH` dictionary\nin `settings.py` file. The defaults can be seen in [utils.py](./ariadne_token_auth/utils.py)\nfile. They can be configured as follows:\n\n```python\n# settings.py\nARIADNE_TOKEN_AUTH = {\n    \'TOKEN_NAME\': \'myBearer\', # case insensitive\n    \'TOKEN_LENGTH\': 15\n}\n```\n\nWith the above settings, you will have to send requests in the following fashion (example\nuses `curl`, but the basic premise stays the same).\n\n```bash\ncurl \'http://127.0.0.1:8000/graphql/\' \\\n      -H \'Content-Type: application/json\' \\\n      -H \'Authorization: myBearer 8496fda8dedad2235921693717c8dc\' \\\n      --data-binary \'{"query":"query {\\n  testQuery {\\n    user\\n  }\\n}"}\'\n```\n\n### Bonus\n\nYou can find a very easy way to add your `*.graphql` files to the django auto-reloader\n[here](https://github.com/IgnisDa/ariadne-token-auth/blob/main/example_project/example_app/apps.py#L6).\n\n## Contributing\n\nPlease read [CONTRIBUTING.md](CONTRIBUTING.md) for details on our code\nof conduct, and the process for submitting pull requests to us.\n\n## Versioning\n\nWe use [SemVer](http://semver.org/) for versioning. For the versions\navailable, see the [tags on this\nrepository](https://github.com/PurpleBooth/a-good-readme-template/tags).\n\n## Authors\n\nSee also the list of [contributors](contributors.md) who participated in this project.\n\n## License\n\nThis project is licensed under the\n[Apache-2.0](https://www.apache.org/licenses/LICENSE-2.0) - see the\n[LICENSE.md](LICENSE.md) file for details\n\n## Acknowledgements\n\n1. **django-token**: Model definitions, middleware and authentication backends adapted from\n   [django-tokens](https://github.com/jasonbeverage/django-token) package.\n2. **ariadne-jwt**: Exceptions, and decorators adapted from\n   [ariadne-jwt](https://github.com/Usama0121/ariadne-jwt) package.\n\n## Others\n\nProject bootstrapped using [cookiecutter](https://github.com/IgnisDa/project-cookiecutter)\nby IgnisDa.\n',
    'author': 'IgnisDa',
    'author_email': 'ignisda2002@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/IgnisDa/ariadne-token-auth',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
