"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * SASL authentication
     */
    static sasl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sasl);
            }
            throw error;
        }
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tls);
            }
            throw error;
        }
        return new ClientAuthentication(undefined, props);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.31.2-alpha.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (props.clientAuthentication?.saslProps?.iam &&
            props.clientAuthentication?.saslProps?.scram) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (props.encryptionInTransit?.clientBroker ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (props.clientAuthentication?.saslProps?.scram ||
                props.clientAuthentication?.saslProps?.iam)) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = props.ebsStorageInfo?.volumeSize ?? 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = props.ebsStorageInfo?.encryptionKey
            ? {
                dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
            }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: props.encryptionInTransit?.clientBroker ??
                ClientBrokerEncryption.TLS,
            inCluster: props.encryptionInTransit?.enableInCluster ?? true,
        };
        const openMonitoring = props.monitoring?.enablePrometheusJmxExporter ||
            props.monitoring?.enablePrometheusNodeExporter
            ? {
                prometheus: {
                    jmxExporter: props.monitoring?.enablePrometheusJmxExporter
                        ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: props.monitoring
                        ?.enablePrometheusNodeExporter
                        ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingBucket = props.logging?.s3?.bucket;
        if (loggingBucket && aws_cdk_lib_1.FeatureFlags.of(this).isEnabled(cx_api_1.S3_CREATE_DEFAULT_LOGGING_POLICY)) {
            const stack = core.Stack.of(this);
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [
                    loggingBucket.arnForObjects(`AWSLogs/${stack.account}/*`),
                ],
                actions: ['s3:PutObject'],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
            loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                principals: [
                    new iam.ServicePrincipal('delivery.logs.amazonaws.com'),
                ],
                resources: [loggingBucket.bucketArn],
                actions: [
                    's3:GetBucketAcl',
                    's3:ListBucket',
                ],
                conditions: {
                    StringEquals: {
                        'aws:SourceAccount': stack.account,
                    },
                    ArnLike: {
                        'aws:SourceArn': stack.formatArn({
                            service: 'logs',
                            resource: '*',
                        }),
                    },
                },
            }));
        }
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: props.logging?.cloudwatchLogGroup !== undefined,
                    logGroup: props.logging?.cloudwatchLogGroup?.logGroupName,
                },
                firehose: {
                    enabled: props.logging?.firehoseDeliveryStreamName !==
                        undefined,
                    deliveryStream: props.logging?.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: loggingBucket !== undefined,
                    bucket: loggingBucket?.bucketName,
                    prefix: props.logging?.s3?.prefix,
                },
            },
        };
        if (props.clientAuthentication?.saslProps?.scram &&
            props.clientAuthentication?.saslProps?.key === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if (props.clientAuthentication?.saslProps?.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
        }
        else if (props.clientAuthentication?.saslProps?.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (props.clientAuthentication?.tlsProps?.certificateAuthorities !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: props.clientAuthentication?.tlsProps?.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: props.monitoring?.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [this.saslScramAuthenticationKey?.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.31.2-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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