!-----------------------------------------------------------------------------
! Copyright (c) 2017-2025, Met Office, on behalf of HMSO and Queen's Printer
! For further details please refer to the file LICENCE.original which you
! should have received as part of this distribution.
!-----------------------------------------------------------------------------
!> @brief This version is for use on discontinous spaces and increments the
!>        output field, hence can be used for WTheta spaces and incrementing
!>        W3 fields

! Tweaked this kernel to allow it to run though PSyAD by adding
! artificial contains and procedure to the metadata as PSyAD currently
! can not process multi-precision metadata.

module dg_inc_matrix_vector_kernel_mod
  use argument_mod,            only : arg_type,                  &
                                      GH_FIELD, GH_OPERATOR,     &
                                      GH_REAL, GH_READ,          &
                                      GH_READWRITE, ANY_SPACE_1, &
                                      ANY_DISCONTINUOUS_SPACE_1, &
                                      CELL_COLUMN
  use constants_mod,           only : i_def, r_single, r_double
  use kernel_mod,              only : kernel_type

  implicit none

  private

  !-------------------------------------------------------------------------------
  ! Public types
  !-------------------------------------------------------------------------------
  type, public, extends(kernel_type) :: dg_inc_matrix_vector_kernel_type
    private
    type(arg_type) :: meta_args(3) = (/                                                  &
         arg_type(GH_FIELD,    GH_REAL, GH_READWRITE, ANY_DISCONTINUOUS_SPACE_1),        &
         arg_type(GH_FIELD,    GH_REAL, GH_READ, ANY_SPACE_1),                           &
         arg_type(GH_OPERATOR, GH_REAL, GH_READ, ANY_DISCONTINUOUS_SPACE_1, ANY_SPACE_1) &
         /)
    integer :: operates_on = CELL_COLUMN
  contains
    procedure, nopass :: dg_inc_matrix_vector_code_r_single
  end type

  !-------------------------------------------------------------------------------
  ! Contained functions/subroutines
  !-------------------------------------------------------------------------------
  public :: dg_inc_matrix_vector_code

  ! Generic interface for real32 and real64 types
  interface dg_inc_matrix_vector_code
    module procedure  &
      dg_inc_matrix_vector_code_r_single, &
      dg_inc_matrix_vector_code_r_double
  end interface

contains

  !> @brief Computes lhs = matrix*x for discontinuous function spaces
  !> @brief real32 and real64 variants
  !! @param[in] cell Horizontal cell index
  !! @param[in] nlayers Number of layers
  !! @param[in,out] lhs Output lhs (A*x)
  !! @param[in] x Input data
  !! @param[in] ncell_3d Total number of cells
  !! @param[in] matrix Local matrix assembly form of the operator A
  !! @param[in] ndf1 Number of degrees of freedom per cell for the output field
  !! @param[in] undf1 Unique number of degrees of freedom  for the output field
  !! @param[in] map1 Dofmap for the cell at the base of the column for the
  !! output field
  !! @param[in] ndf2 Number of degrees of freedom per cell for the input field
  !! @param[in] undf2 Unique number of degrees of freedom for the input field
  !! @param[in] map2 Dofmap for the cell at the base of the column for the input
  !! field

  ! R_SINGLE PRECISION
  ! ==================
  subroutine dg_inc_matrix_vector_code_r_single(cell,              &
                                                nlayers,           &
                                                lhs, x,            &
                                                ncell_3d,          &
                                                matrix,            &
                                                ndf1, undf1, map1, &
                                                ndf2, undf2, map2)

    implicit none

    ! Arguments
    integer(kind=i_def),                  intent(in) :: cell, nlayers, ncell_3d
    integer(kind=i_def),                  intent(in) :: undf1, ndf1
    integer(kind=i_def),                  intent(in) :: undf2, ndf2
    integer(kind=i_def), dimension(ndf1), intent(in) :: map1
    integer(kind=i_def), dimension(ndf2), intent(in) :: map2
    real(kind=r_single), dimension(undf2),              intent(in)    :: x
    real(kind=r_single), dimension(undf1),              intent(inout) :: lhs
    real(kind=r_single), dimension(ncell_3d,ndf1,ndf2), intent(in)    :: matrix

    ! Internal variables
    integer(kind=i_def) :: df, df2, k, ik

    do df = 1, ndf1
      do df2 = 1, ndf2
        do k = 0, nlayers-1
          ik = (cell-1)*nlayers + k + 1
          lhs(map1(df)+k) = lhs(map1(df)+k) + matrix(ik,df,df2)*x(map2(df2)+k)
        end do
      end do
    end do

  end subroutine dg_inc_matrix_vector_code_r_single


  ! R_DOUBLE PRECISION
  ! ==================
  subroutine dg_inc_matrix_vector_code_r_double(cell,              &
                                                nlayers,           &
                                                lhs, x,            &
                                                ncell_3d,          &
                                                matrix,            &
                                                ndf1, undf1, map1, &
                                                ndf2, undf2, map2)

    implicit none

    ! Arguments
    integer(kind=i_def),                  intent(in) :: cell, nlayers, ncell_3d
    integer(kind=i_def),                  intent(in) :: undf1, ndf1
    integer(kind=i_def),                  intent(in) :: undf2, ndf2
    integer(kind=i_def), dimension(ndf1), intent(in) :: map1
    integer(kind=i_def), dimension(ndf2), intent(in) :: map2
    real(kind=r_double), dimension(undf2),              intent(in)    :: x
    real(kind=r_double), dimension(undf1),              intent(inout) :: lhs
    real(kind=r_double), dimension(ncell_3d,ndf1,ndf2), intent(in)    :: matrix

    ! Internal variables
    integer(kind=i_def) :: df, df2, k, ik

    do df = 1, ndf1
      do df2 = 1, ndf2
        do k = 0, nlayers-1
          ik = (cell-1)*nlayers + k + 1
          lhs(map1(df)+k) = lhs(map1(df)+k) + matrix(ik,df,df2)*x(map2(df2)+k)
        end do
      end do
    end do

  end subroutine dg_inc_matrix_vector_code_r_double

end module dg_inc_matrix_vector_kernel_mod
