"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const runner_configuration_1 = require("../runner-configuration");
const cache_1 = require("./cache");
const network_1 = require("./network");
/**
 * The Gitlab Runner autoscaling on EC2 by Docker Machine.
 *
 * @stability stable
 * @example
 *
 * <caption>Provisioning a basic Runner</caption>
 * const app = new cdk.App();
 * const stack = new cdk.Stack(app, "RunnerStack", {
 * env: {
 * account: "000000000000",
 * region: "us-east-1",
 * }
 * });
 *
 * new GitlabRunnerAutoscaling(scope, "GitlabRunner", {
 * gitlabToken: "xxxxxxxxxxxxxxxxxxxx",
 * });
 */
class GitlabRunnerAutoscaling extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        const { manager, cache, runners, network } = props;
        /**
         * S3 Bucket for Runners' cache
         */
        this.cacheBucket = (cache === null || cache === void 0 ? void 0 : cache.bucket) || new cache_1.Cache(scope, "Cache", cache === null || cache === void 0 ? void 0 : cache.options).bucket;
        /**
         * Network
         */
        this.network = new network_1.Network(scope, "Network", network);
        /**
         * IAM
         */
        const ec2ServicePrincipal = new aws_iam_1.ServicePrincipal("ec2.amazonaws.com", {});
        const ec2ManagedPolicyForSSM = aws_iam_1.ManagedPolicy.fromManagedPolicyArn(scope, "AmazonEC2RoleforSSM", "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM");
        /**
         * GitLab Runners
         */
        const runnersSecurityGroupName = `${scope.stackName}-RunnersSecurityGroup`;
        const runnersSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "RunnersSecurityGroup", {
            securityGroupName: runnersSecurityGroupName,
            description: "Security group for GitLab Runners.",
            vpc: this.network.vpc,
        });
        const runnersRole = (runners === null || runners === void 0 ? void 0 : runners.role) ||
            new aws_iam_1.Role(scope, "RunnersRole", {
                assumedBy: ec2ServicePrincipal,
                managedPolicies: [ec2ManagedPolicyForSSM],
            });
        const runnersInstanceProfile = new aws_iam_1.CfnInstanceProfile(scope, "RunnersInstanceProfile", {
            roles: [runnersRole.roleName],
        });
        const runnersInstanceType = (runners === null || runners === void 0 ? void 0 : runners.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
        const runnersLookupMachineImage = new aws_ec2_1.LookupMachineImage({
            name: "ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64-server-*",
            owners: ["099720109477"],
            filters: {
                architecture: ["x86_64"],
                "image-type": ["machine"],
                state: ["available"],
                "root-device-type": ["ebs"],
                "virtualization-type": ["hvm"],
            },
        }).getImage(this);
        const runnersMachineImage = (runners === null || runners === void 0 ? void 0 : runners.machineImage) ||
            aws_ec2_1.MachineImage.genericLinux({
                [scope.region]: runnersLookupMachineImage.imageId,
            });
        /**
         * GitLab Manager
         */
        const managerSecurityGroup = new aws_ec2_1.SecurityGroup(scope, "ManagerSecurityGroup", {
            vpc: this.network.vpc,
            description: "Security group for GitLab Runners Manager.",
        });
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(22), "SSH traffic from Manager");
        managerSecurityGroup.connections.allowTo(runnersSecurityGroup, aws_ec2_1.Port.tcp(2376), "SSH traffic from Docker");
        const managerInstanceType = (manager === null || manager === void 0 ? void 0 : manager.instanceType) || aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.NANO);
        const managerMachineImage = (manager === null || manager === void 0 ? void 0 : manager.machineImage) ||
            aws_ec2_1.MachineImage.latestAmazonLinux({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                edition: aws_ec2_1.AmazonLinuxEdition.STANDARD,
                virtualization: aws_ec2_1.AmazonLinuxVirt.HVM,
                storage: aws_ec2_1.AmazonLinuxStorage.EBS,
                cpuType: aws_ec2_1.AmazonLinuxCpuType.X86_64,
            });
        const managerRole = new aws_iam_1.Role(scope, "ManagerRole", {
            assumedBy: ec2ServicePrincipal,
            managedPolicies: [ec2ManagedPolicyForSSM],
            inlinePolicies: {
                Cache: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["s3:ListObjects*", "s3:GetObject*", "s3:DeleteObject*", "s3:PutObject*"],
                            Resource: [`${this.cacheBucket.bucketArn}/*`],
                        },
                        {
                            Effect: "Allow",
                            Action: ["s3:ListBucket"],
                            Resource: [`${this.cacheBucket.bucketArn}`],
                        },
                    ],
                }),
                Runners: aws_iam_1.PolicyDocument.fromJson({
                    Version: "2012-10-17",
                    Statement: [
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateKeyPair", "ec2:DeleteKeyPair", "ec2:ImportKeyPair", "ec2:Describe*"],
                            Resource: ["*"],
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:CreateTags", "ssm:UpdateInstanceInformation"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "aws:RequestTag/Name": "*gitlab-runner-*",
                                },
                                "ForAllValues:StringEquals": {
                                    "aws:TagKeys": ["Name"],
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RequestSpotInstances", "ec2:CancelSpotInstanceRequests"],
                            Resource: ["*"],
                            Condition: {
                                StringEqualsIfExists: {
                                    "ec2:Region": `${scope.region}`,
                                },
                                ArnEqualsIfExists: {
                                    "ec2:Vpc": `${this.network.vpc.vpcArn}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:RunInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringEquals: {
                                    "ec2:InstanceType": [`${runnersInstanceType.toString()}`],
                                    "ec2:InstanceProfile": `${runnersInstanceProfile.ref}`,
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["ec2:TerminateInstances", "ec2:StopInstances", "ec2:StartInstances", "ec2:RebootInstances"],
                            Resource: ["*"],
                            Condition: {
                                StringLike: {
                                    "ec2:ResourceTag/Name": "*gitlab-runner-*",
                                },
                            },
                        },
                        {
                            Effect: "Allow",
                            Action: ["iam:PassRole"],
                            Resource: [`${runnersRole.roleArn}`],
                        },
                    ],
                }),
            },
        });
        const userData = aws_ec2_1.UserData.forLinux({});
        userData.addCommands(`yum update -y aws-cfn-bootstrap` // !/bin/bash -xe
        );
        const gitlabRunnerConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        gitlabRunnerConfigRestartHandle._addFile("/etc/gitlab-runner/config.toml");
        const rsyslogConfigRestartHandle = new aws_ec2_1.InitServiceRestartHandle();
        rsyslogConfigRestartHandle._addFile("/etc/rsyslog.d/25-gitlab-runner.conf");
        /**
         * Config set keys
         * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html#aws-resource-init-configsets
         */
        const REPOSITORIES = "repositories";
        const PACKAGES = "packages";
        const CONFIG = "config";
        const RESTART = "restart";
        const initConfig = aws_ec2_1.CloudFormationInit.fromConfigSets({
            configSets: {
                default: [REPOSITORIES, PACKAGES, CONFIG, RESTART],
            },
            configs: {
                [REPOSITORIES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://packages.gitlab.com/install/repositories/runner/gitlab-runner/script.rpm.sh | bash", { key: "10-gitlab-runner" }),
                ]),
                [PACKAGES]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitPackage.yum("docker"),
                    aws_ec2_1.InitPackage.yum("gitlab-runner"),
                    aws_ec2_1.InitPackage.yum("tzdata"),
                    aws_ec2_1.InitCommand.shellCommand("curl -L https://gitlab-docker-machine-downloads.s3.amazonaws.com/v0.16.2-gitlab.12/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine", 
                    //"curl -L https://github.com/docker/machine/releases/download/v0.16.2/docker-machine-`uname -s`-`uname -m` > /tmp/docker-machine && install /tmp/docker-machine /usr/bin/docker-machine",
                    { key: "10-docker-machine" }),
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner start", {
                        key: "20-gitlab-runner-start",
                    }),
                ]),
                [CONFIG]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitFile.fromString("/etc/gitlab-runner/config.toml", runner_configuration_1.ConfigurationMapper.withDefaults({
                        globalConfiguration: {
                            concurrent: (_b = props === null || props === void 0 ? void 0 : props.concurrent) !== null && _b !== void 0 ? _b : 10,
                            checkInterval: (_c = props === null || props === void 0 ? void 0 : props.checkInterval) !== null && _c !== void 0 ? _c : 0,
                            logFormat: (props === null || props === void 0 ? void 0 : props.logFormat) || "runner",
                            logLevel: (props === null || props === void 0 ? void 0 : props.logLevel) || "info",
                        },
                        runnerConfiguration: {
                            token: props.gitlabToken,
                            url: props.gitlabUrl || "https://gitlab.com",
                            limit: (_d = runners === null || runners === void 0 ? void 0 : runners.limit) !== null && _d !== void 0 ? _d : 10,
                            outputLimit: (_e = runners === null || runners === void 0 ? void 0 : runners.outputLimit) !== null && _e !== void 0 ? _e : 52428800,
                            environment: (runners === null || runners === void 0 ? void 0 : runners.environment) || ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                        },
                        dockerConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.docker,
                        },
                        machineConfiguration: {
                            ...runners === null || runners === void 0 ? void 0 : runners.machine,
                            machineOptions: {
                                ...(_f = runners === null || runners === void 0 ? void 0 : runners.machine) === null || _f === void 0 ? void 0 : _f.machineOptions,
                                instanceType: runnersInstanceType.toString(),
                                ami: runnersMachineImage.getImage(scope).imageId,
                                region: scope.region,
                                vpcId: this.network.vpc.vpcId,
                                zone: this.network.availabilityZone.slice(-1),
                                subnetId: this.network.subnet.subnetId,
                                securityGroup: `${runnersSecurityGroupName}`,
                                usePrivateAddress: true,
                                iamInstanceProfile: `${runnersInstanceProfile.ref}`,
                            },
                        },
                        autoscalingConfigurations: (runners === null || runners === void 0 ? void 0 : runners.autoscaling) || [],
                        cacheConfiguration: {
                            s3: {
                                serverAddress: `s3.${scope.urlSuffix}`,
                                bucketName: `${this.cacheBucket.bucketName}`,
                                bucketLocation: `${scope.region}`,
                            },
                        },
                    }).toToml(), {
                        owner: "gitlab-runner",
                        group: "gitlab-runner",
                        mode: "000600",
                    }),
                    aws_ec2_1.InitFile.fromString("/etc/rsyslog.d/25-gitlab-runner.conf", `:programname, isequal, "gitlab-runner" /var/log/gitlab-runner.log`, {
                        owner: "root",
                        group: "root",
                        mode: "000644",
                    }),
                    aws_ec2_1.InitService.enable("gitlab-runner", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: gitlabRunnerConfigRestartHandle,
                    }),
                    aws_ec2_1.InitService.enable("rsyslog", {
                        ensureRunning: true,
                        enabled: true,
                        serviceRestartHandle: rsyslogConfigRestartHandle,
                    }),
                ]),
                [RESTART]: new aws_ec2_1.InitConfig([
                    aws_ec2_1.InitCommand.shellCommand("gitlab-runner restart", {
                        key: "10-gitlab-runner-restart",
                    }),
                ]),
            },
        });
        const managerAutoScalingGroup = new aws_autoscaling_1.AutoScalingGroup(scope, "ManagerAutoscalingGroup", {
            vpc: this.network.vpc,
            vpcSubnets: {
                subnets: [this.network.subnet],
            },
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            keyName: manager === null || manager === void 0 ? void 0 : manager.keyPairName,
            securityGroup: managerSecurityGroup,
            role: managerRole,
            userData: userData,
            init: initConfig,
            initOptions: {
                ignoreFailures: false,
            },
            maxCapacity: 1,
            minCapacity: 1,
            desiredCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForCount(1, { timeout: aws_cdk_lib_1.Duration.minutes(15) }),
        });
        this.runners = {
            securityGroupName: runnersSecurityGroupName,
            securityGroup: runnersSecurityGroup,
            role: runnersRole,
            instanceProfile: runnersInstanceProfile,
            instanceType: runnersInstanceType,
            machineImage: runnersMachineImage,
        };
        this.manager = {
            securityGroup: managerSecurityGroup,
            instanceType: managerInstanceType,
            machineImage: managerMachineImage,
            autoScalingGroup: managerAutoScalingGroup,
            role: managerRole,
        };
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "@pepperize/cdk-autoscaling-gitlab-runner.GitlabRunnerAutoscaling", version: "0.0.160" };
//# sourceMappingURL=data:application/json;base64,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