"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateFastAutoscaler = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const sfn_tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const path = require("path");
const AWSCLI_LAYER_ARN = 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-awscli';
const AWSCLI_LAYER_VERSION = '1.16.281';
class FargateFastAutoscaler extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.vpc = props.vpc;
        this.region = cdk.Stack.of(this).region;
        // create a security group that allows all traffic from the same sg
        const sg = new ec2.SecurityGroup(this, 'SharedSecurityGroup', {
            vpc: this.vpc,
        });
        sg.connections.allowFrom(sg, ec2.Port.allTraffic());
        //sg for HTTP public access
        const httpPublicSecurityGroup = new ec2.SecurityGroup(this, 'HttpPublicSecurityGroup', {
            allowAllOutbound: true,
            securityGroupName: 'HttpPublicSecurityGroup',
            vpc: this.vpc,
        });
        httpPublicSecurityGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
        // // Fargate Cluster
        const fgCluster = new ecs.Cluster(this, 'fgCluster', {
            vpc: this.vpc,
        });
        // // task iam role
        const taskIAMRole = new iam.Role(this, 'fgDemoTaskExecutionRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        taskIAMRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['xray:PutTraceSegments'],
        }));
        // ECS task definition
        const demoTaskDef = new ecs.FargateTaskDefinition(this, 'cdk-fargate-demo-taskdef', {
            cpu: 256,
            memoryLimitMiB: 512,
            taskRole: taskIAMRole,
        });
        this.fargateTaskDef = demoTaskDef;
        const mainContainer = demoTaskDef.addContainer('main', {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../nginx')),
            cpu: 0,
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'echo-http-req',
            }),
        });
        const backendContainer = demoTaskDef.addContainer('backend', props.backendContainer);
        // const phpContainer = demoTaskDef.addContainer('backend', {
        //   image: ecs.ContainerImage.fromAsset('./php', {}),
        //   cpu: 0,
        //   logging: new ecs.AwsLogDriver({
        //     streamPrefix: 'echo-http-req'
        //   })
        // })
        // mainContainer.addLink(phpContainer, 'app')
        mainContainer.addPortMappings({
            containerPort: 80,
        });
        backendContainer.addPortMappings(...props.backendContainerPortMapping);
        // phpContainer.addPortMappings({
        //   containerPort: 2015
        // })
        const demoService = new ecs.FargateService(this, 'demo-service', {
            cluster: fgCluster,
            desiredCount: (_a = props.initialTaskNumber) !== null && _a !== void 0 ? _a : 2,
            taskDefinition: demoTaskDef,
            securityGroup: sg,
        });
        this.fargateService = demoService;
        const externalLB = new elbv2.ApplicationLoadBalancer(this, 'external', {
            vpc: this.vpc,
            internetFacing: true,
            securityGroup: httpPublicSecurityGroup,
        });
        const externalListener = externalLB.addListener('PublicListener', {
            port: 80,
        });
        const healthCheckDefault = {
            port: 'traffic-port',
            path: '/',
            intervalSecs: 30,
            timeoutSeconds: 5,
            healthyThresholdCount: 5,
            unhealthyThresholdCount: 2,
            healthyHttpCodes: '200,301,302',
        };
        externalListener.addTargets('fg-echo-req', {
            port: 80,
            protocol: elbv2.ApplicationProtocol.HTTP,
            healthCheck: healthCheckDefault,
            targets: [demoService],
            deregistrationDelay: cdk.Duration.seconds(3),
        });
        const lambdaRole = new iam.Role(this, 'lambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonECS_FullAccess'));
        lambdaRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ReadOnlyAccess'));
        lambdaRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DeleteNetworkInterface',
            ],
        }));
        lambdaRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
        }));
        // const uniqueId = crypto.createHash('md5').update(this.node.path).digest("hex");
        cdk.Stack.of(this).templateOptions.transforms = ['AWS::Serverless-2016-10-31']; // required for AWS::Serverless
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'AWS::Serverless::Application',
            properties: {
                Location: {
                    ApplicationId: (_b = props.awsCliLayerArn) !== null && _b !== void 0 ? _b : AWSCLI_LAYER_ARN,
                    SemanticVersion: (_c = props.awsCliLayerVersion) !== null && _c !== void 0 ? _c : AWSCLI_LAYER_VERSION,
                },
                Parameters: {},
            },
        });
        this.layerVersionArn = cdk.Token.asString(resource.getAtt('Outputs.LayerVersionArn'));
        const fargateWatcherFunc = new lambda.Function(this, 'fargateWatcherFunc', {
            runtime: lambda.Runtime.PROVIDED,
            handler: 'main',
            code: lambda.Code.fromAsset(path.join(__dirname, '../sam/fargateWatcherFunc/func.d')),
            layers: [lambda.LayerVersion.fromLayerVersionArn(this, 'AwsCliLayer', this.layerVersionArn)],
            memorySize: 1024,
            timeout: cdk.Duration.minutes(1),
            role: lambdaRole,
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE },
            securityGroup: sg,
            environment: {
                cluster: fgCluster.clusterName,
                service: demoService.serviceName,
                disable_scalein: props.disableScaleIn === false ? 'no' : 'yes',
                region: this.region,
            },
        });
        this.fargateWatcherFuncArn = fargateWatcherFunc.functionArn;
        // step function
        const wait3 = new sfn.Wait(this, 'Wait 3 Seconds', {
            // time: sfn.WaitTime.secondsPath('$.wait_time') 
            time: sfn.WaitTime.duration(cdk.Duration.seconds(3)),
        });
        const wait60 = new sfn.Wait(this, 'Wait 60 Seconds', {
            time: sfn.WaitTime.duration(cdk.Duration.seconds(60)),
        });
        const getEcsTasks = new sfn.Task(this, 'GetECSTasks', {
            task: new sfn_tasks.InvokeFunction(lambda.Function.fromFunctionArn(this, 'getEcsTasks', fargateWatcherFunc.functionArn)),
            resultPath: '$.status',
        });
        const topic = (_d = props.snsTopic) !== null && _d !== void 0 ? _d : new sns.Topic(this, `${id}-topic`, {
            topicName: `${cdk.Stack.of(this).stackName}-${id}`,
        });
        const snsScaleOut = new sfn.Task(this, 'SNSScaleOut', {
            task: new sfn_tasks.PublishToTopic(topic, {
                // message: sfn.TaskInput.fromDataAt('$'),
                message: sfn.TaskInput.fromObject({
                    'Input.$': '$',
                }),
                subject: 'Fargate Start Scaling Out',
            }),
            resultPath: '$.taskresult',
        });
        const svcScaleOut = new sfn.Task(this, 'ServiceScaleOut', {
            task: new sfn_tasks.InvokeFunction(lambda.Function.fromFunctionArn(this, 'svcScaleOut', fargateWatcherFunc.functionArn)),
        });
        const isServiceOverloaded = new sfn.Choice(this, 'IsServiceOverloaded', {
            inputPath: '$.status',
        });
        const isDone = new sfn.Pass(this, 'Done');
        const desire2 = new sfn.Pass(this, 'Desire2', {
            outputPath: '$',
            result: sfn.Result.fromObject({ Desired: 2 }),
        });
        // const desire5 = new sfn.Pass(this, 'Desire5', {
        //     outputPath: DISCARD,
        //     result: sfn.Result.fromObject({Desired: 5})
        // })
        const desire10 = new sfn.Pass(this, 'Desire10', {
            outputPath: '$',
            result: sfn.Result.fromObject({ Desired: 10 }),
        });
        const desire15 = new sfn.Pass(this, 'Desire15', {
            outputPath: '$',
            result: sfn.Result.fromObject({ Desired: 15 }),
        });
        const desire20 = new sfn.Pass(this, 'Desire20', {
            outputPath: '$',
            result: sfn.Result.fromObject({ Desired: 20 }),
        });
        const chain = sfn.Chain
            .start(getEcsTasks)
            .next(isServiceOverloaded
            .when(sfn.Condition.numberGreaterThanEquals('$.avg', 500), desire20
            .next(snsScaleOut
            .next(svcScaleOut
            .next(wait60
            .next(getEcsTasks)))))
            .when(sfn.Condition.numberGreaterThanEquals('$.avg', 300), desire15
            .next(snsScaleOut))
            .when(sfn.Condition.numberGreaterThanEquals('$.avg', 100), desire10
            .next(snsScaleOut))
            .when(sfn.Condition.numberGreaterThanEquals('$.avg', 50), desire2
            .next(snsScaleOut))
            // .when(sfn.Condition.numberLessThanEquals('$.avg', 10), desire2
            //     .next(snsScaleOut
            // ))
            .when(sfn.Condition.numberLessThan('$.avg', 0), isDone)
            .otherwise(wait3
            .next(getEcsTasks)));
        new sfn.StateMachine(this, 'FargateFastAutoscaler', {
            definition: chain,
            timeout: cdk.Duration.hours(24),
        });
        new cdk.CfnOutput(this, 'ClusterARN: ', { value: fgCluster.clusterArn });
        new cdk.CfnOutput(this, 'URL: ', { value: 'http://' + externalListener.loadBalancer.loadBalancerDnsName });
        new cdk.CfnOutput(this, 'FargateWatcherLambdaArn: ', { value: fargateWatcherFunc.functionArn });
    }
}
exports.FargateFastAutoscaler = FargateFastAutoscaler;
//# sourceMappingURL=data:application/json;base64,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