# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['GWP100_CH4', 'get_emission_data_paths', 'calc_temp_increase', 'calc_gwpstar_emissions', 'read_data',
           'make_gwps', 'make_gwps_improved']

# Cell
from pathlib import Path

import numpy as np
import pandas as pd

from fair.forward import fair_scm
from fair.RCPs import rcp26, rcp45, rcp6, rcp85

# Cell
GWP100_CH4 = 28

# Cell
def get_emission_data_paths(data_path=None):
    """Get paths for emisson at file.

    :param: data_path - path to files, defaults to the `gwp/data`

    :return: dict with scenario names as keys and paths to
              data files as values
    """
    if not data_path:
        import igwp
        lib_base = Path(igwp.__file__).absolute().parent
        data_path = lib_base / 'data'
    rcp_scenarios = {
        'RCP 2.6': 'RCP3PD_EMISSIONS.csv',
        'RCP 4.5': 'RCP45_EMISSIONS.csv',
        'RCP 6': 'RCP6_EMISSIONS.csv',
        'RCP 8.5': 'RCP85_EMISSIONS.csv',
    }

    for name, file_name in rcp_scenarios.items():
        rcp_scenarios[name] = data_path / file_name
    return rcp_scenarios

# Cell
def calc_temp_increase():
    """Calculate temperature increase caused by Methane only

    This uses FaiR: https://github.com/OMS-NetZero/FAIR

    Returns a pandas DataFrame with columns RCP 2.6, RCP 4.5, RCP 6, RCP 8.5
    and rows with yearly data from 1765 to 2500.
    """
    forcings = {
        'RCP 2.6': rcp26.Forcing.forcing,
        'RCP 4.5': rcp45.Forcing.forcing,
        'RCP 6': rcp6.Forcing.forcing,
        'RCP 8.5': rcp85.Forcing.forcing
    }

    methane_pos = 9
    temps = {}
    for name, forcing in forcings.items():
        noco2 = np.zeros_like(forcing[:, methane_pos])
        *_, temp = fair_scm(useMultigas=False, emissions=noco2,
                            other_rf=forcing[:, methane_pos] * 1.65)
        temps[name] = temp
    temperature_increases = pd.DataFrame(temps)
    temperature_increases.index = forcing[:, 0]
    return temperature_increases

# Cell
def calc_gwpstar_emissions(emissons, gwp_h=GWP100_CH4, h=100, delt=20):
    """
    Calculation of GWP* from GWP

    Based on Allen et. al 2018

    :param: emissons pandas Series with emission values, index is year
    :param: gwp_h GWP for `h` years
    :param: h time in years
    :param: delt delta tine in years
    :return: GWP*
    """
    emissons_0 = emissons.iloc[:-delt]
    emissons_1 = emissons.iloc[delt:]
    emissons_0.index = emissons_1.index  # assume pandas Series
    return ((emissons_1 - emissons_0) / delt) * gwp_h * h

# Cell
def read_data(file_name, sep = ',', skiprows=36, index_col=0, species='CH4',
              unit_converter=0.001):
    """Read emssion data from a csv file.

    Per default reads Methane data from a file with the format of the
    files in `igwp/data`.
    These files are from FaiR: https://github.com/OMS-NetZero/FAIR

    """
    data = pd.read_csv(file_name, sep=sep, skiprows=skiprows, index_col=index_col)
    return data[species] * unit_converter

# Cell
def make_gwps(file_name, start_year=1765, end_year=2500, base_year=1765, delt=20):
    """Compute GWP_100, GWP*_100.
    """
    emissons = read_data(file_name)
    start_year = max(1765, start_year - delt)
    start_index = start_year - base_year
    end_index = end_year - base_year + 1
    e_co2_eq = (emissons * GWP100_CH4)[start_index:end_index]
    e_co2_eq_star = calc_gwpstar_emissions(emissons)[start_index:end_index]
    e_co2_eq.name = '$CH_4$ GWP'
    e_co2_eq_star.name = '$CH_4$ GWP*'
    df = pd.DataFrame([e_co2_eq, e_co2_eq_star]).T
    df.index.name = 'Years'
    return df

# Cell
def make_gwps_improved(file_name,
                       additional_emssions=0,
                       start_year=1765,
                       end_year=2500,
                       base_year=1765,
                       delt=20,
                       r=0.75,
                       s=0.25):
    """Compute GWP_100, GWP*_100, and IGWP_100.
    """


    emissons = read_data(file_name)
    if isinstance(additional_emssions, pd.Series):
        additional_emssions = additional_emssions.reindex(emissons.index, fill_value=0)
    emissons += additional_emssions
    start_year = max(1765, start_year - delt)
    start_index = start_year - base_year
    end_index = end_year - base_year + 1
    e_co2_eq = (emissons * GWP100_CH4)[start_index:end_index]
    e_co2_eq_star = calc_gwpstar_emissions(emissons)[start_index:end_index]
    # here's the meat
    e_co2_eq_star_mod = r * e_co2_eq_star + s * e_co2_eq
    e_co2_eq.name = '$CH_4$ GWP'
    e_co2_eq_star.name = '$CH_4$ GWP*'
    e_co2_eq_star_mod.name = '$CH_4$ IGWP'

    df = pd.DataFrame([e_co2_eq, e_co2_eq_star, e_co2_eq_star_mod]).T
    df.index.name = 'Years'
    return df