"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const toml_1 = require("@iarna/toml");
const param_case_1 = require("param-case");
const pascal_case_1 = require("pascal-case");
const snake_case_1 = require("snake-case");
/**
 * @stability stable
 */
class ConfigurationMapper {
    constructor(props) {
        this.props = props;
    }
    /**
     * @stability stable
     */
    static withDefaults(props) {
        const { globalConfiguration, runnerConfiguration, dockerConfiguration, machineConfiguration, autoscalingConfigurations, cacheConfiguration, } = props;
        return new ConfigurationMapper({
            globalConfiguration: {
                concurrent: 10,
                checkInterval: 0,
                logFormat: "runner",
                logLevel: "info",
                ...globalConfiguration,
            },
            runnerConfiguration: {
                name: "gitlab-runner",
                url: "https://gitlab.com",
                limit: 10,
                outputLimit: 52428800,
                executor: "docker+machine",
                environment: ["DOCKER_DRIVER=overlay2", "DOCKER_TLS_CERTDIR=/certs"],
                ...runnerConfiguration,
            },
            dockerConfiguration: {
                tlsVerify: false,
                image: "docker:19.03.5",
                privileged: true,
                capAdd: ["CAP_SYS_ADMIN"],
                waitForServicesTimeout: 300,
                disableCache: false,
                volumes: ["/certs/client", "/cache"],
                shmSize: 0,
                ...dockerConfiguration,
            },
            machineConfiguration: {
                idleCount: 0,
                idleTime: 300,
                maxBuilds: 20,
                machineDriver: "amazonec2",
                machineName: "gitlab-runner-%",
                ...machineConfiguration,
                machineOptions: {
                    requestSpotInstance: true,
                    spotPrice: 0.03,
                    ...machineConfiguration.machineOptions,
                },
            },
            autoscalingConfigurations: autoscalingConfigurations.length
                ? autoscalingConfigurations
                : [
                    {
                        periods: ["* * 7-22 * * mon-fri *"],
                        idleCount: 1,
                        idleTime: 1800,
                        timezone: "Etc/UTC",
                    },
                ],
            cacheConfiguration: {
                type: "s3",
                shared: true,
                ...cacheConfiguration,
            },
        });
    }
    /**
     * @stability stable
     */
    static fromProps(props) {
        return new ConfigurationMapper(props);
    }
    /**
     * @stability stable
     */
    toToml() {
        return toml_1.stringify(this._toJsonMap());
    }
    /**
     * @internal
     */
    _toJsonMap() {
        const { globalConfiguration, runnerConfiguration, dockerConfiguration, machineConfiguration, autoscalingConfigurations, cacheConfiguration, } = this.props;
        const result = toJsonMap(globalConfiguration, snake_case_1.snakeCase);
        const runner = toJsonMap(runnerConfiguration, snake_case_1.snakeCase);
        result.runners = [runner];
        runner.docker = toJsonMap(dockerConfiguration, snake_case_1.snakeCase);
        const machine = toJsonMap(machineConfiguration, pascal_case_1.pascalCase);
        machine.MachineOptions = toProperties(machineConfiguration.machineOptions, (key) => `amazonec2-${param_case_1.paramCase(key)}`);
        machine.autoscaling = autoscalingConfigurations.map((autoscaling) => toJsonMap(autoscaling, pascal_case_1.pascalCase));
        runner.machine = machine;
        const cache = toJsonMap(cacheConfiguration, pascal_case_1.pascalCase);
        delete cache.S3;
        cache.s3 = toJsonMap(cacheConfiguration.s3, pascal_case_1.pascalCase);
        runner.cache = cache;
        return result;
    }
}
exports.ConfigurationMapper = ConfigurationMapper;
_a = JSII_RTTI_SYMBOL_1;
ConfigurationMapper[_a] = { fqn: "@pepperize-testing/cdk-autoscaling-gitlab-runner.ConfigurationMapper", version: "0.0.71" };
/**
 * Transforms configuration objects to JsonMap. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 */
function toJsonMap(configuration, inflector) {
    const result = {};
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result[inflector(key)] = value;
    }
    return result;
}
/**
 * Transforms configuration objects to a property array. Pass an inflector function to transform object keys.
 *
 * @param configuration
 * @param inflector A function to transform the object key
 * @example
 * // returns ["foo=bar"]
 * toProperties({foo: "bar", (s) => s});
 */
function toProperties(configuration, inflector) {
    const result = [];
    for (const key in configuration) {
        const value = configuration[key];
        if (value === undefined) {
            continue;
        }
        result.push(`${inflector(key)}=${value}`);
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZmlndXJhdGlvbi1tYXBwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvcnVubmVyLWNvbmZpZ3VyYXRpb24vY29uZmlndXJhdGlvbi1tYXBwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxzQ0FBMEQ7QUFDMUQsMkNBQXVDO0FBQ3ZDLDZDQUF5QztBQUN6QywyQ0FBdUM7Ozs7QUFpQnZDLE1BQWEsbUJBQW1CO0lBMEU5QixZQUE2QixLQUErQjtRQUEvQixVQUFLLEdBQUwsS0FBSyxDQUEwQjtJQUFHLENBQUM7Ozs7SUF6RXpELE1BQU0sQ0FBQyxZQUFZLENBQUMsS0FBK0I7UUFDeEQsTUFBTSxFQUNKLG1CQUFtQixFQUNuQixtQkFBbUIsRUFDbkIsbUJBQW1CLEVBQ25CLG9CQUFvQixFQUNwQix5QkFBeUIsRUFDekIsa0JBQWtCLEdBQ25CLEdBQUcsS0FBSyxDQUFDO1FBRVYsT0FBTyxJQUFJLG1CQUFtQixDQUFDO1lBQzdCLG1CQUFtQixFQUFFO2dCQUNuQixVQUFVLEVBQUUsRUFBRTtnQkFDZCxhQUFhLEVBQUUsQ0FBQztnQkFDaEIsU0FBUyxFQUFFLFFBQVE7Z0JBQ25CLFFBQVEsRUFBRSxNQUFNO2dCQUNoQixHQUFHLG1CQUFtQjthQUN2QjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixJQUFJLEVBQUUsZUFBZTtnQkFDckIsR0FBRyxFQUFFLG9CQUFvQjtnQkFDekIsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsV0FBVyxFQUFFLFFBQVE7Z0JBQ3JCLFFBQVEsRUFBRSxnQkFBZ0I7Z0JBQzFCLFdBQVcsRUFBRSxDQUFDLHdCQUF3QixFQUFFLDJCQUEyQixDQUFDO2dCQUNwRSxHQUFHLG1CQUFtQjthQUN2QjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixTQUFTLEVBQUUsS0FBSztnQkFDaEIsS0FBSyxFQUFFLGdCQUFnQjtnQkFDdkIsVUFBVSxFQUFFLElBQUk7Z0JBQ2hCLE1BQU0sRUFBRSxDQUFDLGVBQWUsQ0FBQztnQkFDekIsc0JBQXNCLEVBQUUsR0FBRztnQkFDM0IsWUFBWSxFQUFFLEtBQUs7Z0JBQ25CLE9BQU8sRUFBRSxDQUFDLGVBQWUsRUFBRSxRQUFRLENBQUM7Z0JBQ3BDLE9BQU8sRUFBRSxDQUFDO2dCQUNWLEdBQUcsbUJBQW1CO2FBQ3ZCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLFNBQVMsRUFBRSxDQUFDO2dCQUNaLFFBQVEsRUFBRSxHQUFHO2dCQUNiLFNBQVMsRUFBRSxFQUFFO2dCQUNiLGFBQWEsRUFBRSxXQUFXO2dCQUMxQixXQUFXLEVBQUUsaUJBQWlCO2dCQUM5QixHQUFHLG9CQUFvQjtnQkFDdkIsY0FBYyxFQUFFO29CQUNkLG1CQUFtQixFQUFFLElBQUk7b0JBQ3pCLFNBQVMsRUFBRSxJQUFJO29CQUNmLEdBQUcsb0JBQW9CLENBQUMsY0FBYztpQkFDdkM7YUFDRjtZQUNELHlCQUF5QixFQUFFLHlCQUF5QixDQUFDLE1BQU07Z0JBQ3pELENBQUMsQ0FBQyx5QkFBeUI7Z0JBQzNCLENBQUMsQ0FBQztvQkFDRTt3QkFDRSxPQUFPLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQzt3QkFDbkMsU0FBUyxFQUFFLENBQUM7d0JBQ1osUUFBUSxFQUFFLElBQUk7d0JBQ2QsUUFBUSxFQUFFLFNBQVM7cUJBQ3BCO2lCQUNGO1lBQ0wsa0JBQWtCLEVBQUU7Z0JBQ2xCLElBQUksRUFBRSxJQUFJO2dCQUNWLE1BQU0sRUFBRSxJQUFJO2dCQUNaLEdBQUcsa0JBQWtCO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7OztJQUVNLE1BQU0sQ0FBQyxTQUFTLENBQUMsS0FBK0I7UUFDckQsT0FBTyxJQUFJLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3hDLENBQUM7Ozs7SUFJTSxNQUFNO1FBQ1gsT0FBTyxnQkFBUyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7T0FFRztJQUNJLFVBQVU7UUFDZixNQUFNLEVBQ0osbUJBQW1CLEVBQ25CLG1CQUFtQixFQUNuQixtQkFBbUIsRUFDbkIsb0JBQW9CLEVBQ3BCLHlCQUF5QixFQUN6QixrQkFBa0IsR0FDbkIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO1FBRWYsTUFBTSxNQUFNLEdBQVksU0FBUyxDQUFDLG1CQUFtQixFQUFFLHNCQUFTLENBQUMsQ0FBQztRQUVsRSxNQUFNLE1BQU0sR0FBWSxTQUFTLENBQUMsbUJBQW1CLEVBQUUsc0JBQVMsQ0FBQyxDQUFDO1FBQ2xFLE1BQU0sQ0FBQyxPQUFPLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUxQixNQUFNLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQyxtQkFBbUIsRUFBRSxzQkFBUyxDQUFDLENBQUM7UUFFMUQsTUFBTSxPQUFPLEdBQUcsU0FBUyxDQUFDLG9CQUFvQixFQUFFLHdCQUFVLENBQUMsQ0FBQztRQUM1RCxPQUFPLENBQUMsY0FBYyxHQUFHLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLGFBQWEsc0JBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDbkgsT0FBTyxDQUFDLFdBQVcsR0FBRyx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsd0JBQVUsQ0FBQyxDQUFDLENBQUM7UUFDekcsTUFBTSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFFekIsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLGtCQUFrQixFQUFFLHdCQUFVLENBQUMsQ0FBQztRQUN4RCxPQUFPLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDaEIsS0FBSyxDQUFDLEVBQUUsR0FBRyxTQUFTLENBQUMsa0JBQWtCLENBQUMsRUFBRSxFQUFFLHdCQUFVLENBQUMsQ0FBQztRQUN4RCxNQUFNLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUVyQixPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDOztBQS9HSCxrREFnSEM7OztBQUVEOzs7OztHQUtHO0FBQ0gsU0FBUyxTQUFTLENBQUksYUFBZ0IsRUFBRSxTQUFnQztJQUN0RSxNQUFNLE1BQU0sR0FBWSxFQUFFLENBQUM7SUFDM0IsS0FBSyxNQUFNLEdBQUcsSUFBSSxhQUFhLEVBQUU7UUFDL0IsTUFBTSxLQUFLLEdBQUcsYUFBYSxDQUFDLEdBQUcsQ0FBdUIsQ0FBQztRQUN2RCxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsU0FBUztTQUNWO1FBQ0QsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQztLQUNoQztJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFFRDs7Ozs7Ozs7R0FRRztBQUNILFNBQVMsWUFBWSxDQUFJLGFBQWdCLEVBQUUsU0FBZ0M7SUFDekUsTUFBTSxNQUFNLEdBQUcsRUFBRSxDQUFDO0lBQ2xCLEtBQUssTUFBTSxHQUFHLElBQUksYUFBYSxFQUFFO1FBQy9CLE1BQU0sS0FBSyxHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQXVCLENBQUM7UUFDdkQsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLFNBQVM7U0FDVjtRQUNELE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsQ0FBQztLQUMzQztJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBbnlKc29uLCBKc29uTWFwLCBzdHJpbmdpZnkgfSBmcm9tIFwiQGlhcm5hL3RvbWxcIjtcbmltcG9ydCB7IHBhcmFtQ2FzZSB9IGZyb20gXCJwYXJhbS1jYXNlXCI7XG5pbXBvcnQgeyBwYXNjYWxDYXNlIH0gZnJvbSBcInBhc2NhbC1jYXNlXCI7XG5pbXBvcnQgeyBzbmFrZUNhc2UgfSBmcm9tIFwic25ha2UtY2FzZVwiO1xuaW1wb3J0IHsgQXV0b3NjYWxpbmdDb25maWd1cmF0aW9uIH0gZnJvbSBcIi4vYXV0b3NjYWxpbmctY29uZmlndXJhdGlvblwiO1xuaW1wb3J0IHsgQ2FjaGVDb25maWd1cmF0aW9uIH0gZnJvbSBcIi4vY2FjaGUtY29uZmlndXJhdGlvblwiO1xuaW1wb3J0IHsgRG9ja2VyQ29uZmlndXJhdGlvbiB9IGZyb20gXCIuL2RvY2tlci1jb25maWd1cmF0aW9uXCI7XG5pbXBvcnQgeyBHbG9iYWxDb25maWd1cmF0aW9uIH0gZnJvbSBcIi4vZ2xvYmFsLWNvbmZpZ3VyYXRpb25cIjtcbmltcG9ydCB7IE1hY2hpbmVDb25maWd1cmF0aW9uIH0gZnJvbSBcIi4vbWFjaGluZS1jb25maWd1cmF0aW9uXCI7XG5pbXBvcnQgeyBSdW5uZXJDb25maWd1cmF0aW9uIH0gZnJvbSBcIi4vcnVubmVyLWNvbmZpZ3VyYXRpb25cIjtcblxuZXhwb3J0IGludGVyZmFjZSBDb25maWd1cmF0aW9uTWFwcGVyUHJvcHMge1xuICByZWFkb25seSBnbG9iYWxDb25maWd1cmF0aW9uOiBHbG9iYWxDb25maWd1cmF0aW9uO1xuICByZWFkb25seSBydW5uZXJDb25maWd1cmF0aW9uOiBSdW5uZXJDb25maWd1cmF0aW9uO1xuICByZWFkb25seSBkb2NrZXJDb25maWd1cmF0aW9uOiBEb2NrZXJDb25maWd1cmF0aW9uO1xuICByZWFkb25seSBtYWNoaW5lQ29uZmlndXJhdGlvbjogTWFjaGluZUNvbmZpZ3VyYXRpb247XG4gIHJlYWRvbmx5IGF1dG9zY2FsaW5nQ29uZmlndXJhdGlvbnM6IEF1dG9zY2FsaW5nQ29uZmlndXJhdGlvbltdO1xuICByZWFkb25seSBjYWNoZUNvbmZpZ3VyYXRpb246IENhY2hlQ29uZmlndXJhdGlvbjtcbn1cblxuZXhwb3J0IGNsYXNzIENvbmZpZ3VyYXRpb25NYXBwZXIge1xuICBwdWJsaWMgc3RhdGljIHdpdGhEZWZhdWx0cyhwcm9wczogQ29uZmlndXJhdGlvbk1hcHBlclByb3BzKSB7XG4gICAgY29uc3Qge1xuICAgICAgZ2xvYmFsQ29uZmlndXJhdGlvbixcbiAgICAgIHJ1bm5lckNvbmZpZ3VyYXRpb24sXG4gICAgICBkb2NrZXJDb25maWd1cmF0aW9uLFxuICAgICAgbWFjaGluZUNvbmZpZ3VyYXRpb24sXG4gICAgICBhdXRvc2NhbGluZ0NvbmZpZ3VyYXRpb25zLFxuICAgICAgY2FjaGVDb25maWd1cmF0aW9uLFxuICAgIH0gPSBwcm9wcztcblxuICAgIHJldHVybiBuZXcgQ29uZmlndXJhdGlvbk1hcHBlcih7XG4gICAgICBnbG9iYWxDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIGNvbmN1cnJlbnQ6IDEwLFxuICAgICAgICBjaGVja0ludGVydmFsOiAwLFxuICAgICAgICBsb2dGb3JtYXQ6IFwicnVubmVyXCIsXG4gICAgICAgIGxvZ0xldmVsOiBcImluZm9cIixcbiAgICAgICAgLi4uZ2xvYmFsQ29uZmlndXJhdGlvbixcbiAgICAgIH0sXG4gICAgICBydW5uZXJDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIG5hbWU6IFwiZ2l0bGFiLXJ1bm5lclwiLFxuICAgICAgICB1cmw6IFwiaHR0cHM6Ly9naXRsYWIuY29tXCIsXG4gICAgICAgIGxpbWl0OiAxMCxcbiAgICAgICAgb3V0cHV0TGltaXQ6IDUyNDI4ODAwLFxuICAgICAgICBleGVjdXRvcjogXCJkb2NrZXIrbWFjaGluZVwiLFxuICAgICAgICBlbnZpcm9ubWVudDogW1wiRE9DS0VSX0RSSVZFUj1vdmVybGF5MlwiLCBcIkRPQ0tFUl9UTFNfQ0VSVERJUj0vY2VydHNcIl0sXG4gICAgICAgIC4uLnJ1bm5lckNvbmZpZ3VyYXRpb24sXG4gICAgICB9LFxuICAgICAgZG9ja2VyQ29uZmlndXJhdGlvbjoge1xuICAgICAgICB0bHNWZXJpZnk6IGZhbHNlLFxuICAgICAgICBpbWFnZTogXCJkb2NrZXI6MTkuMDMuNVwiLFxuICAgICAgICBwcml2aWxlZ2VkOiB0cnVlLFxuICAgICAgICBjYXBBZGQ6IFtcIkNBUF9TWVNfQURNSU5cIl0sXG4gICAgICAgIHdhaXRGb3JTZXJ2aWNlc1RpbWVvdXQ6IDMwMCxcbiAgICAgICAgZGlzYWJsZUNhY2hlOiBmYWxzZSxcbiAgICAgICAgdm9sdW1lczogW1wiL2NlcnRzL2NsaWVudFwiLCBcIi9jYWNoZVwiXSxcbiAgICAgICAgc2htU2l6ZTogMCxcbiAgICAgICAgLi4uZG9ja2VyQ29uZmlndXJhdGlvbixcbiAgICAgIH0sXG4gICAgICBtYWNoaW5lQ29uZmlndXJhdGlvbjoge1xuICAgICAgICBpZGxlQ291bnQ6IDAsXG4gICAgICAgIGlkbGVUaW1lOiAzMDAsXG4gICAgICAgIG1heEJ1aWxkczogMjAsXG4gICAgICAgIG1hY2hpbmVEcml2ZXI6IFwiYW1hem9uZWMyXCIsXG4gICAgICAgIG1hY2hpbmVOYW1lOiBcImdpdGxhYi1ydW5uZXItJVwiLFxuICAgICAgICAuLi5tYWNoaW5lQ29uZmlndXJhdGlvbixcbiAgICAgICAgbWFjaGluZU9wdGlvbnM6IHtcbiAgICAgICAgICByZXF1ZXN0U3BvdEluc3RhbmNlOiB0cnVlLFxuICAgICAgICAgIHNwb3RQcmljZTogMC4wMyxcbiAgICAgICAgICAuLi5tYWNoaW5lQ29uZmlndXJhdGlvbi5tYWNoaW5lT3B0aW9ucyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBhdXRvc2NhbGluZ0NvbmZpZ3VyYXRpb25zOiBhdXRvc2NhbGluZ0NvbmZpZ3VyYXRpb25zLmxlbmd0aFxuICAgICAgICA/IGF1dG9zY2FsaW5nQ29uZmlndXJhdGlvbnNcbiAgICAgICAgOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHBlcmlvZHM6IFtcIiogKiA3LTIyICogKiBtb24tZnJpICpcIl0sXG4gICAgICAgICAgICAgIGlkbGVDb3VudDogMSxcbiAgICAgICAgICAgICAgaWRsZVRpbWU6IDE4MDAsXG4gICAgICAgICAgICAgIHRpbWV6b25lOiBcIkV0Yy9VVENcIixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgIGNhY2hlQ29uZmlndXJhdGlvbjoge1xuICAgICAgICB0eXBlOiBcInMzXCIsXG4gICAgICAgIHNoYXJlZDogdHJ1ZSxcbiAgICAgICAgLi4uY2FjaGVDb25maWd1cmF0aW9uLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVByb3BzKHByb3BzOiBDb25maWd1cmF0aW9uTWFwcGVyUHJvcHMpIHtcbiAgICByZXR1cm4gbmV3IENvbmZpZ3VyYXRpb25NYXBwZXIocHJvcHMpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihyZWFkb25seSBwcm9wczogQ29uZmlndXJhdGlvbk1hcHBlclByb3BzKSB7fVxuXG4gIHB1YmxpYyB0b1RvbWwoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gc3RyaW5naWZ5KHRoaXMuX3RvSnNvbk1hcCgpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9Kc29uTWFwKCk6IEpzb25NYXAge1xuICAgIGNvbnN0IHtcbiAgICAgIGdsb2JhbENvbmZpZ3VyYXRpb24sXG4gICAgICBydW5uZXJDb25maWd1cmF0aW9uLFxuICAgICAgZG9ja2VyQ29uZmlndXJhdGlvbixcbiAgICAgIG1hY2hpbmVDb25maWd1cmF0aW9uLFxuICAgICAgYXV0b3NjYWxpbmdDb25maWd1cmF0aW9ucyxcbiAgICAgIGNhY2hlQ29uZmlndXJhdGlvbixcbiAgICB9ID0gdGhpcy5wcm9wcztcblxuICAgIGNvbnN0IHJlc3VsdDogSnNvbk1hcCA9IHRvSnNvbk1hcChnbG9iYWxDb25maWd1cmF0aW9uLCBzbmFrZUNhc2UpO1xuXG4gICAgY29uc3QgcnVubmVyOiBKc29uTWFwID0gdG9Kc29uTWFwKHJ1bm5lckNvbmZpZ3VyYXRpb24sIHNuYWtlQ2FzZSk7XG4gICAgcmVzdWx0LnJ1bm5lcnMgPSBbcnVubmVyXTtcblxuICAgIHJ1bm5lci5kb2NrZXIgPSB0b0pzb25NYXAoZG9ja2VyQ29uZmlndXJhdGlvbiwgc25ha2VDYXNlKTtcblxuICAgIGNvbnN0IG1hY2hpbmUgPSB0b0pzb25NYXAobWFjaGluZUNvbmZpZ3VyYXRpb24sIHBhc2NhbENhc2UpO1xuICAgIG1hY2hpbmUuTWFjaGluZU9wdGlvbnMgPSB0b1Byb3BlcnRpZXMobWFjaGluZUNvbmZpZ3VyYXRpb24ubWFjaGluZU9wdGlvbnMsIChrZXkpID0+IGBhbWF6b25lYzItJHtwYXJhbUNhc2Uoa2V5KX1gKTtcbiAgICBtYWNoaW5lLmF1dG9zY2FsaW5nID0gYXV0b3NjYWxpbmdDb25maWd1cmF0aW9ucy5tYXAoKGF1dG9zY2FsaW5nKSA9PiB0b0pzb25NYXAoYXV0b3NjYWxpbmcsIHBhc2NhbENhc2UpKTtcbiAgICBydW5uZXIubWFjaGluZSA9IG1hY2hpbmU7XG5cbiAgICBjb25zdCBjYWNoZSA9IHRvSnNvbk1hcChjYWNoZUNvbmZpZ3VyYXRpb24sIHBhc2NhbENhc2UpO1xuICAgIGRlbGV0ZSBjYWNoZS5TMztcbiAgICBjYWNoZS5zMyA9IHRvSnNvbk1hcChjYWNoZUNvbmZpZ3VyYXRpb24uczMsIHBhc2NhbENhc2UpO1xuICAgIHJ1bm5lci5jYWNoZSA9IGNhY2hlO1xuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxufVxuXG4vKipcbiAqIFRyYW5zZm9ybXMgY29uZmlndXJhdGlvbiBvYmplY3RzIHRvIEpzb25NYXAuIFBhc3MgYW4gaW5mbGVjdG9yIGZ1bmN0aW9uIHRvIHRyYW5zZm9ybSBvYmplY3Qga2V5cy5cbiAqXG4gKiBAcGFyYW0gY29uZmlndXJhdGlvblxuICogQHBhcmFtIGluZmxlY3RvciBBIGZ1bmN0aW9uIHRvIHRyYW5zZm9ybSB0aGUgb2JqZWN0IGtleVxuICovXG5mdW5jdGlvbiB0b0pzb25NYXA8VD4oY29uZmlndXJhdGlvbjogVCwgaW5mbGVjdG9yOiAoczogc3RyaW5nKSA9PiBzdHJpbmcpOiBKc29uTWFwIHtcbiAgY29uc3QgcmVzdWx0OiBKc29uTWFwID0ge307XG4gIGZvciAoY29uc3Qga2V5IGluIGNvbmZpZ3VyYXRpb24pIHtcbiAgICBjb25zdCB2YWx1ZSA9IGNvbmZpZ3VyYXRpb25ba2V5XSBhcyB1bmtub3duIGFzIEFueUpzb247XG4gICAgaWYgKHZhbHVlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgICByZXN1bHRbaW5mbGVjdG9yKGtleSldID0gdmFsdWU7XG4gIH1cblxuICByZXR1cm4gcmVzdWx0O1xufVxuXG4vKipcbiAqIFRyYW5zZm9ybXMgY29uZmlndXJhdGlvbiBvYmplY3RzIHRvIGEgcHJvcGVydHkgYXJyYXkuIFBhc3MgYW4gaW5mbGVjdG9yIGZ1bmN0aW9uIHRvIHRyYW5zZm9ybSBvYmplY3Qga2V5cy5cbiAqXG4gKiBAcGFyYW0gY29uZmlndXJhdGlvblxuICogQHBhcmFtIGluZmxlY3RvciBBIGZ1bmN0aW9uIHRvIHRyYW5zZm9ybSB0aGUgb2JqZWN0IGtleVxuICogQGV4YW1wbGVcbiAqIC8vIHJldHVybnMgW1wiZm9vPWJhclwiXVxuICogdG9Qcm9wZXJ0aWVzKHtmb286IFwiYmFyXCIsIChzKSA9PiBzfSk7XG4gKi9cbmZ1bmN0aW9uIHRvUHJvcGVydGllczxUPihjb25maWd1cmF0aW9uOiBULCBpbmZsZWN0b3I6IChzOiBzdHJpbmcpID0+IHN0cmluZyk6IHN0cmluZ1tdIHtcbiAgY29uc3QgcmVzdWx0ID0gW107XG4gIGZvciAoY29uc3Qga2V5IGluIGNvbmZpZ3VyYXRpb24pIHtcbiAgICBjb25zdCB2YWx1ZSA9IGNvbmZpZ3VyYXRpb25ba2V5XSBhcyB1bmtub3duIGFzIEFueUpzb247XG4gICAgaWYgKHZhbHVlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgICByZXN1bHQucHVzaChgJHtpbmZsZWN0b3Ioa2V5KX09JHt2YWx1ZX1gKTtcbiAgfVxuXG4gIHJldHVybiByZXN1bHQ7XG59XG4iXX0=