"""
Writes the parts of the API that are simple: flags, enums, structs.
"""

import os

from codegen.utils import print, lib_dir, blacken, to_snake_case
from codegen.idlparser import get_idl_parser


flags_preamble = '''
"""
All wgpu flags. Also available in the root wgpu namespace.
"""

# THIS CODE IS AUTOGENERATED - DO NOT EDIT

_use_sphinx_repr = False

class Flags:

    def __init__(self, name, **kwargs):
        self._name = name
        for key, val in kwargs.items():
            setattr(self, key, val)

    def __iter__(self):
        return iter([key for key in dir(self) if not key.startswith("_")])

    def __repr__(self):
        options = ", ".join(self)
        if _use_sphinx_repr:  # no-cover
            return options
        return f"<{self.__class__.__name__} {self._name}: {options}>"

'''.lstrip()


def write_flags():
    idl = get_idl_parser()
    n = len(idl.flags)
    # Generate code
    pylines = [flags_preamble]
    pylines.append(f"# There are {n} flags\n")
    for name, d in idl.flags.items():
        pylines.append(f'{name} = Flags(\n    "{name}",')
        for key, val in d.items():
            pylines.append(f"    {key}={val!r},")
        pylines.append(")  #:\n")
    # Write
    code = blacken("\n".join(pylines))
    with open(os.path.join(lib_dir, "flags.py"), "wb") as f:
        f.write(code.encode())
    print(f"Wrote {n} flags to flags.py")


enums_preamble = '''
"""
All wgpu enums. Also available in the root wgpu namespace.
"""

# THIS CODE IS AUTOGENERATED - DO NOT EDIT

_use_sphinx_repr = False

class Enum:

    def __init__(self, name, **kwargs):
        self._name = name
        for key, val in kwargs.items():
            setattr(self, key, val)

    def __iter__(self):
        return iter(
            [getattr(self, key) for key in dir(self) if not key.startswith("_")]
        )

    def __repr__(self):
        options = ", ".join(f"'{x}'" for x in self)
        if _use_sphinx_repr:  # no-cover
            return options
        return f"<{self.__class__.__name__} {self._name}: {options}>"

'''.lstrip()


def write_enums():
    idl = get_idl_parser()
    n = len(idl.enums)
    # Generate code
    pylines = [enums_preamble]
    pylines.append(f"# There are {n} enums\n")
    for name, d in idl.enums.items():
        pylines.append(f'{name} = Enum(\n    "{name}",')
        for key, val in d.items():
            pylines.append(f'    {key}="{val}",')
        pylines.append(")  #:\n")  # That #: is for Sphinx
    # Write
    code = blacken("\n".join(pylines))
    with open(os.path.join(lib_dir, "enums.py"), "wb") as f:
        f.write(code.encode())
    print(f"Wrote {n} enums to enums.py")


structs_preamble = '''
"""
All wgpu structs.
"""

# THIS CODE IS AUTOGENERATED - DO NOT EDIT

_use_sphinx_repr = False


class Struct:
    def __init__(self, name, **kwargs):
        self._name = name
        for key, val in kwargs.items():
            setattr(self, key, val)

    def __iter__(self):
        return iter(
            [key for key in dir(self) if not key.startswith("_")]
        )

    def __repr__(self):
        options = ", ".join(f"'{x}'" for x in self)
        if _use_sphinx_repr:  # no-cover
            return options
        return f"<{self.__class__.__name__} {self._name}: {options}>"

'''.lstrip()


def write_structs():
    idl = get_idl_parser()
    n = len(idl.structs)
    # Generate code
    pylines = [structs_preamble]
    pylines.append(f"# There are {n} structs\n")
    for name, d in idl.structs.items():
        pylines.append(f'{name} = Struct(\n    "{name}",')
        for field in d.values():
            key = to_snake_case(field.name)
            val = idl.resolve_type(field.typename)
            if not val.startswith(("'", '"')):
                val = f"'{val}'"
            pylines.append(f"    {key}={val},")
        pylines.append(")  #:\n")  # That #: is for Sphinx

    # Write
    code = blacken("\n".join(pylines))
    with open(os.path.join(lib_dir, "structs.py"), "wb") as f:
        f.write(code.encode())
    print(f"Wrote {n} structs to structs.py")
