"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy
            },
            ruleSetName: props.ruleSet.receiptRuleSetName
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16'
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        }
                    })
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
// tslint:disable:no-console
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log("SES Notification:\n", JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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