# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['avr', 'avr.mqtt', 'avr.serial', 'avr.utils']

package_data = \
{'': ['*']}

install_requires = \
['loguru>=0.6.0,<0.7.0']

extras_require = \
{'mqtt': ['paho-mqtt>=1.6.1,<2.0.0'], 'serial': ['pyserial>=3.5,<4.0']}

setup_kwargs = {
    'name': 'bell-avr-libraries',
    'version': '0.1.2',
    'description': '',
    'long_description': '# AVR-Python-Libraries\n\n## Install\n\nTo install the base package, run:\n\n```bash\npip install bell-avr-libraries\n```\n\nAdditionally, the `mqtt` and `serial` extras are available if you want to use\nthe MQTT or PCC functionality.\n\n```bash\npip install bell-avr-libraries[mqtt,serial]\n```\n\n## Usage\n\n### MQTT\n\n```python\nfrom bell.avr import mqtt\n```\n\nThese are MQTT utilities that are used to have a consistent messaging protocol\nthroughout all the AVR software modules.\n\nThe first part of this are the payloads for the MQTT messages themselves. As AVR\nexclusively uses JSON, these are all\n[`TypedDict`](https://docs.python.org/3/library/typing.html#typing.TypedDict)s\nthat have all of the required fields for a message.\n\nExample:\n\n```python\nfrom bell.avr.mqtt.payloads import AvrPcmSetBaseColorPayload\n\npayload = AvrPcmSetBaseColorPayload((128, 232, 142, 0))\n```\n\nThe second part of the MQTT libraries, is the `MQTTModule` class.\nThis is a boilerplate module for AVR that makes it very easy to send\nand recieve MQTT messages and do something with them.\n\nExample:\n\n```python\nfrom bell.avr.mqtt.client import MQTTModule\nfrom bell.avr.mqtt.payloads import AvrFcmVelocityPayload, AvrPcmSetServoOpenClosePayload\n\n\nclass Sandbox(MQTTModule):\n    def __init__(self) -> None:\n        self.topic_map = {"avr/fcm/velocity": self.show_velocity}\n\n    def show_velocity(self, payload: AvrFcmVelocityPayload) -> None:\n        vx = payload["vX"]\n        vy = payload["vY"]\n        vz = payload["vZ"]\n        v_ms = (vx, vy, vz)\n        print(f"Velocity information: {v_ms} m/s")\n\n    def open_servo(self) -> None:\n        payload = AvrPcmSetServoOpenClosePayload(servo=0, action="open")\n        self.send_message("avr/pcm/set_servo_open_close", payload)\n\n\nif __name__ == "__main__":\n    box = Sandbox()\n    box.run()\n```\n\nThe `topic_map` dictionary is a class attribute that maps topics to subscribe to\nand a function that will handle an incoming payload. The `payload` argument\nshould match the appropriate `Payload` class for that topic.\n\nAdditionally, the `message_cache` attribute is a dictionary that holds\na copy of the last payload sent by that module on a given topic. The keys are the\ntopic strings, and the values are the topic payloads.\n\n### Utils\n\n```python\nfrom bell.avr import utils\n```\n\nThese are general purpose utilities.\n\n#### Decorators\n\n```python\nfrom bell.avr.utils import decorators\n```\n\nThere are 3 different function decorators available, which are helpful for MQTT\nmessage callbacks. First is the `try_except` decorator, which wraps the\nfunction in a `try: except:` statement and will log any exceptions to the console:\n\n```python\n    @decorators.try_except\n    def assemble_hil_gps_message(self) -> None:\n        ...\n```\n\nAdditionally, there is the `reraise` argument, which can be set to `True` to raise\nany exceptions that are encountered. This is helpful if you still want exceptions\nto propagate up, but log them.\n\nThere is an async version of this decorator as well with an `async_` prefix.\n\n```python\n    @decorators.async_try_except()\n    async def connected_status_telemetry(self) -> None:\n        ...\n```\n\nThe last decorator is `run_forever` which will run the wrapped function forever,\nwith a given `period` or `frequency`.\n\n```python\n    @decorators.run_forever(frequency=5)\n    def read_data(self) -> None:\n        ...\n```\n\n#### Timing\n\n```python\nfrom bell.avr.utils import timing\n```\n\nHere is a `rate_limit` function which take a callable and a\nperiod or frequency, and only run the callable at that given rate.\n\n```python\nfor _ in range(10):\n    # add() will only run twice\n    timing.rate_limit(add, period=5)\n    time.sleep(1)\n```\n\nThis works tracking calls to the `rate_limit` function from a line number\nwithin a file, so multiple calls to `rate_limit` say within a loop\nwith the same callable and period will be treated seperately. This allows\nfor dynamic frequency manipulation.\n\n### Serial\n\n```python\nfrom bell.avr import serial\n```\n\nThese are serial utilities that help facilitate finding and communicating\nwith the AVR peripherial control computer.\n\n#### Client\n\n```python\nfrom bell.avr.serial import client\n```\n\nThe `SerialLoop` class is a small wrapper around the `pyserial` `serial.Serial`\nclass which adds a `run` method that will try to read data from the serial device\nas fast as possible.\n\n```python\nser = client.SerialLoop()\n```\n\n#### PCC\n\n```python\nfrom bell.avr.serial import client\n```\n\nThe `PeripheralControlComputer` class sends serial messages\nto the AVR peripherial control computer, via easy-to-use class methods.\n\n```python\nimport bell.avr.serial\nimport threading\n\nclient = bell.avr.serial.client.SerialLoop()\nclient.port = port\nclient.baudrate = baudrate\n\npcc = bell.avr.serial.pcc.PeripheralControlComputer(client)\n\nclient_thread = threading.Thread(target=client.run)\nclient_thread.start()\n\npcc.set_servo_max(0, 100)\n```\n\n#### Ports\n\n```python\nfrom bell.avr.serial import ports\n```\n\nHere is a `list_serial_ports` function which returns a list of detected serial\nports connected to the system.\n\n```python\nserial_ports = ports.list_serial_ports()\n# ["COM1", "COM5", ...]\n```\n\n## Development\n\nInstall [`poetry`](https://python-poetry.org/) and run\n`poetry install --extras mqtt --extras serial` to install of the dependencies\ninside a virtual environment.\n\nBuild the auto-generated code with `poetry run python build.py`. From here,\nyou can now produce a package with `poetry build`.\n\nTo add new message definitions, add entries to the `bell/avr/mqtt/messages.jsonc` file.\nThe 3 parts of a new message are as follows:\n\n1. "topic": This is the full topic path for the message. This must be all lower case and\n   start with "avr/".\n2. "payload": These are the keys of the payload for the message.\n   This is a list of key entries (see below).\n3. "docs": This is an optional list of Markdown strings that explains what this\n   message does. Each list item is a new line.\n\nThe key entries for a message have the following elements:\n\n1. "key": This is the name of the key. Must be a valid Python variable name.\n2. "type": This is the data type of the key such as `Tuple[int, int, int, int]`.\n   Must be a valid Python type hint. Otherwise, this can be a list of more\n   key entries, effectively creating a nested dictionary.\n3. "docs": This is an optional list of Markdown strings that explains what the\n   key is. Each list item is a new line.\n\nThe `bell/avr/mqtt/schema.json` file will help ensure the correct schema is maintained,\nassuming you are using VS Code.\n\n## MQTT Documentation\n\n### Data Types\n\n#### AvrApriltagsSelectedPos\n\nThe position of the vehicle in world frame **in cm**\n\n- `"n"` (`float`): The +north position of the vehicle relative to the world origin in world frame\n- `"e"` (`float`): The +east position of the vehicle relative to the world origin in world frame\n- `"d"` (`float`): The +down position of the vehicle relative to the world origin in world frame\n\n#### AvrApriltagsRawTags\n\n- `"id"` (`int`): The ID of the tag\n- `"pos"` (`AvrApriltagsRawTagsPos`)\n- `"rotation"` (`Tuple[Tuple[float, float, float], Tuple[float, float, float], Tuple[float, float, float]]`): The 3x3 rotation matrix\n\n#### AvrApriltagsRawTagsPos\n\n- `"x"` (`float`): The position **in meters** of the camera relative to the **tag\'s x** frame\n- `"y"` (`float`): The position **in meters** of the camera relative to the **tag\'s y** frame\n- `"z"` (`float`): The position **in meters** of the camera relative to the **tag\'s z** frame\n\n#### AvrApriltagsVisibleTags\n\n- `"id"` (`int`): The ID of the tag\n- `"horizontal_dist"` (`float`): The horizontal scalar distance from vehicle to tag, **in cm**\n- `"vertical_dist"` (`float`): The vertical scalar distance from vehicle to tag, **in cm**\n- `"angle_to_tag"` (`float`): The angle formed by the vector pointing from the vehicles body to the tag in world frame relative to world-north\n- `"heading"` (`float`): The heading of the vehicle in world frame\n- `"pos_rel"` (`AvrApriltagsVisibleTagsPosRel`): The relative position of the vehicle to the tag in world frame **in cm**\n- `"pos_world"` (`AvrApriltagsVisibleTagsPosWorld`): The position of the vehicle in world frame **in cm** (if the tag has no truth data, this will not be present in the output)\n\n#### AvrApriltagsVisibleTagsPosRel\n\nThe relative position of the vehicle to the tag in world frame **in cm**\n\n- `"x"` (`float`): The x (+north/-south) position of the vehicle relative to the tag in world frame (for reference the mountain is **north** of the beach)\n- `"y"` (`float`): The y (+east/-west) position of the vehicle relative to the tag in world frame\n- `"z"` (`float`): The z (+down/-up) position of the vehicle relative to the tag in world frame (no, this is not a typo, up is really - )\n\n#### AvrApriltagsVisibleTagsPosWorld\n\nThe position of the vehicle in world frame **in cm** (if the tag has no truth data, this will not be present in the output)\n\n- `"x"` (`Optional[float]`): The x position of the vehicle relative to the world origin (this is the ship) in world frame (for reference the mountain is **north** of the beach)\n- `"y"` (`Optional[float]`): The y position of the vehicle relative to the world origin in world frame\n- `"z"` (`Optional[float]`): The z position of the vehicle relative to the world origin in world frame\n\n### Message Payloads\n\n#### AvrAutonomousPayload\n\nTopic: `avr/autonomous`\n\nThis enables enable or disable autonomous mode. This is not used by any Bell code, but available to students to listen to who may wish to not always be running their autonomous mode,\n\n- `"enable"` (`bool`)\n\n#### AvrPcmSetBaseColorPayload\n\nTopic: `avr/pcm/set_base_color`\n\nThis sets the color of the LED strip on the PCC\n\n- `"wrgb"` (`Tuple[int, int, int, int]`): A list of 4 `int`s between 0 and 255 to set the base color of the LEDs. Example: [255, 0, 128, 255].\n\n#### AvrPcmSetTempColorPayload\n\nTopic: `avr/pcm/set_temp_color`\n\nThis sets the color of the LED strip on the PCC temporarily\n\n- `"wrgb"` (`Tuple[int, int, int, int]`): A list of 4 `int`s between 0 and 255 to set the base color of the LEDs. Example: [255, 0, 128, 255].\n- `"time"` (`float`): Optional `float` for the number of seconds the color should be set for. Default is 0.5.\n\n#### AvrPcmSetLaserOnPayload\n\nTopic: `avr/pcm/set_laser_on`\n\nThere is no content for this class\n\n#### AvrPcmSetLaserOffPayload\n\nTopic: `avr/pcm/set_laser_off`\n\nThere is no content for this class\n\n#### AvrPcmSetServoOpenClosePayload\n\nTopic: `avr/pcm/set_servo_open_close`\n\n- `"servo"` (`int`): ID of the servo to open or close as an `int`. This is 0-indexed.\n- `"action"` (`Literal["open", "close"]`): Either the literal string "open" or "close".\n\n#### AvrPcmSetServoMinPayload\n\nTopic: `avr/pcm/set_servo_min`\n\n- `"servo"` (`int`): ID of the servo to set the minimum pulse width as an `int`. This is 0-indexed.\n- `"min_pulse"` (`int`): A `int` between 0 and 1000.\n\n#### AvrPcmSetServoMaxPayload\n\nTopic: `avr/pcm/set_servo_max`\n\n- `"servo"` (`int`): ID of the servo to set the maximum pulse width as an `int`. This is 0-indexed.\n- `"max_pulse"` (`int`): A `int` between 0 and 1000.\n\n#### AvrPcmSetServoPctPayload\n\nTopic: `avr/pcm/set_servo_pct`\n\n- `"servo"` (`int`): ID of the servo to set the percent as an `int`. This is 0-indexed.\n- `"percent"` (`int`): A `int` between 0 and 100.\n\n#### AvrPcmResetPayload\n\nTopic: `avr/pcm/reset`\n\nThis resets the PCC\n\nThere is no content for this class\n\n#### AvrFcmHilGpsStatsPayload\n\nTopic: `avr/fcm/hil_gps_stats`\n\n- `"num_frames"` (`int`)\n\n#### AvrFcmEventsPayload\n\nTopic: `avr/fcm/events`\n\n- `"name"` (`str`)\n- `"payload"` (`str`)\n- `"timestamp"` (`str`)\n\n#### AvrFcmBatteryPayload\n\nTopic: `avr/fcm/battery`\n\n- `"voltage"` (`float`): Battery voltage\n- `"soc"` (`float`): State of charge (0 - 100)\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmStatusPayload\n\nTopic: `avr/fcm/status`\n\n- `"armed"` (`bool`): True/False if the drone is currently armed\n- `"mode"` (`str`): Current flight mode, which is one of the following:\n    - \'UNKNOWN\'\n    - \'READY\'\n    - \'TAKEOFF\'\n    - \'HOLD\'\n    - \'MISSION\'\n    - \'RETURN_TO_LAUNCH\'\n    - \'LAND\'\n    - \'OFFBOARD\'\n    - \'FOLLOW_ME\'\n    - \'MANUAL\'\n    - \'ALTCTL\'\n    - \'POSCTL\'\n    - \'ACRO\'\n    - \'STABILIZED\'\n    - \'RATTITUDE\'\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmLocationLocalPayload\n\nTopic: `avr/fcm/location/local`\n\n- `"dX"` (`float`): X position in a local North/East/Down coordinate system\n- `"dY"` (`float`): Y position in a local North/East/Down coordinate system\n- `"dZ"` (`float`): Z position in a local North/East/Down coordinate system\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmLocationGlobalPayload\n\nTopic: `avr/fcm/location/global`\n\n- `"lat"` (`float`): Latitude in global coordinates\n- `"lon"` (`float`): Longitude in global coordinates\n- `"alt"` (`float`): Relative altitude in global coordinates\n- `"hdg"` (`float`): Heading\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmLocationHomePayload\n\nTopic: `avr/fcm/location/home`\n\n- `"lat"` (`float`): Latitude relative to the home position\n- `"lon"` (`float`): Longitude relative to the home position\n- `"alt"` (`float`): Relative altitude to the home position\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmAttitudeEulerPayload\n\nTopic: `avr/fcm/attitude/euler`\n\n- `"roll"` (`float`): Roll in degrees\n- `"pitch"` (`float`): Pitch in degrees\n- `"yaw"` (`float`): Yaw in degrees\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmVelocityPayload\n\nTopic: `avr/fcm/velocity`\n\n- `"vX"` (`float`): X velocity in a local North/East/Down coordinate system\n- `"vY"` (`float`): Y velocity in a local North/East/Down coordinate system\n- `"vZ"` (`float`): Z velocity in a local North/East/Down coordinate system\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFcmGpsInfoPayload\n\nTopic: `avr/fcm/gps_info`\n\n- `"num_satellites"` (`int`): Number of visible satellites in use\n- `"fix_type"` (`str`): GPS fix type\n- `"timestamp"` (`str`): Time the message was sent in [ISO 8601 format](https://docs.python.org/3/library/datetime.html#datetime.datetime.isoformat)\n\n#### AvrFusionPositionNedPayload\n\nTopic: `avr/fusion/position/ned`\n\n- `"n"` (`float`)\n- `"e"` (`float`)\n- `"d"` (`float`)\n\n#### AvrFusionVelocityNedPayload\n\nTopic: `avr/fusion/velocity/ned`\n\n- `"Vn"` (`float`)\n- `"Ve"` (`float`)\n- `"Vd"` (`float`)\n\n#### AvrFusionGeoPayload\n\nTopic: `avr/fusion/geo`\n\n- `"lat"` (`float`)\n- `"lon"` (`float`)\n- `"alt"` (`float`)\n\n#### AvrFusionGroundspeedPayload\n\nTopic: `avr/fusion/groundspeed`\n\n- `"groundspeed"` (`float`)\n\n#### AvrFusionCoursePayload\n\nTopic: `avr/fusion/course`\n\n- `"course"` (`float`)\n\n#### AvrFusionClimbratePayload\n\nTopic: `avr/fusion/climbrate`\n\n- `"climb_rate_fps"` (`float`)\n\n#### AvrFusionAttitudeQuatPayload\n\nTopic: `avr/fusion/attitude/quat`\n\n- `"w"` (`float`)\n- `"x"` (`float`)\n- `"y"` (`float`)\n- `"z"` (`float`)\n\n#### AvrFusionAttitudeEulerPayload\n\nTopic: `avr/fusion/attitude/euler`\n\n- `"psi"` (`float`)\n- `"theta"` (`float`)\n- `"phi"` (`float`)\n\n#### AvrFusionAttitudeHeadingPayload\n\nTopic: `avr/fusion/attitude/heading`\n\n- `"heading"` (`float`)\n\n#### AvrFusionHilGpsPayload\n\nTopic: `avr/fusion/hil_gps`\n\n- `"time_usec"` (`int`)\n- `"fix_type"` (`int`)\n- `"lat"` (`int`)\n- `"lon"` (`int`)\n- `"alt"` (`int`)\n- `"eph"` (`int`)\n- `"epv"` (`int`)\n- `"vel"` (`int`)\n- `"vn"` (`int`)\n- `"ve"` (`int`)\n- `"vd"` (`int`)\n- `"cog"` (`int`)\n- `"satellites_visible"` (`int`)\n- `"heading"` (`int`)\n\n#### AvrVioResyncPayload\n\nTopic: `avr/vio/resync`\n\n- `"n"` (`float`)\n- `"e"` (`float`)\n- `"d"` (`float`)\n- `"heading"` (`float`)\n\n#### AvrVioPositionNedPayload\n\nTopic: `avr/vio/position/ned`\n\n- `"n"` (`float`)\n- `"e"` (`float`)\n- `"d"` (`float`)\n\n#### AvrVioVelocityNedPayload\n\nTopic: `avr/vio/velocity/ned`\n\n- `"n"` (`float`)\n- `"e"` (`float`)\n- `"d"` (`float`)\n\n#### AvrVioOrientationEulPayload\n\nTopic: `avr/vio/orientation/eul`\n\n- `"psi"` (`float`)\n- `"theta"` (`float`)\n- `"phi"` (`float`)\n\n#### AvrVioOrientationQuatPayload\n\nTopic: `avr/vio/orientation/quat`\n\n- `"w"` (`float`)\n- `"x"` (`float`)\n- `"y"` (`float`)\n- `"z"` (`float`)\n\n#### AvrVioHeadingPayload\n\nTopic: `avr/vio/heading`\n\n- `"degrees"` (`float`)\n\n#### AvrVioConfidencePayload\n\nTopic: `avr/vio/confidence`\n\n- `"tracker"` (`float`): Number between 0 and 100 of tracking confidence\n\n#### AvrApriltagsSelectedPayload\n\nTopic: `avr/apriltags/selected`\n\nThis topic publishes its best candidate for position feedback\n\n- `"tag_id"` (`int`): The id of the tag\n- `"pos"` (`AvrApriltagsSelectedPos`): The position of the vehicle in world frame **in cm**\n- `"heading"` (`float`)\n\n#### AvrApriltagsRawPayload\n\nTopic: `avr/apriltags/raw`\n\nThis topic publishes the raw tag data\n\n- `"tags"` (`List[AvrApriltagsRawTags]`)\n\n#### AvrApriltagsVisiblePayload\n\nTopic: `avr/apriltags/visible`\n\nThis topic publishes the transformed tag data\n\n- `"tags"` (`List[AvrApriltagsVisibleTags]`)\n\n#### AvrApriltagsFpsPayload\n\nTopic: `avr/apriltags/fps`\n\n- `"fps"` (`int`)\n\n#### AvrThermalReadingPayload\n\nTopic: `avr/thermal/reading`\n\n- `"data"` (`str`)\n\n#### AvrStatusLightPcmPayload\n\nTopic: `avr/status/light/pcm`\n\nThere is no content for this class\n\n#### AvrStatusLightVioPayload\n\nTopic: `avr/status/light/vio`\n\nThere is no content for this class\n\n#### AvrStatusLightApriltagsPayload\n\nTopic: `avr/status/light/apriltags`\n\nThere is no content for this class\n\n#### AvrStatusLightFcmPayload\n\nTopic: `avr/status/light/fcm`\n\nThere is no content for this class\n\n#### AvrStatusLightThermalPayload\n\nTopic: `avr/status/light/thermal`\n\nThere is no content for this class\n',
    'author': 'Chris Padilla',
    'author_email': 'cpadilla@bellflight.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bellflight/AVR-Python-Libraries',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
