# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lithic', 'lithic.resources', 'lithic.types']

package_data = \
{'': ['*']}

install_requires = \
['anyio>=3.5.0,<4.0.0',
 'httpx>=0.22.0,<0.23.0',
 'pydantic>=1.9.0,<2.0.0',
 'typing-extensions>=4.1.1,<5.0.0']

setup_kwargs = {
    'name': 'lithic',
    'version': '0.1.4',
    'description': 'Client library for the lithic API',
    'long_description': '# Lithic Python API Library\n\nThe Lithic Python library provides convenient access to the Lithic REST API from any Python 3.7+\napplication. It includes type definitions for all request params and response fields,\nand offers both synchronous and asynchronous clients powered by [httpx](https://github.com/encode/httpx).\n\n## Installation\n\n```sh\npip install lithic\n```\n\n## Usage\n\n```python\nfrom lithic import Lithic\n\nclient = Lithic(\n    api_key="my api key", # defaults to os.environ.get("LITHIC_API_KEY")\n    environment="sandbox" # defaults to "production"\n)\n\ncard = client.cards.create({\n    "type": "SINGLE_USE"\n})\n\nprint(card.token)\n```\n\nWhile you can provide an `api_key` keyword argument, we recommend using [python-dotenv](https://pypi.org/project/python-dotenv/)\nand adding `LITHIC_API_KEY="my api key"` to your `.env` file so that your API keys are not stored in source control.\n\n## Async Usage\n\nSimply import `AsyncLithic` instead of `Lithic` and use `await` with each API call:\n\n```python\nfrom lithic import AsyncLithic\nimport asyncio # or the async environment of your choice\n\nclient = AsyncLithic(\n    api_key="my api key", # defaults to os.environ.get("LITHIC_API_KEY")\n    environment="sandbox" # defaults to "production"\n)\n\nasync def main():\n    card = await client.cards.create({\n        "type": "SINGLE_USE"\n    })\n\n    print(card.token)\n\nasyncio.run(main())\n```\n\nFunctionality between the synchronous and asynchronous clients are otherwise identical.\n\n## Using Types\n\nRequest parameters are [TypedDicts](https://docs.python.org/3/library/typing.html#typing.TypedDict), while responses are [Pydantic](https://pydantic-docs.helpmanual.io/) models. This helps provide autocomplete and documentation within your editor.\n\nIf you would like to see type errors in VS Code to help catch bugs earlier, set `python.analysis.typeCheckingMode` to `"basic"`.\n\n## Pagination\n\nList methods in the Lithic API are paginated.\n\nThis library provides auto-paginating iterators with each list response, so you do not have to request successive pages manually:\n\n```python\nimport lithic\nfrom typing import List\n\nclient = lithic.Lithic()\n\nall_cards = []\n# Iterate through items across all pages, issuing requests as needed.\nfor card in client.cards.list():\n    all_cards.append(card)\n```\n\nOr, asynchronously:\n\n```python\nimport lithic\nfrom typing import List\n\nclient = lithic.AsyncLithic()\n\nasync def get_all_cards() -> List[lithic.types.Card]:\n    cards = []\n    # Iterate through items across all pages, issuing requests as needed.\n    async for card in client.cards.list():\n        cards.append(card)\n    return cards\n```\n\nAlternatively, you can use the `.has_next_page()`, `.next_page_params()`,\nor `.get_next_page()` methods for more granular control working with pages:\n\n```python\nfirst_page = await client.cards.list({"page_size": 2})\nif first_page.has_next_page():\n    print("will fetch next page, with params", first_page.next_page_params())\n    next_page = await first_page.get_next_page()\n    print(f"number of items we just fetched: {len(next_page.data)}")\n\n# Remove `await` for non-async usage.\n```\n\nOr just work directly with the returned data:\n\n```python\nfirst_page = await client.cards.list()\n\nprint(f"page number: {first_page.page}") # => "page number: 1"\nfor card in first_page.data:\n    print(card.token)\n\n# Remove `await` for non-async usage.\n```\n\n## Handling errors\n\nWhen the library is unable to connect to the API (e.g., due to network connection problems or a timeout), a subclass of `lithic.APIConnectionError` is raised.\n\nWhen the API returns a non-success status code (i.e., 4xx or 5xx\nresponse), a subclass of `lithic.APIStatusError` will be raised, containing `status_code` and `response` properties.\n\nAll errors inherit from `lithic.APIError`.\n\n```python\nimport lithic\n\nclient = lithic.Lithic()\n\ntry:\n    client.cards.create()\n\nexcept lithic.APIConnectionError as e:\n    print(\'The server could not be reached\')\n    print(e.__cause__) # an underlying Exception, likely raised within httpx.\n\nexcept lithic.RateLimitError as e:\n    print(\'A 429 status code was received; we should back off a bit.\')\n\nexcept lithic.APIStatusError as e:\n    print(\'Another non-200-range status code was received\')\n    print(e.status_code)\n    print(e.response)\n```\n\nError codes are as followed:\n\n| Status Code | Error Type                 |\n| ----------- | -------------------------- |\n| 400         | `BadRequestError`          |\n| 401         | `AuthenticationError`      |\n| 403         | `PermissionDeniedError`    |\n| 404         | `NotFoundError`            |\n| 422         | `UnprocessableEntityError` |\n| 429         | `RateLimitError`           |\n| >=500       | `InternalServerError`      |\n| N/A         | `APIConnectionError`       |\n\n### Retries\n\nCertain errors will be automatically retried 2 times by default, with a short exponential backoff.\nConnection errors (for example, due to a network connectivity problem), 409 Conflict, 429 Rate Limit,\nand >=500 Internal errors will all be retried by default.\n\nYou can use the `max_retries` option to configure or disable this:\n\n```python\nfrom lithic import Lithic\n\n# Configure the default for all requests:\nclient = Lithic(max_retries=0)\n\n# Override per-request:\nclient.cards.list({"page_size": 10}, max_retries=5)\n```\n\n### Timeouts\n\nRequests time out after 60 seconds by default. You can configure this with a `timeout` option,\nwhich accepts a float or an [`httpx.Timeout`](https://www.python-httpx.org/advanced/#fine-tuning-the-configuration):\n\n```python\n# Configure the default for all requests:\nclient = Lithic(\n  timeout=20.0 # default is 60s.\n)\n\n# More granular control:\nclient = Lithic(\n  timeout=httpx.Timeout(60.0, read=5.0, write=10.0, connect=2.0)\n)\n\n# Override per-request:\nclient.cards.list({"page_size": 10}, timeout=5.0)\n```\n\nOn timeout, an `APITimeoutError` is thrown.\n\nNote that requests which time out will be [retried twice by default](#retries).\n\n## Advanced: Configuring custom URLs, proxies, and transports\n\nYou can configure the following keyword arguments when instantiating the client:\n\n```python\nimport httpx\nimport lithic\n\nclient = lithic.Lithic(\n    base_url="http://my.test.server.example.com:8083", # Use a custom base URL\n    proxies="http://my.test.proxy.example.com",\n    transport=httpx.HTTPTransport(local_address="0.0.0.0")\n)\n```\n\nSee the httpx documentation for information about the [`proxies`](https://www.python-httpx.org/advanced/#http-proxying) and [`transport`](https://www.python-httpx.org/advanced/#custom-transports) keyword arguments.\n\n## Status\n\nThis package is in beta. Its internals and interfaces are not stable and subject to change without a major semver bump;\nplease reach out if you rely on any undocumented behavior.\n\nWe are keen for your feedback; please email us at [sdk-feedback@lithic.com](mailto:sdk-feedback@lithic.com) or open an issue with questions,\nbugs, or suggestions.\n\n## Requirements\n\nPython 3.7 or higher.\n',
    'author': 'Lithic',
    'author_email': 'sdk-feedback@lithic.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/lithic-com/lithic-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
