# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tact']

package_data = \
{'': ['*']}

install_requires = \
['DendroPy>=4.4,<5.0', 'click>=7.0,<8.0', 'numpy>=1.17,<2.0', 'scipy>=1.3,<2.0']

entry_points = \
{'console_scripts': ['tact_add_taxa = tact.cli_add_taxa:main',
                     'tact_build_taxonomic_tree = tact.cli_taxonomy:main',
                     'tact_check_results = tact.cli_check_trees:main']}

setup_kwargs = {
    'name': 'tact',
    'version': '0.3.1',
    'description': 'Taxonomic addition for complete trees: Adds tips to a backbone phylogeny using taxonomy simulated with birth-death models',
    'long_description': '# TACT - Taxonomy addition for complete trees\n\n[![PyPI](https://img.shields.io/pypi/v/tact.svg)](https://pypi.org/project/tact/)\n[![Build status](https://github.com/jonchang/tact/workflows/Python%20package/badge.svg)](https://github.com/jonchang/tact/actions)\n\nAdds tips to a backbone phylogeny using taxonomy simulated with birth-death models\n\n# Installation\n\nTACT requires Python 3. When possible, we recommend using the PyPy 3 implementation as it can significantly speed up TACT analyses, particularly on large datasets. In addition, TACT depends on the click, DendroPy, NumPy, and SciPy packages.\n\n## Homebrew\n\nUsing Homebrew is the recommended way to install TACT. [Install Homebrew on macOS](https://brew.sh) or [Install Homebrew on Linux or Windows 10](https://docs.brew.sh/Homebrew-on-Linux). Once Homebrew has been installed, run\n\n    brew install jonchang/biology/tact\n\n## pipx\n\nIf you are unable or unwilling to use Homebrew, the next recommended way to install TACT is via `pipx`. [Install `pipx`](https://pipxproject.github.io/pipx/installation/), then run:\n\n    pipx install tact\n\nIf you have PyPy3 installed (via e.g. `brew install pypy3` or [some other source](https://www.pypy.org/download.html)) and feel particularly adventurous, you can try to install a faster PyPy version using:\n\n    pipx install --python pypy3 tact\n\nNote that this will take much longer to install, and the installation will almost certainly fail unless you have the proper compilers set up. If it succeeds though, you should see a rather dramatic improvement in TACT\'s performance.\n\n## Docker\n\nYou can also try using the Docker image if you can\'t get your Python to cooperate. Install [Docker Desktop](https://www.docker.com/products/docker-desktop) and run the following to download the TACT image:\n\n    docker pull jonchang/tact:latest\n\nThen, run TACT from the container image, giving it access to your current working directory:\n\n    mkdir -p examples\n    cd examples\n    curl -LO https://raw.githubusercontent.com/jonchang/tact/master/examples/Carangaria.csv\n    curl -LO https://raw.githubusercontent.com/jonchang/tact/master/examples/Carangaria.tre\n    docker run -it -v "$(pwd)":/workdir -w /workdir jonchang/tact tact_build_taxonomic_tree Carangaria.csv --output Carangaria.taxonomy.tre\n    docker run -it -v "$(pwd)":/workdir -w /workdir jonchang/tact tact_add_taxa --backbone Carangaria.tre --taxonomy Carangaria.taxonomy.tre --output Carangaria.tacted\n\nNote that this will be much slower than the other options, especially on macOS. Here\'s a screencast of what this looks like:\n\n[![asciicast](https://asciinema.org/a/347571.svg)](https://asciinema.org/a/347571)\n\n## Other\n\nOther ways of installing TACT, including unpacking the tarball somewhere or directly using `pip`, are neither supported nor recommended.\n\n# Example\n\nFiles used are in the [examples](https://github.com/jonchang/tact/tree/master/examples) folder.\n\n```console\ncurl -LO https://raw.githubusercontent.com/jonchang/tact/master/examples/Carangaria.csv\ncurl -LO https://raw.githubusercontent.com/jonchang/tact/master/examples/Carangaria.tre\n```\n\nBuild a taxonomic tree using the provided CSV file. Run `tact_build_taxonomic_tree --help` to see the required format for this file.\n\n```console\n$ tact_build_taxonomic_tree Carangaria.csv --output Carangaria.taxonomy.tre\nOutput written to: Carangaria.taxonomy.tre\n```\n\n`Carangaria.taxonomy.tre` now contains a Newick phylogeny with many polytomies and named nodes indicating relevant taxonomic ranks. Now run the TACT stochastic polytomy resolver algorithm in conjunction with the backbone phylogeny `Caragaria.tre`.\n\n```console\n$ tact_add_taxa --backbone Carangaria.tre --taxonomy Carangaria.taxonomy.tre --output Carangaria.tacted --verbose --verbose\nRates  [####################################]  226/226\nTACT  [####################################]  642/642  Carangaria\n```\n\nThere will be several files created with the prefix `Carangaria.tacted`. These include `newick.tre` and `nexus.tre` (your primary output in the form of Newick and NEXUS format phylogenies), `rates.csv` (estimated diversification rates on the backbone phylogeny), and `log.txt` (extremely verbose output on what TACT is doing and why).\n\nYou should check the TACT results now for any issues:\n\n```console\n$ tact_check_results Carangaria.tacted.newick.tre --backbone Carangaria.tre --taxonomy Carangaria.taxonomy.tre > checkresults.csv\n```\n\nOpen up `checkresults.csv` in your favorite spreadsheet viewer and check the `warnings` column for any issues.\n\n# Contributing\n\nDevelopment on TACT uses [`poetry`](https://poetry.eustace.io/). Simply clone the repository and install:\n\n```console\n$ git clone https://github.com/jonchang/tact.git\n$ cd tact\n$ poetry install\n```\n\nWhen releasing a new version of tact, run its tests and bump its revision like so:\n\n```console\n$ poetry run pytest # optionally with --script-launch-mode=subprocess\n$ poetry version patch # or minor, etc.\n$ git commit -p\n$ git tag VERSION\n$ git push\n```\n\nA GitHub Actions workflow will build and publish the new version on PyPI.\n\n# Citation\n\nTACT is described more fully in its manuscript. If you use TACT, please cite:\n\n* Chang, J., Rabosky, D. L., & Alfaro, M. E. (2019). Estimating diversification rates on incompletely-sampled phylogenies: theoretical concerns and practical solutions. Systematic Biology. doi:[10.1093/sysbio/syz081](https://doi.org/10.1093/sysbio/syz081)\n\nTACT owes its existence to much foundational work in the area of stochastic polytomy resolution, namely PASTIS and CorSiM.\n\n* Thomas, G. H., Hartmann, K., Jetz, W., Joy, J. B., Mimoto, A., & Mooers, A. O. (2013). PASTIS: an R package to facilitate phylogenetic assembly with soft taxonomic inferences. Methods in Ecology and Evolution, 4(11), 1011–1017. doi:[10.1111/2041-210x.12117](https://doi.org/10.1111/2041-210X.12117)\n\n* Cusimano, N., Stadler, T., & Renner, S. S. (2012). A New Method for Handling Missing Species in Diversification Analysis Applicable to Randomly or Nonrandomly Sampled Phylogenies. Systematic Biology, 61(5), 785–792. doi:[10.1093/sysbio/sys031](https://doi.org/10.1093/sysbio/sys031)\n\n# Sponsorship\n\nPlease consider sponsoring the maintenance of TACT via [GitHub Sponsors](https://github.com/sponsors/jonchang).\n',
    'author': 'Jonathan Chang',
    'author_email': 'jonathan.chang@ucla.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jonchang/tact',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<3.9',
}


setup(**setup_kwargs)
