"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomDataset = exports.CustomDatasetInputFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const pre_bundled_pyspark_job_executable_1 = require("../common/pre-bundled-pyspark-job-executable");
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
const prepared_dataset_1 = require("./prepared-dataset");
const constructs_1 = require("constructs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const ara_bucket_1 = require("../ara-bucket");
const synchronous_glue_job_1 = require("../synchronous-glue-job");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const singleton_kms_key_1 = require("../singleton-kms-key");
var CustomDatasetInputFormat;
(function (CustomDatasetInputFormat) {
    CustomDatasetInputFormat["CSV"] = "csv";
    CustomDatasetInputFormat["PARQUET"] = "parquet";
    CustomDatasetInputFormat["JSON"] = "json";
})(CustomDatasetInputFormat = exports.CustomDatasetInputFormat || (exports.CustomDatasetInputFormat = {}));
/**
 * A CustomDataset is a dataset that you need to prepare for the [BatchReplayer](@link BatchReplayer) to generate data.
 * The dataset is transformed into a [PreparedDataset](@link PreparedDataset) by a Glue Job that runs synchronously during the CDK deploy.
 * The Glue job is sized based on the approximate size of the input data or uses autoscaling (max 100) if no data size is provided.
 *
 * The Glue job is applying the following transformations to the input dataset:
 * 1. Read the input dataset based on its format. Currently, it supports data in CSV, JSON and Parquet
 * 2. Group rows into tumbling windows based on the partition range parameter provided.
 * The partition range should be adapted to the data volume and the total dataset time range
 * 3. Convert dates from MM-dd-yyyy HH:mm:ss.SSS to MM-dd-yyyyTHH:mm:ss.SSSZ format and remove null values
 * 4. Write data into the output bucket partitioned by the tumbling window time.
 * For example, one partition for every 5 minutes.
 * 5. Generate a manifest file based on the previous output to be used by the BatchReplayer for generating data
 *
 * The CloudWatch log group is stored as an object parameter to help check any error with the Glue job.
 *
 * Usage example:
 * ```typescript
 * import { CustomDataset, CustomDatasetInputFormat } from './data-generator/custom-dataset';
 *
 * const app = new App();
 * const stack = new Stack(app, 'CustomDatasetStack');
 *
 * const custom = new CustomDataset(stack, 'CustomDataset', {
 *   s3Location: {
 *     bucketName: 'aws-analytics-reference-architecture',
 *     objectKey: 'datasets/custom',
 *   },
 *   inputFormat: CustomDatasetInputFormat.CSV,
 *   datetimeColumn: 'tpep_pickup_datetime',
 *   datetimeColumnsToAdjust: ['tpep_pickup_datetime'],
 *   partitionRange: Duration.minutes(5),
 *   approximateDataSize: 1,
 * });
 *
 * new CfnOutput(this, 'LogGroupName', {
 *   exportName: 'logGroupName,
 *   value: custom.glueJobLogGroup,
 * });
 * ```
 *
 * An example of a custom dataset that can be processed by this construct is available in s3://aws-analytics-reference-architecture/datasets/custom
 */
class CustomDataset extends constructs_1.Construct {
    /**
     * Constructs a new instance of a CustomDataset construct that extends a PreparedDataset
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CustomDatasetProps} props the CustomDataset [properties]{@link CustomDatasetProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        // the source bucket to read data from
        const sourceBucket = aws_s3_1.Bucket.fromBucketAttributes(scope, 'InputBucket', {
            bucketName: props.s3Location.bucketName,
        });
        // the sink bucket used by the Glue job to write the prepared dataset
        const outputBucket = ara_bucket_1.AraBucket.getOrCreate(scope, {
            bucketName: 'custom-dataset',
            serverAccessLogsPrefix: 'custom-dataset',
        });
        const glueRole = new aws_iam_1.Role(scope, 'CustomDatasetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
            // add inline policy to encrypt logs
            inlinePolicies: {
                SecurityConfig: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                "logs:AssociateKmsKey"
                            ],
                            resources: [
                                `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/jobs/*`
                            ],
                        })
                    ]
                })
            }
        });
        // grant permissions on the eS3 buckets to the glue job role
        sourceBucket.grantRead(glueRole, props.s3Location.objectKey + '*');
        outputBucket.grantReadWrite(glueRole, props.s3Location.objectKey + '*');
        // the SSM parameter used to stored the output of the Glue job
        const ssm = new aws_ssm_1.StringParameter(this, 'JobOutputParameter', {
            stringValue: 'minDatetime',
        });
        ssm.grantWrite(glueRole);
        glueRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'));
        // create a KMS key for the Glue security configureation
        const glueKey = singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey');
        // We add a resource policy so the key can be used in Cloudwatch logs
        glueKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [
                new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`)
            ],
            actions: [
                "kms:Encrypt*",
                "kms:Decrypt*",
                "kms:ReEncrypt*",
                "kms:GenerateDataKey*",
                "kms:Describe*"
            ],
            resources: ["*"],
            conditions: { ArnLike: { "kms:EncryptionContext:aws:logs:arn": `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*` } },
        }));
        // the Glue job security configuration following best practices
        const conf = new aws_glue_alpha_1.SecurityConfiguration(this, 'SecurityConfiguration', {
            securityConfigurationName: id,
            cloudWatchEncryption: {
                mode: aws_glue_alpha_1.CloudWatchEncryptionMode.KMS,
                kmsKey: glueKey,
            },
            jobBookmarksEncryption: {
                mode: aws_glue_alpha_1.JobBookmarksEncryptionMode.CLIENT_SIDE_KMS,
                kmsKey: glueKey,
            },
            s3Encryption: {
                mode: aws_glue_alpha_1.S3EncryptionMode.S3_MANAGED,
            }
        });
        // calculate the size of the Glue job based on input data size (1 + 1 DPU per 2GB)
        var workerNum = 9;
        if (props.approximateDataSize !== undefined && props.approximateDataSize > 16) {
            workerNum = Math.ceil(props.approximateDataSize / 2) + 1;
        }
        // Glue job to prepare the dataset
        const glueJob = new synchronous_glue_job_1.SynchronousGlueJob(scope, 'CustomDatasetJob', {
            executable: pre_bundled_pyspark_job_executable_1.PreBundledPysparkJobExecutable.pythonEtl({
                glueVersion: aws_glue_alpha_1.GlueVersion.V3_0,
                pythonVersion: aws_glue_alpha_1.PythonVersion.THREE,
                codePath: 'data-generator/resources/glue/custom-dataset/script.py',
            }),
            defaultArguments: {
                '--s3_input_path': `s3://${props.s3Location.bucketName}/${props.s3Location.objectKey}`,
                '--s3_output_path': `s3://${outputBucket.bucketName}/${props.s3Location.objectKey}`,
                '--input_format': props.inputFormat,
                '--datetime_column': props.datetimeColumn,
                '--partition_range': props.partitionRange.toMinutes().toString(),
                '--ssm_parameter': ssm.parameterName,
                '--enable-auto-scaling': 'true',
            },
            role: glueRole,
            workerCount: props.approximateDataSize ? workerNum : 100,
            workerType: aws_glue_alpha_1.WorkerType.G_1X,
            continuousLogging: {
                enabled: true,
            },
            securityConfiguration: conf,
        });
        this.glueJobLogGroup = glueJob.glueJobLogStream;
        // Get the offset value calculated in the SynchronousGlueJob
        // We cannot rely on the SSM parameter resource created previously because the offset is generated during deploy time
        const getParameter = new custom_resources_1.AwsCustomResource(this, 'GetParameter', {
            onCreate: {
                service: 'SSM',
                action: 'getParameter',
                parameters: {
                    Name: ssm.parameterName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(Date.now().toString()),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [ssm.parameterArn],
            })
        });
        // Add a dependency on the synchronous glue job to only get the value after processing
        getParameter.node.addDependency(glueJob);
        // Create a prepared dataset based on the output of the Glue job
        this.preparedDataset = new prepared_dataset_1.PreparedDataset({
            location: {
                bucketName: outputBucket.bucketName,
                objectKey: props.s3Location.objectKey,
            },
            offset: getParameter.getResponseField('Parameter.Value'),
            manifestLocation: {
                bucketName: outputBucket.bucketName,
                objectKey: props.s3Location.objectKey + '-manifest.csv',
            },
            dateTimeColumnToFilter: props.datetimeColumn,
            dateTimeColumnsToAdjust: props.datetimeColumnsToAdjust,
        });
    }
}
exports.CustomDataset = CustomDataset;
_a = JSII_RTTI_SYMBOL_1;
CustomDataset[_a] = { fqn: "aws-analytics-reference-architecture.CustomDataset", version: "2.7.2" };
//# sourceMappingURL=data:application/json;base64,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