"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_iam_1 = require("monocdk/aws-iam");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * IAM policies do not grant full access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_iam_1.CfnPolicy || node instanceof aws_iam_1.CfnManagedPolicy) {
        if (checkDocument(node, node.policyDocument)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_iam_1.CfnGroup || node instanceof aws_iam_1.CfnRole) {
        if (node.policies != undefined && checkDocument(node, node.policies)) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function for parsing through the policy document
 * @param node the CfnResource to Check
 * @param policyDoc the JSON policy document
 * @returns boolean
 */
function checkDocument(node, policyDoc) {
    const resolvedDoc = monocdk_1.Stack.of(node).resolve(policyDoc);
    const reg = /"Action":\[?(.*,)?"(?:\w+:)?\*"(,.*)?\]?,"Effect":"Allow"/gm;
    if (JSON.stringify(resolvedDoc).search(reg) != -1) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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