"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PCIDSS321Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const codebuild_1 = require("../rules/codebuild");
const dms_1 = require("../rules/dms");
const ec2_1 = require("../rules/ec2");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for PCI DSS 3.2.1 compliance. Based on the PCI DSS 3.2.1 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-pci-dss.html.
 *
 * @stability stable
 */
class PCIDSS321Checks extends nag_pack_1.NagPack {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        this.packName = 'PCI.DSS.321';
    }
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkEC2(node);
            this.checkECS(node);
            this.checkEFS(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSecretsManager(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL - (Control ID: 6.6).',
            explanation: 'AWS WAF enables you to configure a set of rules (called a web access control list (web ACL)) that allow, block, or count web requests based on customizable web security rules and conditions that you define. Ensure your Amazon API Gateway stage is associated with a WAF Web ACL to protect it from malicious attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control ID: 3.4).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWSSLEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group (which is associated with a load balancer) does not utilize ELB healthchecks - (Control ID: 2.2).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
        this.applyRule({
            info: 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingLaunchConfigPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.3, 10.5.4).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have encryption enabled - (Control IDs: 2.2, 3.4, 10.5).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control IDs: 2.2, 10.5.2, 10.5, 10.5.5, 11.5).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control ID: 3.4).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: 3.1, 10.7).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control ID: 8.2.1).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectEnvVarAwsCred,
            node: node,
        });
        this.applyRule({
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: 8.2.1).',
            explanation: 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectSourceRepoUrl,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Elastic Compute Cloud (Amazon EC2) instances cannot be publicly accessed. Amazon EC2 instances can contain sensitive information and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have an instance profile attached - (Control IDs: 2.2, 7.1.1, 7.2.1).',
            explanation: 'EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceProfileAttached,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Deploy Amazon Elastic Compute Cloud (Amazon EC2) instances within an Amazon Virtual Private Cloud (Amazon VPC) to enable secure communication between an instance and other services within the amazon VPC, without requiring an internet gateway, NAT device, or VPN connection. All traffic remains securely within the AWS Cloud. Because of their logical isolation, domains that reside within anAmazon VPC have an extra layer of security when compared to domains that use public endpoints. Assign Amazon EC2 instances to an Amazon VPC to properly manage access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333) - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2, 2.2.2).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check ECS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkECS(node) {
        this.applyRule({
            info: "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control ID: 7.2.1).",
            explanation: 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionUserForHostMode,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 4.1, 8.2.1).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 2.3, 4.1, 8.2.1).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control ID: 6.6).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 4.1, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 2.3, 4.1, 8.2.1).',
            explanation: 'Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBv2ACMCertificateRequired,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            info: 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control ID: 7.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group does not have at least one IAM User - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMGroupHasUsers,
            node: node,
        });
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants full access - (Control IDs: 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithFullAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: 2.2, 7.1.2, 7.1.3, 7.2.1, 7.2.2).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control IDs: 2.2, 3.5, 3.6, 3.6.4).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 2.2.2).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchErrorLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control ID: 4.1).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control ID: 6.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSAutomaticMinorVersionUpgradeEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance does not have all CloudWatch log types exported - (Control IDs: 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB Instance or Aurora Cluster does not have storage encrypted - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS instances, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 3.4, 8.2.1, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control ID: 6.2).',
            explanation: 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.                                                                                                                                                                                                                                                                                                                                                              ',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterMaintenanceSettings,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 1.2, 1.3, 1.3.1, 1.3.2).',
            explanation: 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftEnhancedVPCRoutingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: 2.3, 4.1).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Buckets does not have server access logs enabled - (Control IDs: 2.2, 10.1, 10.2.1, 10.2.2, 10.2.3, 10.2.4, 10.2.5, 10.2.7, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2, 2.2.2).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: 2.2, 10.5.3).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 2.2, 3.4, 8.2.1, 10.5).',
            explanation: 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control ID: 10.5.3).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 3.4, 8.2.1, 10.5).',
            explanation: 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3DefaultEncryptionKMS,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker resource endpoint is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Secrets Manager Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node) {
        this.applyRule({
            info: 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 3.4, 8.2.1).',
            explanation: 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerUsingKMSKey,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control ID: 8.2.1).',
            explanation: 'To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS) Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.1, 2.2, 2.2.2).",
            explanation: 'Amazon Elastic Compute Cloud (Amazon EC2) security groups can help in the management of network access by providing stateful filtering of ingress and egress network traffic to AWS resources. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: 2.2, 10.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: "The route table may contain one or more unrestricted route(s) to an IGW ('0.0.0.0/0' or '::/0') - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 2.2.2).",
            explanation: 'Ensure Amazon EC2 route tables do not have unrestricted routes to an internet gateway. Removing or limiting the access to the internet for workloads within Amazon VPCs can reduce unintended access within your environment.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCNoUnrestrictedRouteToIGW,
            node: node,
        });
        this.applyRule({
            info: 'The subnet auto-assigns public IP addresses - (Control IDs: 1.2, 1.2.1, 1.3, 1.3.1, 1.3.2, 1.3.4, 1.3.6, 2.2.2).',
            explanation: 'Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCSubnetAutoAssignPublicIpDisabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: 10.1, 10.3.1, 10.3.2, 10.3.3, 10.3.4, 10.3.5, 10.3.6, 10.5.4).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.PCIDSS321Checks = PCIDSS321Checks;
_a = JSII_RTTI_SYMBOL_1;
PCIDSS321Checks[_a] = { fqn: "monocdk-nag.PCIDSS321Checks", version: "1.3.5" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGNpLWRzcy0zMjEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvcGFja3MvcGNpLWRzcy0zMjEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFFRixxQ0FBa0Q7QUFDbEQsMENBQXFFO0FBQ3JFLDBDQUt3QjtBQUN4QixzREFHOEI7QUFDOUIsb0RBSTZCO0FBQzdCLG9EQUc2QjtBQUM3QixrREFHNEI7QUFDNUIsc0NBQXVEO0FBQ3ZELHNDQU1zQjtBQUN0QixzQ0FBZ0U7QUFDaEUsc0NBQTRDO0FBQzVDLHNDQVFzQjtBQUN0QixzQ0FBa0Q7QUFDbEQsc0NBT3NCO0FBQ3RCLHNDQUE0RDtBQUM1RCw0Q0FBa0Q7QUFDbEQsb0RBSzZCO0FBQzdCLHNDQUtzQjtBQUN0QixnREFNMkI7QUFDM0Isb0NBU3FCO0FBQ3JCLGtEQUk0QjtBQUM1Qiw0REFBb0U7QUFDcEUsc0NBQStDO0FBQy9DLHNDQUtzQjtBQUN0QixzQ0FBbUQ7Ozs7OztBQUduRCxNQUFhLGVBQWdCLFNBQVEsa0JBQU87Ozs7SUFDMUMsWUFBWSxLQUFvQjtRQUM5QixLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDYixJQUFJLENBQUMsUUFBUSxHQUFHLGFBQWEsQ0FBQztJQUNoQyxDQUFDOzs7Ozs7SUFDTSxLQUFLLENBQUMsSUFBZ0I7UUFDM0IsSUFBSSxJQUFJLFlBQVkscUJBQVcsRUFBRTtZQUMvQixJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM1QixJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzNCLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDM0IsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMxQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3BCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNwQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3BCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNwQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3BCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN2QixJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzNCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN6QixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ25CLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDMUIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQy9CLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNwQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3JCO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxVQUFVLENBQUMsSUFBaUI7UUFDbEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxrRkFBa0Y7WUFDeEYsV0FBVyxFQUNULDRUQUE0VDtZQUM5VCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4QkFBc0I7WUFDNUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHdHQUF3RztZQUM5RyxXQUFXLEVBQ1QsZ05BQWdOO1lBQ2xOLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFDQUE2QjtZQUNuQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsOEpBQThKO1lBQ3BLLFdBQVcsRUFDVCxrS0FBa0s7WUFDcEssS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsb0NBQTRCO1lBQ2xDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwwR0FBMEc7WUFDaEgsV0FBVyxFQUNULG1LQUFtSztZQUNySyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1QkFBZTtZQUNyQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZ0JBQWdCLENBQUMsSUFBaUI7UUFDeEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwwSEFBMEg7WUFDaEksV0FBVyxFQUNULGtaQUFrWjtZQUNwWixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxvREFBc0M7WUFDNUMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHVKQUF1SjtZQUM3SixXQUFXLEVBQ1QscVJBQXFSO1lBQ3ZSLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFEQUF1QztZQUM3QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLElBQWlCO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsNktBQTZLO1lBQ25MLFdBQVcsRUFDVCwyS0FBMks7WUFDN0ssS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNENBQStCO1lBQ3JDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw2RUFBNkU7WUFDbkYsV0FBVyxFQUNULGlJQUFpSTtZQUNuSSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx3Q0FBMkI7WUFDakMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHVHQUF1RztZQUM3RyxXQUFXLEVBQ1QsaWJBQWliO1lBQ25iLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLCtDQUFrQztZQUN4QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLElBQWlCO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsb0ZBQW9GO1lBQzFGLFdBQVcsRUFDVCw2R0FBNkc7WUFDL0csS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0NBQTJCO1lBQ2pDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw0R0FBNEc7WUFDbEgsV0FBVyxFQUNULCtQQUErUDtZQUNqUSxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4Q0FBaUM7WUFDdkMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQjtRQUN0QyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLDJLQUEySztZQUNqTCxXQUFXLEVBQ1QsOElBQThJO1lBQ2hKLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHlDQUE2QjtZQUNuQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsbUlBQW1JO1lBQ3pJLFdBQVcsRUFDVCw4TkFBOE47WUFDaE8sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUseUNBQTZCO1lBQ25DLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUI7UUFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw2R0FBNkc7WUFDbkgsV0FBVyxFQUNULCtHQUErRztZQUNqSCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw2QkFBdUI7WUFDN0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQjtRQUNoQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLDhIQUE4SDtZQUNwSSxXQUFXLEVBQ1Qsc09BQXNPO1lBQ3hPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDJCQUFxQjtZQUMzQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsaUdBQWlHO1lBQ3ZHLFdBQVcsRUFDVCx3S0FBd0s7WUFDMUssS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsZ0NBQTBCO1lBQ2hDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwyR0FBMkc7WUFDakgsV0FBVyxFQUNULDhpQkFBOGlCO1lBQ2hqQixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1QkFBaUI7WUFDdkIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHVOQUF1TjtZQUM3TixXQUFXLEVBQ1QsbVBBQW1QO1lBQ3JQLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhCQUF3QjtZQUM5QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsK0dBQStHO1lBQ3JILFdBQVcsRUFDVCwrSEFBK0g7WUFDakksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsc0JBQWdCO1lBQ3RCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUI7UUFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSx5TUFBeU07WUFDL00sV0FBVyxFQUNULHNSQUFzUjtZQUN4UixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxzQ0FBZ0M7WUFDdEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQjtRQUNoQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLCtFQUErRTtZQUNyRixXQUFXLEVBQ1QsNElBQTRJO1lBQzlJLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGtCQUFZO1lBQ2xCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUI7UUFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSx5RkFBeUY7WUFDL0YsV0FBVyxFQUNULGtMQUFrTDtZQUNwTCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxxQ0FBK0I7WUFDckMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLG9HQUFvRztZQUMxRyxXQUFXLEVBQ1QsbU9BQW1PO1lBQ3JPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLCtCQUF5QjtZQUMvQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsdUVBQXVFO1lBQzdFLFdBQVcsRUFDVCw0TUFBNE07WUFDOU0sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsbUJBQWE7WUFDbkIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHVIQUF1SDtZQUM3SCxXQUFXLEVBQ1QsbVJBQW1SO1lBQ3JSLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLCtCQUF5QjtZQUMvQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsc0hBQXNIO1lBQzVILFdBQVcsRUFDVCw4VkFBOFY7WUFDaFcsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsdUJBQWlCO1lBQ3ZCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwrR0FBK0c7WUFDckgsV0FBVyxFQUNULHNMQUFzTDtZQUN4TCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw4QkFBd0I7WUFDOUIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHFJQUFxSTtZQUMzSSxXQUFXLEVBQ1QsbVJBQW1SO1lBQ3JSLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGlDQUEyQjtZQUNqQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCO1FBQ2hDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsdUlBQXVJO1lBQzdJLFdBQVcsRUFDVCxtUkFBbVI7WUFDclIsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0JBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUI7UUFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxnR0FBZ0c7WUFDdEcsV0FBVyxFQUNULCtWQUErVjtZQUNqVyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxzQkFBZ0I7WUFDdEIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLDBHQUEwRztZQUNoSCxXQUFXLEVBQ1QsZ0xBQWdMO1lBQ2xMLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHVCQUFpQjtZQUN2QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsc0ZBQXNGO1lBQzVGLFdBQVcsRUFDVCwrVkFBK1Y7WUFDalcsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMENBQW9DO1lBQzFDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxnRkFBZ0Y7WUFDdEYsV0FBVyxFQUNULHVOQUF1TjtZQUN6TixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx5Q0FBbUM7WUFDekMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLGdHQUFnRztZQUN0RyxXQUFXLEVBQ1Qsb1NBQW9TO1lBQ3RTLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDRCQUFzQjtZQUM1QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsZ0dBQWdHO1lBQ3RHLFdBQVcsRUFDVCw0SUFBNEk7WUFDOUksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsdUJBQWlCO1lBQ3ZCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUI7UUFDaEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwyR0FBMkc7WUFDakgsV0FBVyxFQUNULDRHQUE0RztZQUM5RyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxrQ0FBNEI7WUFDbEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFdBQVcsQ0FBQyxJQUFpQjtRQUNuQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHNHQUFzRztZQUM1RyxXQUFXLEVBQ1Qsa0tBQWtLO1lBQ3BLLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdCQUFlO1lBQ3JCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsSUFBaUI7UUFDdkMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxxR0FBcUc7WUFDM0csV0FBVyxFQUNULGtLQUFrSztZQUNwSyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxzQ0FBeUI7WUFDL0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLGdJQUFnSTtZQUN0SSxXQUFXLEVBQ1QsK0VBQStFO1lBQ2pGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGdDQUFtQjtZQUN6QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsa05BQWtOO1lBQ3hOLFdBQVcsRUFDVCxpUEFBaVA7WUFDblAsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNENBQStCO1lBQ3JDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxrR0FBa0c7WUFDeEcsV0FBVyxFQUNULDhKQUE4SjtZQUNoSyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwyQ0FBOEI7WUFDcEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQjtRQUNoQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHNIQUFzSDtZQUM1SCxXQUFXLEVBQ1QsK05BQStOO1lBQ2pPLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDRDQUFzQztZQUM1QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsK0dBQStHO1lBQ3JILFdBQVcsRUFDVCxtSUFBbUk7WUFDckksS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNkJBQXVCO1lBQzdCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxvTEFBb0w7WUFDMUwsV0FBVyxFQUNULDBQQUEwUDtZQUM1UCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx1QkFBaUI7WUFDdkIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLG9HQUFvRztZQUMxRyxXQUFXLEVBQ1Qsd0hBQXdIO1lBQzFILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHlCQUFtQjtZQUN6QixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssYUFBYSxDQUFDLElBQWlCO1FBQ3JDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsbU1BQW1NO1lBQ3pNLFdBQVcsRUFDVCw2U0FBNlM7WUFDL1MsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsdUNBQTRCO1lBQ2xDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSwrS0FBK0s7WUFDckwsV0FBVyxFQUNULDZpQkFBNmlCO1lBQy9pQixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSw2Q0FBa0M7WUFDeEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLGdIQUFnSDtZQUN0SCxXQUFXLEVBQ1QsNkhBQTZIO1lBQy9ILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHNDQUEyQjtZQUNqQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsMEdBQTBHO1lBQ2hILFdBQVcsRUFDVCwrU0FBK1M7WUFDalQsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsNENBQWlDO1lBQ3ZDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSxxRkFBcUY7WUFDM0YsV0FBVyxFQUNULDJMQUEyTDtZQUM3TCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxnQ0FBcUI7WUFDM0IsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLE9BQU8sQ0FBQyxJQUFpQjtRQUMvQixJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLGtKQUFrSjtZQUN4SixXQUFXLEVBQ1QsMEdBQTBHO1lBQzVHLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHdDQUFtQztZQUN6QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUscUxBQXFMO1lBQzNMLFdBQVcsRUFDVCxrY0FBa2M7WUFDcGMsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkJBQXNCO1lBQzVCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw2TEFBNkw7WUFDbk0sV0FBVyxFQUNULG9GQUFvRjtZQUN0RixLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxpQ0FBNEI7WUFDbEMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLDhMQUE4TDtZQUNwTSxXQUFXLEVBQ1Qsb0ZBQW9GO1lBQ3RGLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLGtDQUE2QjtZQUNuQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsK0VBQStFO1lBQ3JGLFdBQVcsRUFDVCx3UUFBd1E7WUFDMVEsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsK0JBQTBCO1lBQ2hDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw0R0FBNEc7WUFDbEgsV0FBVyxFQUNULDZHQUE2RztZQUMvRyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSx3Q0FBbUM7WUFDekMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHdFQUF3RTtZQUM5RSxXQUFXLEVBQ1QsZ05BQWdOO1lBQ2xOLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDhCQUF5QjtZQUMvQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsNkZBQTZGO1lBQ25HLFdBQVcsRUFDVCxxTkFBcU47WUFDdk4sS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsMkJBQXNCO1lBQzVCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxjQUFjLENBQUMsSUFBaUI7UUFDdEMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSw4RkFBOEY7WUFDcEcsV0FBVyxFQUNULHNIQUFzSDtZQUN4SCxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSwwREFBOEM7WUFDcEQsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHFGQUFxRjtZQUMzRixXQUFXLEVBQ1Qsc0hBQXNIO1lBQ3hILEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHFEQUF5QztZQUMvQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUscUlBQXFJO1lBQzNJLFdBQVcsRUFDVCw4R0FBOEc7WUFDaEgsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsbURBQXVDO1lBQzdDLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxtQkFBbUIsQ0FBQyxJQUFpQjtRQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLDBGQUEwRjtZQUNoRyxXQUFXLEVBQ1QsaVFBQWlRO1lBQ25RLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLDBDQUF5QjtZQUMvQixJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCO1FBQ2hDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsMkVBQTJFO1lBQ2pGLFdBQVcsRUFDVCwwUkFBMFI7WUFDNVIsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUscUJBQWU7WUFDckIsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQjtRQUNoQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLHNJQUFzSTtZQUM1SSxXQUFXLEVBQ1Qsb1RBQW9UO1lBQ3RULEtBQUssRUFBRSwwQkFBZSxDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLG1DQUE2QjtZQUNuQyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsa0hBQWtIO1lBQ3hILFdBQVcsRUFDVCx1VEFBdVQ7WUFDelQsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUsd0JBQWtCO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNiLElBQUksRUFBRSx3SkFBd0o7WUFDOUosV0FBVyxFQUNULCtOQUErTjtZQUNqTyxLQUFLLEVBQUUsMEJBQWUsQ0FBQyxLQUFLO1lBQzVCLElBQUksRUFBRSxpQ0FBMkI7WUFDakMsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2IsSUFBSSxFQUFFLGtIQUFrSDtZQUN4SCxXQUFXLEVBQ1QsOFRBQThUO1lBQ2hVLEtBQUssRUFBRSwwQkFBZSxDQUFDLEtBQUs7WUFDNUIsSUFBSSxFQUFFLHlDQUFtQztZQUN6QyxJQUFJLEVBQUUsSUFBSTtTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCO1FBQ2hDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDYixJQUFJLEVBQUUsZ0lBQWdJO1lBQ3RJLFdBQVcsRUFDVCwyUUFBMlE7WUFDN1EsS0FBSyxFQUFFLDBCQUFlLENBQUMsS0FBSztZQUM1QixJQUFJLEVBQUUseUJBQW1CO1lBQ3pCLElBQUksRUFBRSxJQUFJO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUF2dkJILDBDQXd2QkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cblxuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIElDb25zdHJ1Y3QgfSBmcm9tICdtb25vY2RrJztcbmltcG9ydCB7IE5hZ01lc3NhZ2VMZXZlbCwgTmFnUGFjaywgTmFnUGFja1Byb3BzIH0gZnJvbSAnLi4vbmFnLXBhY2snO1xuaW1wb3J0IHtcbiAgQVBJR1dBc3NvY2lhdGVkV2l0aFdBRixcbiAgQVBJR1dDYWNoZUVuYWJsZWRBbmRFbmNyeXB0ZWQsXG4gIEFQSUdXRXhlY3V0aW9uTG9nZ2luZ0VuYWJsZWQsXG4gIEFQSUdXU1NMRW5hYmxlZCxcbn0gZnJvbSAnLi4vcnVsZXMvYXBpZ3cnO1xuaW1wb3J0IHtcbiAgQXV0b1NjYWxpbmdHcm91cEVMQkhlYWx0aENoZWNrUmVxdWlyZWQsXG4gIEF1dG9TY2FsaW5nTGF1bmNoQ29uZmlnUHVibGljSXBEaXNhYmxlZCxcbn0gZnJvbSAnLi4vcnVsZXMvYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHtcbiAgQ2xvdWRUcmFpbENsb3VkV2F0Y2hMb2dzRW5hYmxlZCxcbiAgQ2xvdWRUcmFpbEVuY3J5cHRpb25FbmFibGVkLFxuICBDbG91ZFRyYWlsTG9nRmlsZVZhbGlkYXRpb25FbmFibGVkLFxufSBmcm9tICcuLi9ydWxlcy9jbG91ZHRyYWlsJztcbmltcG9ydCB7XG4gIENsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZCxcbiAgQ2xvdWRXYXRjaExvZ0dyb3VwUmV0ZW50aW9uUGVyaW9kLFxufSBmcm9tICcuLi9ydWxlcy9jbG91ZHdhdGNoJztcbmltcG9ydCB7XG4gIENvZGVCdWlsZFByb2plY3RFbnZWYXJBd3NDcmVkLFxuICBDb2RlQnVpbGRQcm9qZWN0U291cmNlUmVwb1VybCxcbn0gZnJvbSAnLi4vcnVsZXMvY29kZWJ1aWxkJztcbmltcG9ydCB7IERNU1JlcGxpY2F0aW9uTm90UHVibGljIH0gZnJvbSAnLi4vcnVsZXMvZG1zJztcbmltcG9ydCB7XG4gIEVDMkluc3RhbmNlTm9QdWJsaWNJcCxcbiAgRUMySW5zdGFuY2VQcm9maWxlQXR0YWNoZWQsXG4gIEVDMkluc3RhbmNlc0luVlBDLFxuICBFQzJSZXN0cmljdGVkQ29tbW9uUG9ydHMsXG4gIEVDMlJlc3RyaWN0ZWRTU0gsXG59IGZyb20gJy4uL3J1bGVzL2VjMic7XG5pbXBvcnQgeyBFQ1NUYXNrRGVmaW5pdGlvblVzZXJGb3JIb3N0TW9kZSB9IGZyb20gJy4uL3J1bGVzL2Vjcyc7XG5pbXBvcnQgeyBFRlNFbmNyeXB0ZWQgfSBmcm9tICcuLi9ydWxlcy9lZnMnO1xuaW1wb3J0IHtcbiAgQUxCSHR0cERyb3BJbnZhbGlkSGVhZGVyRW5hYmxlZCxcbiAgQUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbixcbiAgQUxCV0FGRW5hYmxlZCxcbiAgRUxCQUNNQ2VydGlmaWNhdGVSZXF1aXJlZCxcbiAgRUxCTG9nZ2luZ0VuYWJsZWQsXG4gIEVMQlRsc0h0dHBzTGlzdGVuZXJzT25seSxcbiAgRUxCdjJBQ01DZXJ0aWZpY2F0ZVJlcXVpcmVkLFxufSBmcm9tICcuLi9ydWxlcy9lbGInO1xuaW1wb3J0IHsgRU1SS2VyYmVyb3NFbmFibGVkIH0gZnJvbSAnLi4vcnVsZXMvZW1yJztcbmltcG9ydCB7XG4gIElBTUdyb3VwSGFzVXNlcnMsXG4gIElBTU5vSW5saW5lUG9saWN5LFxuICBJQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoQWRtaW5BY2Nlc3MsXG4gIElBTVBvbGljeU5vU3RhdGVtZW50c1dpdGhGdWxsQWNjZXNzLFxuICBJQU1Vc2VyR3JvdXBNZW1iZXJzaGlwLFxuICBJQU1Vc2VyTm9Qb2xpY2llcyxcbn0gZnJvbSAnLi4vcnVsZXMvaWFtJztcbmltcG9ydCB7IEtNU0JhY2tpbmdLZXlSb3RhdGlvbkVuYWJsZWQgfSBmcm9tICcuLi9ydWxlcy9rbXMnO1xuaW1wb3J0IHsgTGFtYmRhSW5zaWRlVlBDIH0gZnJvbSAnLi4vcnVsZXMvbGFtYmRhJztcbmltcG9ydCB7XG4gIE9wZW5TZWFyY2hFbmNyeXB0ZWRBdFJlc3QsXG4gIE9wZW5TZWFyY2hFcnJvckxvZ3NUb0Nsb3VkV2F0Y2gsXG4gIE9wZW5TZWFyY2hJblZQQ09ubHksXG4gIE9wZW5TZWFyY2hOb2RlVG9Ob2RlRW5jcnlwdGlvbixcbn0gZnJvbSAnLi4vcnVsZXMvb3BlbnNlYXJjaCc7XG5pbXBvcnQge1xuICBSRFNBdXRvbWF0aWNNaW5vclZlcnNpb25VcGdyYWRlRW5hYmxlZCxcbiAgUkRTSW5zdGFuY2VQdWJsaWNBY2Nlc3MsXG4gIFJEU0xvZ2dpbmdFbmFibGVkLFxuICBSRFNTdG9yYWdlRW5jcnlwdGVkLFxufSBmcm9tICcuLi9ydWxlcy9yZHMnO1xuaW1wb3J0IHtcbiAgUmVkc2hpZnRDbHVzdGVyQ29uZmlndXJhdGlvbixcbiAgUmVkc2hpZnRDbHVzdGVyTWFpbnRlbmFuY2VTZXR0aW5ncyxcbiAgUmVkc2hpZnRDbHVzdGVyUHVibGljQWNjZXNzLFxuICBSZWRzaGlmdEVuaGFuY2VkVlBDUm91dGluZ0VuYWJsZWQsXG4gIFJlZHNoaWZ0UmVxdWlyZVRsc1NTTCxcbn0gZnJvbSAnLi4vcnVsZXMvcmVkc2hpZnQnO1xuaW1wb3J0IHtcbiAgUzNCdWNrZXRMZXZlbFB1YmxpY0FjY2Vzc1Byb2hpYml0ZWQsXG4gIFMzQnVja2V0TG9nZ2luZ0VuYWJsZWQsXG4gIFMzQnVja2V0UHVibGljUmVhZFByb2hpYml0ZWQsXG4gIFMzQnVja2V0UHVibGljV3JpdGVQcm9oaWJpdGVkLFxuICBTM0J1Y2tldFJlcGxpY2F0aW9uRW5hYmxlZCxcbiAgUzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQsXG4gIFMzQnVja2V0VmVyc2lvbmluZ0VuYWJsZWQsXG4gIFMzRGVmYXVsdEVuY3J5cHRpb25LTVMsXG59IGZyb20gJy4uL3J1bGVzL3MzJztcbmltcG9ydCB7XG4gIFNhZ2VNYWtlckVuZHBvaW50Q29uZmlndXJhdGlvbktNU0tleUNvbmZpZ3VyZWQsXG4gIFNhZ2VNYWtlck5vdGVib29rSW5zdGFuY2VLTVNLZXlDb25maWd1cmVkLFxuICBTYWdlTWFrZXJOb3RlYm9va05vRGlyZWN0SW50ZXJuZXRBY2Nlc3MsXG59IGZyb20gJy4uL3J1bGVzL3NhZ2VtYWtlcic7XG5pbXBvcnQgeyBTZWNyZXRzTWFuYWdlclVzaW5nS01TS2V5IH0gZnJvbSAnLi4vcnVsZXMvc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHsgU05TRW5jcnlwdGVkS01TIH0gZnJvbSAnLi4vcnVsZXMvc25zJztcbmltcG9ydCB7XG4gIFZQQ0RlZmF1bHRTZWN1cml0eUdyb3VwQ2xvc2VkLFxuICBWUENGbG93TG9nc0VuYWJsZWQsXG4gIFZQQ05vVW5yZXN0cmljdGVkUm91dGVUb0lHVyxcbiAgVlBDU3VibmV0QXV0b0Fzc2lnblB1YmxpY0lwRGlzYWJsZWQsXG59IGZyb20gJy4uL3J1bGVzL3ZwYyc7XG5pbXBvcnQgeyBXQUZ2MkxvZ2dpbmdFbmFibGVkIH0gZnJvbSAnLi4vcnVsZXMvd2FmJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBQQ0lEU1MzMjFDaGVja3MgZXh0ZW5kcyBOYWdQYWNrIHtcbiAgY29uc3RydWN0b3IocHJvcHM/OiBOYWdQYWNrUHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gICAgdGhpcy5wYWNrTmFtZSA9ICdQQ0kuRFNTLjMyMSc7XG4gIH1cbiAgcHVibGljIHZpc2l0KG5vZGU6IElDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIENmblJlc291cmNlKSB7XG4gICAgICB0aGlzLmNoZWNrQVBJR1cobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrQXV0b1NjYWxpbmcobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrQ2xvdWRUcmFpbChub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tDbG91ZFdhdGNoKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja0NvZGVCdWlsZChub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tETVMobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrRUMyKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja0VDUyhub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tFRlMobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrRUxCKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja0VNUihub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tJQU0obm9kZSk7XG4gICAgICB0aGlzLmNoZWNrS01TKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja0xhbWJkYShub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tPcGVuU2VhcmNoKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja1JEUyhub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tSZWRzaGlmdChub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tTMyhub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tTYWdlTWFrZXIobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrU2VjcmV0c01hbmFnZXIobm9kZSk7XG4gICAgICB0aGlzLmNoZWNrU05TKG5vZGUpO1xuICAgICAgdGhpcy5jaGVja1ZQQyhub2RlKTtcbiAgICAgIHRoaXMuY2hlY2tXQUYobm9kZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEFQSSBHYXRld2F5IFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tBUElHVyhub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUkVTVCBBUEkgc3RhZ2UgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBBV1MgV0FGdjIgd2ViIEFDTCAtIChDb250cm9sIElEOiA2LjYpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FXUyBXQUYgZW5hYmxlcyB5b3UgdG8gY29uZmlndXJlIGEgc2V0IG9mIHJ1bGVzIChjYWxsZWQgYSB3ZWIgYWNjZXNzIGNvbnRyb2wgbGlzdCAod2ViIEFDTCkpIHRoYXQgYWxsb3csIGJsb2NrLCBvciBjb3VudCB3ZWIgcmVxdWVzdHMgYmFzZWQgb24gY3VzdG9taXphYmxlIHdlYiBzZWN1cml0eSBydWxlcyBhbmQgY29uZGl0aW9ucyB0aGF0IHlvdSBkZWZpbmUuIEVuc3VyZSB5b3VyIEFtYXpvbiBBUEkgR2F0ZXdheSBzdGFnZSBpcyBhc3NvY2lhdGVkIHdpdGggYSBXQUYgV2ViIEFDTCB0byBwcm90ZWN0IGl0IGZyb20gbWFsaWNpb3VzIGF0dGFja3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBUElHV0Fzc29jaWF0ZWRXaXRoV0FGLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIEFQSSBHYXRld2F5IHN0YWdlIGRvZXMgbm90IGhhdmUgY2FjaGluZyBlbmFibGVkIGFuZCBlbmNyeXB0ZWQgZm9yIGFsbCBtZXRob2RzIC0gKENvbnRyb2wgSUQ6IDMuNCkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICBcIlRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgQVBJIEdhdGV3YXkgc3RhZ2UncyBjYWNoZS4gQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gYmUgY2FwdHVyZWQgZm9yIHRoZSBBUEkgbWV0aG9kLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuXCIsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQVBJR1dDYWNoZUVuYWJsZWRBbmRFbmNyeXB0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgQVBJIEdhdGV3YXkgc3RhZ2UgZG9lcyBub3QgaGF2ZSBleGVjdXRpb24gbG9nZ2luZyBlbmFibGVkIGZvciBhbGwgbWV0aG9kcyAtIChDb250cm9sIElEczogMTAuMSwgMTAuMy4xLCAxMC4zLjIsIDEwLjMuMywgMTAuMy40LCAxMC4zLjUsIDEwLjMuNiwgMTAuNS40KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBUEkgR2F0ZXdheSBsb2dnaW5nIGRpc3BsYXlzIGRldGFpbGVkIHZpZXdzIG9mIHVzZXJzIHdobyBhY2Nlc3NlZCB0aGUgQVBJIGFuZCB0aGUgd2F5IHRoZXkgYWNjZXNzZWQgdGhlIEFQSS4gVGhpcyBpbnNpZ2h0IGVuYWJsZXMgdmlzaWJpbGl0eSBvZiB1c2VyIGFjdGl2aXRpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBUElHV0V4ZWN1dGlvbkxvZ2dpbmdFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIEFQSSBHYXRld2F5IFJFU1QgQVBJIHN0YWdlIGlzIG5vdCBjb25maWd1cmVkIHdpdGggU1NMIGNlcnRpZmljYXRlcyAtIChDb250cm9sIElEczogMi4zLCA0LjEsIDguMi4xKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbnN1cmUgQW1hem9uIEFQSSBHYXRld2F5IFJFU1QgQVBJIHN0YWdlcyBhcmUgY29uZmlndXJlZCB3aXRoIFNTTCBjZXJ0aWZpY2F0ZXMgdG8gYWxsb3cgYmFja2VuZCBzeXN0ZW1zIHRvIGF1dGhlbnRpY2F0ZSB0aGF0IHJlcXVlc3RzIG9yaWdpbmF0ZSBmcm9tIEFQSSBHYXRld2F5LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQVBJR1dTU0xFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBdXRvIFNjYWxpbmcgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0F1dG9TY2FsaW5nKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgKHdoaWNoIGlzIGFzc29jaWF0ZWQgd2l0aCBhIGxvYWQgYmFsYW5jZXIpIGRvZXMgbm90IHV0aWxpemUgRUxCIGhlYWx0aGNoZWNrcyAtIChDb250cm9sIElEOiAyLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBFbGFzdGljIExvYWQgQmFsYW5jZXIgKEVMQikgaGVhbHRoIGNoZWNrcyBmb3IgQW1hem9uIEVsYXN0aWMgQ29tcHV0ZSBDbG91ZCAoQW1hem9uIEVDMikgQXV0byBTY2FsaW5nIGdyb3VwcyBzdXBwb3J0IG1haW50ZW5hbmNlIG9mIGFkZXF1YXRlIGNhcGFjaXR5IGFuZCBhdmFpbGFiaWxpdHkuIFRoZSBsb2FkIGJhbGFuY2VyIHBlcmlvZGljYWxseSBzZW5kcyBwaW5ncywgYXR0ZW1wdHMgY29ubmVjdGlvbnMsIG9yIHNlbmRzIHJlcXVlc3RzIHRvIHRlc3QgQW1hem9uIEVDMiBpbnN0YW5jZXMgaGVhbHRoIGluIGFuIGF1dG8tc2NhbGluZyBncm91cC4gSWYgYW4gaW5zdGFuY2UgaXMgbm90IHJlcG9ydGluZyBiYWNrLCB0cmFmZmljIGlzIHNlbnQgdG8gYSBuZXcgQW1hem9uIEVDMiBpbnN0YW5jZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEF1dG9TY2FsaW5nR3JvdXBFTEJIZWFsdGhDaGVja1JlcXVpcmVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIEF1dG8gU2NhbGluZyBsYXVuY2ggY29uZmlndXJhdGlvbiBkb2VzIG5vdCBoYXZlIHB1YmxpYyBJUCBhZGRyZXNzZXMgZGlzYWJsZWQgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAxLjMuNCwgMS4zLjYsIDIuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdJZiB5b3UgY29uZmlndXJlIHlvdXIgTmV0d29yayBJbnRlcmZhY2VzIHdpdGggYSBwdWJsaWMgSVAgYWRkcmVzcywgdGhlbiB0aGUgYXNzb2NpYXRlZCByZXNvdXJjZXMgdG8gdGhvc2UgTmV0d29yayBJbnRlcmZhY2VzIGFyZSByZWFjaGFibGUgZnJvbSB0aGUgaW50ZXJuZXQuIEVDMiByZXNvdXJjZXMgc2hvdWxkIG5vdCBiZSBwdWJsaWNseSBhY2Nlc3NpYmxlLCBhcyB0aGlzIG1heSBhbGxvdyB1bmludGVuZGVkIGFjY2VzcyB0byB5b3VyIGFwcGxpY2F0aW9ucyBvciBzZXJ2ZXJzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQXV0b1NjYWxpbmdMYXVuY2hDb25maWdQdWJsaWNJcERpc2FibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBDbG91ZFRyYWlsIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tDbG91ZFRyYWlsKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSB0cmFpbCBkb2VzIG5vdCBoYXZlIENsb3VkV2F0Y2ggbG9ncyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAyLjIsIDEwLjEsIDEwLjIuMSwgMTAuMi4yLCAxMC4yLjMsIDEwLjIuNSwgMTAuMy4xLCAxMC4zLjIsIDEwLjMuMywgMTAuMy40LCAxMC4zLjUsIDEwLjMuNiwgMTAuNS4zLCAxMC41LjQpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1VzZSBBbWF6b24gQ2xvdWRXYXRjaCB0byBjZW50cmFsbHkgY29sbGVjdCBhbmQgbWFuYWdlIGxvZyBldmVudCBhY3Rpdml0eS4gSW5jbHVzaW9uIG9mIEFXUyBDbG91ZFRyYWlsIGRhdGEgcHJvdmlkZXMgZGV0YWlscyBvZiBBUEkgY2FsbCBhY3Rpdml0eSB3aXRoaW4geW91ciBBV1MgYWNjb3VudC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENsb3VkVHJhaWxDbG91ZFdhdGNoTG9nc0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgdHJhaWwgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDIuMiwgMy40LCAxMC41KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIG1heSBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBV1MgQ2xvdWRUcmFpbCB0cmFpbHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDbG91ZFRyYWlsRW5jcnlwdGlvbkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgdHJhaWwgZG9lcyBub3QgaGF2ZSBsb2cgZmlsZSB2YWxpZGF0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDIuMiwgMTAuNS4yLCAxMC41LCAxMC41LjUsIDExLjUpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1V0aWxpemUgQVdTIENsb3VkVHJhaWwgbG9nIGZpbGUgdmFsaWRhdGlvbiB0byBjaGVjayB0aGUgaW50ZWdyaXR5IG9mIENsb3VkVHJhaWwgbG9ncy4gTG9nIGZpbGUgdmFsaWRhdGlvbiBoZWxwcyBkZXRlcm1pbmUgaWYgYSBsb2cgZmlsZSB3YXMgbW9kaWZpZWQgb3IgZGVsZXRlZCBvciB1bmNoYW5nZWQgYWZ0ZXIgQ2xvdWRUcmFpbCBkZWxpdmVyZWQgaXQuIFRoaXMgZmVhdHVyZSBpcyBidWlsdCB1c2luZyBpbmR1c3RyeSBzdGFuZGFyZCBhbGdvcml0aG1zOiBTSEEtMjU2IGZvciBoYXNoaW5nIGFuZCBTSEEtMjU2IHdpdGggUlNBIGZvciBkaWdpdGFsIHNpZ25pbmcuIFRoaXMgbWFrZXMgaXQgY29tcHV0YXRpb25hbGx5IGluZmVhc2libGUgdG8gbW9kaWZ5LCBkZWxldGUgb3IgZm9yZ2UgQ2xvdWRUcmFpbCBsb2cgZmlsZXMgd2l0aG91dCBkZXRlY3Rpb24uJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDbG91ZFRyYWlsTG9nRmlsZVZhbGlkYXRpb25FbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBDbG91ZFdhdGNoIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tDbG91ZFdhdGNoKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBDbG91ZFdhdGNoIExvZyBHcm91cCBpcyBub3QgZW5jcnlwdGVkIHdpdGggYW4gQVdTIEtNUyBrZXkgLSAoQ29udHJvbCBJRDogMy40KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUbyBoZWxwIHByb3RlY3Qgc2Vuc2l0aXZlIGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBbWF6b24gQ2xvdWRXYXRjaCBMb2cgR3JvdXBzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQ2xvdWRXYXRjaExvZ0dyb3VwRW5jcnlwdGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIENsb3VkV2F0Y2ggTG9nIEdyb3VwIGRvZXMgbm90IGhhdmUgYW4gZXhwbGljaXQgcmV0ZW50aW9uIHBlcmlvZCBjb25maWd1cmVkIC0gKENvbnRyb2wgSURzOiAzLjEsIDEwLjcpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSBhIG1pbmltdW0gZHVyYXRpb24gb2YgZXZlbnQgbG9nIGRhdGEgaXMgcmV0YWluZWQgZm9yIHlvdXIgbG9nIGdyb3VwcyB0byBoZWxwIHdpdGggdHJvdWJsZXNob290aW5nIGFuZCBmb3JlbnNpY3MgaW52ZXN0aWdhdGlvbnMuIFRoZSBsYWNrIG9mIGF2YWlsYWJsZSBwYXN0IGV2ZW50IGxvZyBkYXRhIG1ha2VzIGl0IGRpZmZpY3VsdCB0byByZWNvbnN0cnVjdCBhbmQgaWRlbnRpZnkgcG90ZW50aWFsbHkgbWFsaWNpb3VzIGV2ZW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENsb3VkV2F0Y2hMb2dHcm91cFJldGVudGlvblBlcmlvZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQ29kZUJ1aWxkIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tDb2RlQnVpbGQobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIENvZGVCdWlsZCBlbnZpcm9ubWVudCBzdG9yZXMgc2Vuc2l0aXZlIGNyZWRlbnRpYWxzIChzdWNoIGFzIEFXU19BQ0NFU1NfS0VZX0lEIGFuZC9vciBBV1NfU0VDUkVUX0FDQ0VTU19LRVkpIGFzIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgLSAoQ29udHJvbCBJRDogOC4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0RvIG5vdCBzdG9yZSB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dC4gU3RvcmluZyB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dCBsZWFkcyB0byB1bmludGVuZGVkIGRhdGEgZXhwb3N1cmUgYW5kIHVuYXV0aG9yaXplZCBhY2Nlc3MuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBDb2RlQnVpbGRQcm9qZWN0RW52VmFyQXdzQ3JlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBDb2RlQnVpbGQgcHJvamVjdCB3aGljaCB1dGlsaXplcyBlaXRoZXIgYSBHaXRIdWIgb3IgQml0QnVja2V0IHNvdXJjZSByZXBvc2l0b3J5IGRvZXMgbm90IHV0aWxpemUgT0FVVEggLSAoQ29udHJvbCBJRDogOC4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ09BVVRIIGlzIHRoZSBtb3N0IHNlY3VyZSBtZXRob2Qgb2YgYXV0aGVudGljYXRpbmcgeW91ciBDb2RlQnVpbGQgYXBwbGljYXRpb24uIFVzZSBPQXV0aCBpbnN0ZWFkIG9mIHBlcnNvbmFsIGFjY2VzcyB0b2tlbnMgb3IgYSB1c2VyIG5hbWUgYW5kIHBhc3N3b3JkIHRvIGdyYW50IGF1dGhvcml6YXRpb24gZm9yIGFjY2Vzc2luZyBHaXRIdWIgb3IgQml0YnVja2V0IHJlcG9zaXRvcmllcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IENvZGVCdWlsZFByb2plY3RTb3VyY2VSZXBvVXJsLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBETVMgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0RNUyhub2RlOiBDZm5SZXNvdXJjZSkge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgRE1TIHJlcGxpY2F0aW9uIGluc3RhbmNlIGlzIHB1YmxpYyAtIChDb250cm9sIElEczogMS4yLCAxLjIuMSwgMS4zLCAxLjMuMSwgMS4zLjIsIDEuMy40LCAxLjMuNiwgMi4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0RNUyByZXBsaWNhdGlvbiBpbnN0YW5jZXMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uIGFuZCBhY2Nlc3MgY29udHJvbCBpcyByZXF1aXJlZCBmb3Igc3VjaCBhY2NvdW50cy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IERNU1JlcGxpY2F0aW9uTm90UHVibGljLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFQzIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VDMihub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgRUMyIGluc3RhbmNlIGlzIGFzc29jaWF0ZWQgd2l0aCBhIHB1YmxpYyBJUCBhZGRyZXNzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnTWFuYWdlIGFjY2VzcyB0byB0aGUgQVdTIENsb3VkIGJ5IGVuc3VyaW5nIEFtYXpvbiBFbGFzdGljIENvbXB1dGUgQ2xvdWQgKEFtYXpvbiBFQzIpIGluc3RhbmNlcyBjYW5ub3QgYmUgcHVibGljbHkgYWNjZXNzZWQuIEFtYXpvbiBFQzIgaW5zdGFuY2VzIGNhbiBjb250YWluIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFQzJJbnN0YW5jZU5vUHVibGljSXAsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgRUMyIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgYW4gaW5zdGFuY2UgcHJvZmlsZSBhdHRhY2hlZCAtIChDb250cm9sIElEczogMi4yLCA3LjEuMSwgNy4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VDMiBpbnN0YW5jZSBwcm9maWxlcyBwYXNzIGFuIElBTSByb2xlIHRvIGFuIEVDMiBpbnN0YW5jZS4gQXR0YWNoaW5nIGFuIGluc3RhbmNlIHByb2ZpbGUgdG8geW91ciBpbnN0YW5jZXMgY2FuIGFzc2lzdCB3aXRoIGxlYXN0IHByaXZpbGVnZSBhbmQgcGVybWlzc2lvbnMgbWFuYWdlbWVudC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDMkluc3RhbmNlUHJvZmlsZUF0dGFjaGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIEVDMiBpbnN0YW5jZSBpcyBub3Qgd2l0aGluIGEgVlBDIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRGVwbG95IEFtYXpvbiBFbGFzdGljIENvbXB1dGUgQ2xvdWQgKEFtYXpvbiBFQzIpIGluc3RhbmNlcyB3aXRoaW4gYW4gQW1hem9uIFZpcnR1YWwgUHJpdmF0ZSBDbG91ZCAoQW1hem9uIFZQQykgdG8gZW5hYmxlIHNlY3VyZSBjb21tdW5pY2F0aW9uIGJldHdlZW4gYW4gaW5zdGFuY2UgYW5kIG90aGVyIHNlcnZpY2VzIHdpdGhpbiB0aGUgYW1hem9uIFZQQywgd2l0aG91dCByZXF1aXJpbmcgYW4gaW50ZXJuZXQgZ2F0ZXdheSwgTkFUIGRldmljZSwgb3IgVlBOIGNvbm5lY3Rpb24uIEFsbCB0cmFmZmljIHJlbWFpbnMgc2VjdXJlbHkgd2l0aGluIHRoZSBBV1MgQ2xvdWQuIEJlY2F1c2Ugb2YgdGhlaXIgbG9naWNhbCBpc29sYXRpb24sIGRvbWFpbnMgdGhhdCByZXNpZGUgd2l0aGluIGFuQW1hem9uIFZQQyBoYXZlIGFuIGV4dHJhIGxheWVyIG9mIHNlY3VyaXR5IHdoZW4gY29tcGFyZWQgdG8gZG9tYWlucyB0aGF0IHVzZSBwdWJsaWMgZW5kcG9pbnRzLiBBc3NpZ24gQW1hem9uIEVDMiBpbnN0YW5jZXMgdG8gYW4gQW1hem9uIFZQQyB0byBwcm9wZXJseSBtYW5hZ2UgYWNjZXNzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRUMySW5zdGFuY2VzSW5WUEMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgRUMyIGluc3RhbmNlIGFsbG93cyB1bnJlc3RyaWN0ZWQgaW5ib3VuZCBJUHY0IFRDUCB0cmFmZmljIG9uIG9uZSBvciBtb3JlIGNvbW1vbiBwb3J0cyAoYnkgZGVmYXVsdCB0aGVzZSBwb3J0cyBpbmNsdWRlIDIwLCAyMSwgMzM4OSwgMzMwOSwgMzMwNiwgNDMzMykgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAyLjIsIDIuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdOb3QgcmVzdHJpY3RpbmcgYWNjZXNzIHRvIHBvcnRzIHRvIHRydXN0ZWQgc291cmNlcyBjYW4gbGVhZCB0byBhdHRhY2tzIGFnYWluc3QgdGhlIGF2YWlsYWJpbGl0eSwgaW50ZWdyaXR5IGFuZCBjb25maWRlbnRpYWxpdHkgb2Ygc3lzdGVtcy4gQnkgZGVmYXVsdCwgY29tbW9uIHBvcnRzIHdoaWNoIHNob3VsZCBiZSByZXN0cmljdGVkIGluY2x1ZGUgcG9ydCBudW1iZXJzIDIwLCAyMSwgMzM4OSwgMzMwNiwgYW5kIDQzMzMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFQzJSZXN0cmljdGVkQ29tbW9uUG9ydHMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgU2VjdXJpdHkgR3JvdXAgYWxsb3dzIHVucmVzdHJpY3RlZCBTU0ggYWNjZXNzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMi4yLCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnTm90IGFsbG93aW5nIGluZ3Jlc3MgKG9yIHJlbW90ZSkgdHJhZmZpYyBmcm9tIDAuMC4wLjAvMCBvciA6Oi8wIHRvIHBvcnQgMjIgb24geW91ciByZXNvdXJjZXMgaGVscHMgdG8gcmVzdHJpY3QgcmVtb3RlIGFjY2Vzcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVDMlJlc3RyaWN0ZWRTU0gsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVDUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRUNTKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogXCJUaGUgRUNTIHRhc2sgZGVmaW5pdGlvbiBpcyBjb25maWd1cmVkIGZvciBob3N0IG5ldHdvcmtpbmcgYW5kIGhhcyBhdCBsZWFzdCBvbmUgY29udGFpbmVyIHdpdGggZGVmaW5pdGlvbnMgd2l0aCAncHJpdmlsZWdlZCcgc2V0IHRvIGZhbHNlIG9yIGVtcHR5IG9yICd1c2VyJyBzZXQgdG8gcm9vdCBvciBlbXB0eSAtIChDb250cm9sIElEOiA3LjIuMSkuXCIsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0lmIGEgdGFzayBkZWZpbml0aW9uIGhhcyBlbGV2YXRlZCBwcml2aWxlZ2VzIGl0IGlzIGJlY2F1c2UgeW91IGhhdmUgc3BlY2lmaWNhbGx5IG9wdGVkLWluIHRvIHRob3NlIGNvbmZpZ3VyYXRpb25zLiBUaGlzIHJ1bGUgY2hlY2tzIGZvciB1bmV4cGVjdGVkIHByaXZpbGVnZSBlc2NhbGF0aW9uIHdoZW4gYSB0YXNrIGRlZmluaXRpb24gaGFzIGhvc3QgbmV0d29ya2luZyBlbmFibGVkIGJ1dCB0aGUgY3VzdG9tZXIgaGFzIG5vdCBvcHRlZC1pbiB0byBlbGV2YXRlZCBwcml2aWxlZ2VzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogRUNTVGFza0RlZmluaXRpb25Vc2VyRm9ySG9zdE1vZGUsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVGUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRUZTKG5vZGU6IENmblJlc291cmNlKSB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBFRlMgZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZCAtIChDb250cm9sIElEczogMy40LCA4LjIuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgQW1hem9uIEVsYXN0aWMgRmlsZSBTeXN0ZW0gKEVGUykuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFRlNFbmNyeXB0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVsYXN0aWMgTG9hZCBCYWxhbmNlciBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRUxCKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBBTEIgZG9lcyBub3QgaGF2ZSBpbnZhbGlkIEhUVFAgaGVhZGVyIGRyb3BwaW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDQuMSwgOC4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSB0aGF0IHlvdXIgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlcnMgKEFMQikgYXJlIGNvbmZpZ3VyZWQgdG8gZHJvcCBodHRwIGhlYWRlcnMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBBTEJIdHRwRHJvcEludmFsaWRIZWFkZXJFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiBcIlRoZSBBTEIncyBIVFRQIGxpc3RlbmVycyBhcmUgbm90IGNvbmZpZ3VyZWQgdG8gcmVkaXJlY3QgdG8gSFRUUFMgLSAoQ29udHJvbCBJRHM6IDIuMywgNC4xLCA4LjIuMSkuXCIsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RvIGhlbHAgcHJvdGVjdCBkYXRhIGluIHRyYW5zaXQsIGVuc3VyZSB0aGF0IHlvdXIgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlciBhdXRvbWF0aWNhbGx5IHJlZGlyZWN0cyB1bmVuY3J5cHRlZCBIVFRQIHJlcXVlc3RzIHRvIEhUVFBTLiBCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogQUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbixcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBBTEIgaXMgbm90IGFzc29jaWF0ZWQgd2l0aCBBV1MgV0FGdjIgd2ViIEFDTCAtIChDb250cm9sIElEOiA2LjYpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0EgV0FGIGhlbHBzIHRvIHByb3RlY3QgeW91ciB3ZWIgYXBwbGljYXRpb25zIG9yIEFQSXMgYWdhaW5zdCBjb21tb24gd2ViIGV4cGxvaXRzLiBUaGVzZSB3ZWIgZXhwbG9pdHMgbWF5IGFmZmVjdCBhdmFpbGFiaWxpdHksIGNvbXByb21pc2Ugc2VjdXJpdHksIG9yIGNvbnN1bWUgZXhjZXNzaXZlIHJlc291cmNlcyB3aXRoaW4geW91ciBlbnZpcm9ubWVudC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEFMQldBRkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgQ0xCIGRvZXMgbm90IHV0aWxpemUgYW4gU1NMIGNlcnRpZmljYXRlIHByb3ZpZGVkIGJ5IEFDTSAoQW1hem9uIENlcnRpZmljYXRlIE1hbmFnZXIpIC0gKENvbnRyb2wgSURzOiA0LjEsIDguMi4xKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgdHJhbnNpdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBFbGFzdGljIExvYWQgQmFsYW5jaW5nLiBVc2UgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgdG8gbWFuYWdlLCBwcm92aXNpb24gYW5kIGRlcGxveSBwdWJsaWMgYW5kIHByaXZhdGUgU1NML1RMUyBjZXJ0aWZpY2F0ZXMgd2l0aCBBV1Mgc2VydmljZXMgYW5kIGludGVybmFsIHJlc291cmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVMQkFDTUNlcnRpZmljYXRlUmVxdWlyZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgRUxCIGRvZXMgbm90IGhhdmUgbG9nZ2luZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxMC4xLCAxMC4zLjEsIDEwLjMuMiwgMTAuMy4zLCAxMC4zLjQsIDEwLjMuNSwgMTAuMy42LCAxMC41LjQpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgXCJFbGFzdGljIExvYWQgQmFsYW5jaW5nIGFjdGl2aXR5IGlzIGEgY2VudHJhbCBwb2ludCBvZiBjb21tdW5pY2F0aW9uIHdpdGhpbiBhbiBlbnZpcm9ubWVudC4gRW5zdXJlIEVMQiBsb2dnaW5nIGlzIGVuYWJsZWQuIFRoZSBjb2xsZWN0ZWQgZGF0YSBwcm92aWRlcyBkZXRhaWxlZCBpbmZvcm1hdGlvbiBhYm91dCByZXF1ZXN0cyBzZW50IHRvIFRoZSBFTEIuIEVhY2ggbG9nIGNvbnRhaW5zIGluZm9ybWF0aW9uIHN1Y2ggYXMgdGhlIHRpbWUgdGhlIHJlcXVlc3Qgd2FzIHJlY2VpdmVkLCB0aGUgY2xpZW50J3MgSVAgYWRkcmVzcywgbGF0ZW5jaWVzLCByZXF1ZXN0IHBhdGhzLCBhbmQgc2VydmVyIHJlc3BvbnNlcy5cIixcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFTEJMb2dnaW5nRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBDTEIgZG9lcyBub3QgcmVzdHJpY3QgaXRzIGxpc3RlbmVycyB0byBvbmx5IHRoZSBTU0wgYW5kIEhUVFBTIHByb3RvY29scyAtIChDb250cm9sIElEczogMi4zLCA0LjEsIDguMi4xKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbnN1cmUgdGhhdCB5b3VyIENsYXNzaWMgTG9hZCBCYWxhbmNlcnMgKENMQnMpIGFyZSBjb25maWd1cmVkIHdpdGggU1NMIG9yIEhUVFBTIGxpc3RlbmVycy4gQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QsIGVuYWJsZSBlbmNyeXB0aW9uIGluIHRyYW5zaXQgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVMQlRsc0h0dHBzTGlzdGVuZXJzT25seSxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBBTEIsIE5MQiwgb3IgR0xCIGxpc3RlbmVyIGRvZXMgbm90IHV0aWxpemUgYW4gU1NMIGNlcnRpZmljYXRlIHByb3ZpZGVkIGJ5IEFDTSAoQW1hem9uIENlcnRpZmljYXRlIE1hbmFnZXIpIC0gKENvbnRyb2wgSUQ6IDQuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHRyYW5zaXQsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgRWxhc3RpYyBMb2FkIEJhbGFuY2luZy4gVXNlIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIHRvIG1hbmFnZSwgcHJvdmlzaW9uIGFuZCBkZXBsb3kgcHVibGljIGFuZCBwcml2YXRlIFNTTC9UTFMgY2VydGlmaWNhdGVzIHdpdGggQVdTIHNlcnZpY2VzIGFuZCBpbnRlcm5hbCByZXNvdXJjZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBFTEJ2MkFDTUNlcnRpZmljYXRlUmVxdWlyZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVNUiBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRU1SKG5vZGU6IENmblJlc291cmNlKSB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBBTEIsIE5MQiwgb3IgR0xCIGxpc3RlbmVyIGRvZXMgbm90IHV0aWxpemUgYW4gU1NMIGNlcnRpZmljYXRlIHByb3ZpZGVkIGJ5IEFDTSAoQW1hem9uIENlcnRpZmljYXRlIE1hbmFnZXIpIC0gKENvbnRyb2wgSUQ6IDcuMi4xKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgdHJhbnNpdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBFbGFzdGljIExvYWQgQmFsYW5jaW5nLiBVc2UgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgdG8gbWFuYWdlLCBwcm92aXNpb24gYW5kIGRlcGxveSBwdWJsaWMgYW5kIHByaXZhdGUgU1NML1RMUyBjZXJ0aWZpY2F0ZXMgd2l0aCBBV1Mgc2VydmljZXMgYW5kIGludGVybmFsIHJlc291cmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEVNUktlcmJlcm9zRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgSUFNIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tJQU0obm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIElBTSBHcm91cCBkb2VzIG5vdCBoYXZlIGF0IGxlYXN0IG9uZSBJQU0gVXNlciAtIChDb250cm9sIElEczogNy4xLjIswqA3LjEuMyzCoDcuMi4xLMKgNy4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgY2FuIGhlbHAgeW91IGluY29ycG9yYXRlIHRoZSBwcmluY2lwbGVzIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMgd2l0aCBhY2Nlc3MgcGVybWlzc2lvbnMgYW5kIGF1dGhvcml6YXRpb25zLCBieSBlbnN1cmluZyB0aGF0IElBTSBncm91cHMgaGF2ZSBhdCBsZWFzdCBvbmUgSUFNIHVzZXIuIFBsYWNpbmcgSUFNIHVzZXJzIGluIGdyb3VwcyBiYXNlZCBvbiB0aGVpciBhc3NvY2lhdGVkIHBlcm1pc3Npb25zIG9yIGpvYiBmdW5jdGlvbiBpcyBvbmUgd2F5IHRvIGluY29ycG9yYXRlIGxlYXN0IHByaXZpbGVnZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IElBTUdyb3VwSGFzVXNlcnMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgSUFNIEdyb3VwLCBVc2VyLCBvciBSb2xlIGNvbnRhaW5zIGFuIGlubGluZSBwb2xpY3kgLSAoQ29udHJvbCBJRHM6IDIuMiwgNy4xLjIsIDcuMS4zLCA3LjIuMSwgNy4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FXUyByZWNvbW1lbmRzIHRvIHVzZSBtYW5hZ2VkIHBvbGljaWVzIGluc3RlYWQgb2YgaW5saW5lIHBvbGljaWVzLiBUaGUgbWFuYWdlZCBwb2xpY2llcyBhbGxvdyByZXVzYWJpbGl0eSwgdmVyc2lvbmluZyBhbmQgcm9sbGluZyBiYWNrLCBhbmQgZGVsZWdhdGluZyBwZXJtaXNzaW9ucyBtYW5hZ2VtZW50LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogSUFNTm9JbmxpbmVQb2xpY3ksXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgSUFNIHBvbGljeSBncmFudHMgYWRtaW4gYWNjZXNzIC0gKENvbnRyb2wgSURzOiAyLjIsIDcuMS4yLCA3LjEuMywgNy4yLjEsIDcuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBV1MgSWRlbnRpdHkgYW5kIEFjY2VzcyBNYW5hZ2VtZW50IChJQU0pIGNhbiBoZWxwIHlvdSBpbmNvcnBvcmF0ZSB0aGUgcHJpbmNpcGxlcyBvZiBsZWFzdCBwcml2aWxlZ2UgYW5kIHNlcGFyYXRpb24gb2YgZHV0aWVzIHdpdGggYWNjZXNzIHBlcm1pc3Npb25zIGFuZCBhdXRob3JpemF0aW9ucywgYnkgZW5zdXJpbmcgdGhhdCBJQU0gZ3JvdXBzIGhhdmUgYXQgbGVhc3Qgb25lIElBTSB1c2VyLiBQbGFjaW5nIElBTSB1c2VycyBpbiBncm91cHMgYmFzZWQgb24gdGhlaXIgYXNzb2NpYXRlZCBwZXJtaXNzaW9ucyBvciBqb2IgZnVuY3Rpb24gaXMgb25lIHdheSB0byBpbmNvcnBvcmF0ZSBsZWFzdCBwcml2aWxlZ2UuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBJQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoQWRtaW5BY2Nlc3MsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgSUFNIHBvbGljeSBncmFudHMgZnVsbCBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IDcuMS4yLCA3LjEuMywgNy4yLjEsIDcuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbnN1cmUgSUFNIEFjdGlvbnMgYXJlIHJlc3RyaWN0ZWQgdG8gb25seSB0aG9zZSBhY3Rpb25zIHRoYXQgYXJlIG5lZWRlZC4gQWxsb3dpbmcgdXNlcnMgdG8gaGF2ZSBtb3JlIHByaXZpbGVnZXMgdGhhbiBuZWVkZWQgdG8gY29tcGxldGUgYSB0YXNrIG1heSB2aW9sYXRlIHRoZSBwcmluY2lwbGUgb2YgbGVhc3QgcHJpdmlsZWdlIGFuZCBzZXBhcmF0aW9uIG9mIGR1dGllcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IElBTVBvbGljeU5vU3RhdGVtZW50c1dpdGhGdWxsQWNjZXNzLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIElBTSB1c2VyIGRvZXMgbm90IGJlbG9uZyB0byBhbnkgZ3JvdXAocykgLSAoQ29udHJvbCBJRHM6IDIuMiwgNy4xLjIsIDcuMS4zLCA3LjIuMSwgNy4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0FXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgY2FuIGhlbHAgeW91IHJlc3RyaWN0IGFjY2VzcyBwZXJtaXNzaW9ucyBhbmQgYXV0aG9yaXphdGlvbnMgYnkgZW5zdXJpbmcgSUFNIHVzZXJzIGFyZSBtZW1iZXJzIG9mIGF0IGxlYXN0IG9uZSBncm91cC4gQWxsb3dpbmcgdXNlcnMgbW9yZSBwcml2aWxlZ2VzIHRoYW4gbmVlZGVkIHRvIGNvbXBsZXRlIGEgdGFzayBtYXkgdmlvbGF0ZSB0aGUgcHJpbmNpcGxlIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBJQU1Vc2VyR3JvdXBNZW1iZXJzaGlwLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIElBTSBwb2xpY3kgaXMgYXR0YWNoZWQgYXQgdGhlIHVzZXIgbGV2ZWwgLSAoQ29udHJvbCBJRHM6IDIuMiwgNy4xLjIsIDcuMS4zLCA3LjIuMSwgNy4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Fzc2lnbmluZyBwcml2aWxlZ2VzIGF0IHRoZSBncm91cCBvciB0aGUgcm9sZSBsZXZlbCBoZWxwcyB0byByZWR1Y2Ugb3Bwb3J0dW5pdHkgZm9yIGFuIGlkZW50aXR5IHRvIHJlY2VpdmUgb3IgcmV0YWluIGV4Y2Vzc2l2ZSBwcml2aWxlZ2VzLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogSUFNVXNlck5vUG9saWNpZXMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEtNUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrS01TKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBLTVMgU3ltbWV0cmljIGtleSBkb2VzIG5vdCBoYXZlIGF1dG9tYXRpYyBrZXkgcm90YXRpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogMi4yLMKgMy41LMKgMy42LMKgMy42LjQpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuYWJsZSBrZXkgcm90YXRpb24gdG8gZW5zdXJlIHRoYXQga2V5cyBhcmUgcm90YXRlZCBvbmNlIHRoZXkgaGF2ZSByZWFjaGVkIHRoZSBlbmQgb2YgdGhlaXIgY3J5cHRvIHBlcmlvZC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IEtNU0JhY2tpbmdLZXlSb3RhdGlvbkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIExhbWJkYSBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrTGFtYmRhKG5vZGU6IENmblJlc291cmNlKSB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBMYW1iZGEgZnVuY3Rpb24gaXMgbm90IFZQQyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDIuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIG9mIHRoZWlyIGxvZ2ljYWwgaXNvbGF0aW9uLCBkb21haW5zIHRoYXQgcmVzaWRlIHdpdGhpbiBhbiBBbWF6b24gVlBDIGhhdmUgYW4gZXh0cmEgbGF5ZXIgb2Ygc2VjdXJpdHkgd2hlbiBjb21wYXJlZCB0byBkb21haW5zIHRoYXQgdXNlIHB1YmxpYyBlbmRwb2ludHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBMYW1iZGFJbnNpZGVWUEMsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIE9wZW5TZWFyY2ggUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja09wZW5TZWFyY2gobm9kZTogQ2ZuUmVzb3VyY2UpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgaGF2ZSBlbmNyeXB0aW9uIGF0IHJlc3QgZW5hYmxlZCAtIChDb250cm9sIElEczogMy40LCA4LjIuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSAoT3BlblNlYXJjaCBTZXJ2aWNlKSBkb21haW5zLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogT3BlblNlYXJjaEVuY3J5cHRlZEF0UmVzdCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGlzIG5vdCBydW5uaW5nIHdpdGhpbiBhIFZQQyAtIChDb250cm9sIElEczogMS4yLCAxLjIuMSwgMS4zLCAxLjMuMSwgMS4zLjIsIDEuMy40LCAxLjMuNiwgMi4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1ZQQ3MgaGVscCBzZWN1cmUgeW91ciBBV1MgcmVzb3VyY2VzIGFuZCBwcm92aWRlIGFuIGV4dHJhIGxheWVyIG9mIHByb3RlY3Rpb24uJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBPcGVuU2VhcmNoSW5WUENPbmx5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3Qgc3RyZWFtIGVycm9yIGxvZ3MgKEVTX0FQUExJQ0FUSU9OX0xPR1MpIHRvIENsb3VkV2F0Y2ggTG9ncyAtIChDb250cm9sIElEczogMTAuMSwgMTAuMi4xLCAxMC4yLjIsIDEwLjIuMywgMTAuMi40LCAxMC4yLjUsIDEwLjMuMSwgMTAuMy4yLCAxMC4zLjMsIDEwLjMuNCwgMTAuMy41LCAxMC4zLjYpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbnMgaGF2ZSBlcnJvciBsb2dzIGVuYWJsZWQgYW5kIHN0cmVhbWVkIHRvIEFtYXpvbiBDbG91ZFdhdGNoIExvZ3MgZm9yIHJldGVudGlvbiBhbmQgcmVzcG9uc2UuIERvbWFpbiBlcnJvciBsb2dzIGNhbiBhc3Npc3Qgd2l0aCBzZWN1cml0eSBhbmQgYWNjZXNzIGF1ZGl0cywgYW5kIGNhbiBoZWxwIHRvIGRpYWdub3NlIGF2YWlsYWJpbGl0eSBpc3N1ZXMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBPcGVuU2VhcmNoRXJyb3JMb2dzVG9DbG91ZFdhdGNoLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgaGF2ZSBub2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IDQuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QsIGVuYWJsZSBlbmNyeXB0aW9uIGluIHRyYW5zaXQgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YSB3aXRoaW4geW91ciBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIChPcGVuU2VhcmNoIFNlcnZpY2UpIGRvbWFpbnMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBPcGVuU2VhcmNoTm9kZVRvTm9kZUVuY3J5cHRpb24sXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFJEUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUkRTKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogXCJUaGUgcm91dGUgdGFibGUgbWF5IGNvbnRhaW4gb25lIG9yIG1vcmUgdW5yZXN0cmljdGVkIHJvdXRlKHMpIHRvIGFuIElHVyAoJzAuMC4wLjAvMCcgb3IgJzo6LzAnKSAtIChDb250cm9sIElEOiA2LjIpLlwiLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdFbnN1cmUgQW1hem9uIEVDMiByb3V0ZSB0YWJsZXMgZG8gbm90IGhhdmUgdW5yZXN0cmljdGVkIHJvdXRlcyB0byBhbiBpbnRlcm5ldCBnYXRld2F5LiBSZW1vdmluZyBvciBsaW1pdGluZyB0aGUgYWNjZXNzIHRvIHRoZSBpbnRlcm5ldCBmb3Igd29ya2xvYWRzIHdpdGhpbiBBbWF6b24gVlBDcyBjYW4gcmVkdWNlIHVuaW50ZW5kZWQgYWNjZXNzIHdpdGhpbiB5b3VyIGVudmlyb25tZW50LicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUkRTQXV0b21hdGljTWlub3JWZXJzaW9uVXBncmFkZUVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIEluc3RhbmNlIGFsbG93cyBwdWJsaWMgYWNjZXNzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQW1hem9uIFJEUyBkYXRhYmFzZSBpbnN0YW5jZXMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uLCBhbmQgcHJpbmNpcGxlcyBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSRFNJbnN0YW5jZVB1YmxpY0FjY2VzcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBSRFMgREIgSW5zdGFuY2UgZG9lcyBub3QgaGF2ZSBhbGwgQ2xvdWRXYXRjaCBsb2cgdHlwZXMgZXhwb3J0ZWQgLSAoQ29udHJvbCBJRHM6IDEwLjEsIDEwLjIuMSwgMTAuMi4yLCAxMC4yLjMsIDEwLjIuNCwgMTAuMi41LCAxMC4zLjEsIDEwLjMuMiwgMTAuMy4zLCAxMC4zLjQsIDEwLjMuNSwgMTAuMy42KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdUbyBoZWxwIHdpdGggbG9nZ2luZyBhbmQgbW9uaXRvcmluZyB3aXRoaW4geW91ciBlbnZpcm9ubWVudCwgZW5zdXJlIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKEFtYXpvbiBSRFMpIGxvZ2dpbmcgaXMgZW5hYmxlZC4gV2l0aCBBbWF6b24gUkRTIGxvZ2dpbmcsIHlvdSBjYW4gY2FwdHVyZSBldmVudHMgc3VjaCBhcyBjb25uZWN0aW9ucywgZGlzY29ubmVjdGlvbnMsIHF1ZXJpZXMsIG9yIHRhYmxlcyBxdWVyaWVkLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUkRTTG9nZ2luZ0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUkRTIERCIEluc3RhbmNlIG9yIEF1cm9yYSBDbHVzdGVyIGRvZXMgbm90IGhhdmUgc3RvcmFnZSBlbmNyeXB0ZWQgLSAoQ29udHJvbCBJRHM6IDMuNCwgOC4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gQW1hem9uIFJEUyBpbnN0YW5jZXMsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJEU1N0b3JhZ2VFbmNyeXB0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFJlZHNoaWZ0IFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tSZWRzaGlmdChub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gb3IgYXVkaXQgbG9nZ2luZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAzLjQsIDguMi4xLCAxMC4xLCAxMC4yLjEsIDEwLjIuMiwgMTAuMi4zLCAxMC4yLjQsIDEwLjIuNSwgMTAuMy4xLCAxMC4zLjIsIDEwLjMuMywgMTAuMy40LCAxMC4zLjUsIDEwLjMuNikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVG8gcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSB0aGF0IGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBbWF6b24gUmVkc2hpZnQgY2x1c3RlcnMuIFlvdSBtdXN0IGFsc28gZW5zdXJlIHRoYXQgcmVxdWlyZWQgY29uZmlndXJhdGlvbnMgYXJlIGRlcGxveWVkIG9uIEFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycy4gVGhlIGF1ZGl0IGxvZ2dpbmcgc2hvdWxkIGJlIGVuYWJsZWQgdG8gcHJvdmlkZSBpbmZvcm1hdGlvbiBhYm91dCBjb25uZWN0aW9ucyBhbmQgdXNlciBhY3Rpdml0aWVzIGluIHRoZSBkYXRhYmFzZS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0Q2x1c3RlckNvbmZpZ3VyYXRpb24sXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHZlcnNpb24gdXBncmFkZXMgZW5hYmxlZCwgYXV0b21hdGVkIHNuYXBzaG90IHJldGVudGlvbiBwZXJpb2RzIGVuYWJsZWQsIGFuZCBhbiBleHBsaWNpdCBtYWludGVuYW5jZSB3aW5kb3cgY29uZmlndXJlZCAtIChDb250cm9sIElEOiA2LjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSB0aGF0IEFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycyBoYXZlIHRoZSBwcmVmZXJyZWQgc2V0dGluZ3MgZm9yIHlvdXIgb3JnYW5pemF0aW9uLiBTcGVjaWZpY2FsbHksIHRoYXQgdGhleSBoYXZlIHByZWZlcnJlZCBtYWludGVuYW5jZSB3aW5kb3dzIGFuZCBhdXRvbWF0ZWQgc25hcHNob3QgcmV0ZW50aW9uIHBlcmlvZHMgZm9yIHRoZSBkYXRhYmFzZS4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICcsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUmVkc2hpZnRDbHVzdGVyTWFpbnRlbmFuY2VTZXR0aW5ncyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGFsbG93cyBwdWJsaWMgYWNjZXNzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQW1hem9uIFJlZHNoaWZ0IGNsdXN0ZXJzIGNhbiBjb250YWluIHNlbnNpdGl2ZSBpbmZvcm1hdGlvbiBhbmQgcHJpbmNpcGxlcyBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdENsdXN0ZXJQdWJsaWNBY2Nlc3MsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuaGFuY2VkIFZQQyByb3V0aW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4zLCAxLjMuMSwgMS4zLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0VuaGFuY2VkIFZQQyByb3V0aW5nIGZvcmNlcyBhbGwgQ09QWSBhbmQgVU5MT0FEIHRyYWZmaWMgYmV0d2VlbiB0aGUgY2x1c3RlciBhbmQgZGF0YSByZXBvc2l0b3JpZXMgdG8gZ28gdGhyb3VnaCB5b3VyIEFtYXpvbiBWUEMuIFlvdSBjYW4gdGhlbiB1c2UgVlBDIGZlYXR1cmVzIHN1Y2ggYXMgc2VjdXJpdHkgZ3JvdXBzIGFuZCBuZXR3b3JrIGFjY2VzcyBjb250cm9sIGxpc3RzIHRvIHNlY3VyZSBuZXR3b3JrIHRyYWZmaWMuIFlvdSBjYW4gYWxzbyB1c2UgVlBDIGZsb3cgbG9ncyB0byBtb25pdG9yIG5ldHdvcmsgdHJhZmZpYy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFJlZHNoaWZ0RW5oYW5jZWRWUENSb3V0aW5nRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGRvZXMgbm90IHJlcXVpcmUgVExTL1NTTCBlbmNyeXB0aW9uIC0gKENvbnRyb2wgSURzOiAyLjMswqA0LjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSB0aGF0IHlvdXIgQW1hem9uIFJlZHNoaWZ0IGNsdXN0ZXJzIHJlcXVpcmUgVExTL1NTTCBlbmNyeXB0aW9uIHRvIGNvbm5lY3QgdG8gU1FMIGNsaWVudHMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBSZWRzaGlmdFJlcXVpcmVUbHNTU0wsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEFtYXpvbiBTMyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUzMobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFMzIGJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgYWNjZXNzIHRocm91Z2ggYnVja2V0IGxldmVsIHNldHRpbmdzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnS2VlcCBzZW5zaXRpdmUgZGF0YSBzYWZlIGZyb20gdW5hdXRob3JpemVkIHJlbW90ZSB1c2VycyBieSBwcmV2ZW50aW5nIHB1YmxpYyBhY2Nlc3MgYXQgdGhlIGJ1Y2tldCBsZXZlbC4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFMzQnVja2V0TGV2ZWxQdWJsaWNBY2Nlc3NQcm9oaWJpdGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldHMgZG9lcyBub3QgaGF2ZSBzZXJ2ZXIgYWNjZXNzIGxvZ3MgZW5hYmxlZCAtIChDb250cm9sIElEczogMi4yLCAxMC4xLCAxMC4yLjEsIDEwLjIuMiwgMTAuMi4zLCAxMC4yLjQsIDEwLjIuNSwgMTAuMi43LCAxMC4zLjEsIDEwLjMuMiwgMTAuMy4zLCAxMC4zLjQsIDEwLjMuNSwgMTAuMy42KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gU2ltcGxlIFN0b3JhZ2UgU2VydmljZSAoQW1hem9uIFMzKSBzZXJ2ZXIgYWNjZXNzIGxvZ2dpbmcgcHJvdmlkZXMgYSBtZXRob2QgdG8gbW9uaXRvciB0aGUgbmV0d29yayBmb3IgcG90ZW50aWFsIGN5YmVyc2VjdXJpdHkgZXZlbnRzLiBUaGUgZXZlbnRzIGFyZSBtb25pdG9yZWQgYnkgY2FwdHVyaW5nIGRldGFpbGVkIHJlY29yZHMgZm9yIHRoZSByZXF1ZXN0cyB0aGF0IGFyZSBtYWRlIHRvIGFuIEFtYXpvbiBTMyBidWNrZXQuIEVhY2ggYWNjZXNzIGxvZyByZWNvcmQgcHJvdmlkZXMgZGV0YWlscyBhYm91dCBhIHNpbmdsZSBhY2Nlc3MgcmVxdWVzdC4gVGhlIGRldGFpbHMgaW5jbHVkZSB0aGUgcmVxdWVzdGVyLCBidWNrZXQgbmFtZSwgcmVxdWVzdCB0aW1lLCByZXF1ZXN0IGFjdGlvbiwgcmVzcG9uc2Ugc3RhdHVzLCBhbmQgYW4gZXJyb3IgY29kZSwgaWYgcmVsZXZhbnQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTM0J1Y2tldExvZ2dpbmdFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgcmVhZCBhY2Nlc3MgdGhyb3VnaCBpdHMgQmxvY2sgUHVibGljIEFjY2VzcyBjb25maWd1cmF0aW9ucyBhbmQgYnVja2V0IEFDTHMgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAxLjMuNCwgMS4zLjYsIDIuMiwgMi4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBtYW5hZ2VtZW50IG9mIGFjY2VzcyBzaG91bGQgYmUgY29uc2lzdGVudCB3aXRoIHRoZSBjbGFzc2lmaWNhdGlvbiBvZiB0aGUgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFMzQnVja2V0UHVibGljUmVhZFByb2hpYml0ZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IHByb2hpYml0IHB1YmxpYyB3cml0ZSBhY2Nlc3MgdGhyb3VnaCBpdHMgQmxvY2sgUHVibGljIEFjY2VzcyBjb25maWd1cmF0aW9ucyBhbmQgYnVja2V0IEFDTHMgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAxLjMuNCwgMS4zLjYsIDIuMiwgMi4yLjIpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RoZSBtYW5hZ2VtZW50IG9mIGFjY2VzcyBzaG91bGQgYmUgY29uc2lzdGVudCB3aXRoIHRoZSBjbGFzc2lmaWNhdGlvbiBvZiB0aGUgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFMzQnVja2V0UHVibGljV3JpdGVQcm9oaWJpdGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBoYXZlIHJlcGxpY2F0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDIuMiwgMTAuNS4zKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gU2ltcGxlIFN0b3JhZ2UgU2VydmljZSAoQW1hem9uIFMzKSBDcm9zcy1SZWdpb24gUmVwbGljYXRpb24gKENSUikgc3VwcG9ydHMgbWFpbnRhaW5pbmcgYWRlcXVhdGUgY2FwYWNpdHkgYW5kIGF2YWlsYWJpbGl0eS4gQ1JSIGVuYWJsZXMgYXV0b21hdGljLCBhc3luY2hyb25vdXMgY29weWluZyBvZiBvYmplY3RzIGFjcm9zcyBBbWF6b24gUzMgYnVja2V0cyB0byBoZWxwIGVuc3VyZSB0aGF0IGRhdGEgYXZhaWxhYmlsaXR5IGlzIG1haW50YWluZWQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTM0J1Y2tldFJlcGxpY2F0aW9uRW5hYmxlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSBkZWZhdWx0IHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogMi4yLCAzLjQsIDguMi4xLCAxMC41KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIEFtYXpvbiBTMyBidWNrZXRzLCBlbmFibGUgZW5jcnlwdGlvbiB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgdmVyc2lvbmluZyBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IDEwLjUuMykuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVXNlIHZlcnNpb25pbmcgdG8gcHJlc2VydmUsIHJldHJpZXZlLCBhbmQgcmVzdG9yZSBldmVyeSB2ZXJzaW9uIG9mIGV2ZXJ5IG9iamVjdCBzdG9yZWQgaW4geW91ciBBbWF6b24gUzMgYnVja2V0LiBWZXJzaW9uaW5nIGhlbHBzIHlvdSB0byBlYXNpbHkgcmVjb3ZlciBmcm9tIHVuaW50ZW5kZWQgdXNlciBhY3Rpb25zIGFuZCBhcHBsaWNhdGlvbiBmYWlsdXJlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFMzQnVja2V0VmVyc2lvbmluZ0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgUzMgQnVja2V0IGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBLZXkgYnkgZGVmYXVsdCAtIChDb250cm9sIElEczogMy40LCA4LjIuMSwgMTAuNSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnRW5zdXJlIHRoYXQgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBTaW1wbGUgU3RvcmFnZSBTZXJ2aWNlIChBbWF6b24gUzMpIGJ1Y2tldHMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gYW4gQW1hem9uIFMzIGJ1Y2tldCwgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogUzNEZWZhdWx0RW5jcnlwdGlvbktNUyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2FnZU1ha2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTYWdlTWFrZXIobm9kZTogQ2ZuUmVzb3VyY2UpIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFNhZ2VNYWtlciByZXNvdXJjZSBlbmRwb2ludCBpcyBub3QgZW5jcnlwdGVkIHdpdGggYSBLTVMga2V5IC0gKENvbnRyb2wgSURzOiAzLjQsIDguMi4xKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIFNhZ2VNYWtlciBlbmRwb2ludCwgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogU2FnZU1ha2VyRW5kcG9pbnRDb25maWd1cmF0aW9uS01TS2V5Q29uZmlndXJlZCxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBTYWdlTWFrZXIgbm90ZWJvb2sgaXMgbm90IGVuY3J5cHRlZCB3aXRoIGEgS01TIGtleSAtIChDb250cm9sIElEczogMy40LCA4LjIuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBTYWdlTWFrZXIgbm90ZWJvb2ssIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFNhZ2VNYWtlck5vdGVib29rSW5zdGFuY2VLTVNLZXlDb25maWd1cmVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFNhZ2VNYWtlciBub3RlYm9vayBkb2VzIG5vdCBkaXNhYmxlIGRpcmVjdCBpbnRlcm5ldCBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAxLjMuNCwgMS4zLjYsIDIuMi4yKS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdCeSBwcmV2ZW50aW5nIGRpcmVjdCBpbnRlcm5ldCBhY2Nlc3MsIHlvdSBjYW4ga2VlcCBzZW5zaXRpdmUgZGF0YSBmcm9tIGJlaW5nIGFjY2Vzc2VkIGJ5IHVuYXV0aG9yaXplZCB1c2Vycy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFNhZ2VNYWtlck5vdGVib29rTm9EaXJlY3RJbnRlcm5ldEFjY2VzcyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2VjcmV0cyBNYW5hZ2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTZWNyZXRzTWFuYWdlcihub2RlOiBDZm5SZXNvdXJjZSk6IHZvaWQge1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgc2VjcmV0IGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBDdXN0b21lciBtYW5hZ2VkIGtleSAtIChDb250cm9sIElEczogMy40LCA4LjIuMSkuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gd2l0aCBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykgaXMgZW5hYmxlZCBmb3IgQVdTIFNlY3JldHMgTWFuYWdlciBzZWNyZXRzLiBCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIFNlY3JldHMgTWFuYWdlciBzZWNyZXRzLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBTZWNyZXRzTWFuYWdlclVzaW5nS01TS2V5LFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbWF6b24gU05TIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tTTlMobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiAnVGhlIFNOUyB0b3BpYyBkb2VzIG5vdCBoYXZlIEtNUyBlbmNyeXB0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRDogOC4yLjEpLicsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ1RvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHJlc3QsIGVuc3VyZSB0aGF0IHlvdXIgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoQW1hem9uIFNOUykgdG9waWNzIHJlcXVpcmUgZW5jcnlwdGlvbiB1c2luZyBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykgQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBwdWJsaXNoZWQgbWVzc2FnZXMsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5FUlJPUixcbiAgICAgIHJ1bGU6IFNOU0VuY3J5cHRlZEtNUyxcbiAgICAgIG5vZGU6IG5vZGUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgVlBDIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tWUEMobm9kZTogQ2ZuUmVzb3VyY2UpOiB2b2lkIHtcbiAgICB0aGlzLmFwcGx5UnVsZSh7XG4gICAgICBpbmZvOiBcIlRoZSBWUEMncyBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwIGFsbG93cyBpbmJvdW5kIG9yIG91dGJvdW5kIHRyYWZmaWMgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAyLjEsIDIuMiwgMi4yLjIpLlwiLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBbWF6b24gRWxhc3RpYyBDb21wdXRlIENsb3VkIChBbWF6b24gRUMyKSBzZWN1cml0eSBncm91cHMgY2FuIGhlbHAgaW4gdGhlIG1hbmFnZW1lbnQgb2YgbmV0d29yayBhY2Nlc3MgYnkgcHJvdmlkaW5nIHN0YXRlZnVsIGZpbHRlcmluZyBvZiBpbmdyZXNzIGFuZCBlZ3Jlc3MgbmV0d29yayB0cmFmZmljIHRvIEFXUyByZXNvdXJjZXMuIFJlc3RyaWN0aW5nIGFsbCB0aGUgdHJhZmZpYyBvbiB0aGUgZGVmYXVsdCBzZWN1cml0eSBncm91cCBoZWxwcyBpbiByZXN0cmljdGluZyByZW1vdGUgYWNjZXNzIHRvIHlvdXIgQVdTIHJlc291cmNlcy4nLFxuICAgICAgbGV2ZWw6IE5hZ01lc3NhZ2VMZXZlbC5XQVJOLFxuICAgICAgcnVsZTogVlBDRGVmYXVsdFNlY3VyaXR5R3JvdXBDbG9zZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgVlBDIGRvZXMgbm90IGhhdmUgYW4gYXNzb2NpYXRlZCBGbG93IExvZyAtIChDb250cm9sIElEczogMi4yLCAxMC4xLCAxMC4zLjIsIDEwLjMuMywgMTAuMy40LCAxMC4zLjUsIDEwLjMuNikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnVGhlIFZQQyBmbG93IGxvZ3MgcHJvdmlkZSBkZXRhaWxlZCByZWNvcmRzIGZvciBpbmZvcm1hdGlvbiBhYm91dCB0aGUgSVAgdHJhZmZpYyBnb2luZyB0byBhbmQgZnJvbSBuZXR3b3JrIGludGVyZmFjZXMgaW4geW91ciBBbWF6b24gVmlydHVhbCBQcml2YXRlIENsb3VkIChBbWF6b24gVlBDKS4gQnkgZGVmYXVsdCwgdGhlIGZsb3cgbG9nIHJlY29yZCBpbmNsdWRlcyB2YWx1ZXMgZm9yIHRoZSBkaWZmZXJlbnQgY29tcG9uZW50cyBvZiB0aGUgSVAgZmxvdywgaW5jbHVkaW5nIHRoZSBzb3VyY2UsIGRlc3RpbmF0aW9uLCBhbmQgcHJvdG9jb2wuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBWUENGbG93TG9nc0VuYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86IFwiVGhlIHJvdXRlIHRhYmxlIG1heSBjb250YWluIG9uZSBvciBtb3JlIHVucmVzdHJpY3RlZCByb3V0ZShzKSB0byBhbiBJR1cgKCcwLjAuMC4wLzAnIG9yICc6Oi8wJykgLSAoQ29udHJvbCBJRHM6IDEuMiwgMS4yLjEsIDEuMywgMS4zLjEsIDEuMy4yLCAyLjIuMikuXCIsXG4gICAgICBleHBsYW5hdGlvbjpcbiAgICAgICAgJ0Vuc3VyZSBBbWF6b24gRUMyIHJvdXRlIHRhYmxlcyBkbyBub3QgaGF2ZSB1bnJlc3RyaWN0ZWQgcm91dGVzIHRvIGFuIGludGVybmV0IGdhdGV3YXkuIFJlbW92aW5nIG9yIGxpbWl0aW5nIHRoZSBhY2Nlc3MgdG8gdGhlIGludGVybmV0IGZvciB3b3JrbG9hZHMgd2l0aGluIEFtYXpvbiBWUENzIGNhbiByZWR1Y2UgdW5pbnRlbmRlZCBhY2Nlc3Mgd2l0aGluIHlvdXIgZW52aXJvbm1lbnQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBWUENOb1VucmVzdHJpY3RlZFJvdXRlVG9JR1csXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICAgIHRoaXMuYXBwbHlSdWxlKHtcbiAgICAgIGluZm86ICdUaGUgc3VibmV0IGF1dG8tYXNzaWducyBwdWJsaWMgSVAgYWRkcmVzc2VzIC0gKENvbnRyb2wgSURzOiAxLjIsIDEuMi4xLCAxLjMsIDEuMy4xLCAxLjMuMiwgMS4zLjQsIDEuMy42LCAyLjIuMikuJyxcbiAgICAgIGV4cGxhbmF0aW9uOlxuICAgICAgICAnTWFuYWdlIGFjY2VzcyB0byB0aGUgQVdTIENsb3VkIGJ5IGVuc3VyaW5nIEFtYXpvbiBWaXJ0dWFsIFByaXZhdGUgQ2xvdWQgKFZQQykgc3VibmV0cyBhcmUgbm90IGF1dG9tYXRpY2FsbHkgYXNzaWduZWQgYSBwdWJsaWMgSVAgYWRkcmVzcy4gQW1hem9uIEVsYXN0aWMgQ29tcHV0ZSBDbG91ZCAoRUMyKSBpbnN0YW5jZXMgdGhhdCBhcmUgbGF1bmNoZWQgaW50byBzdWJuZXRzIHRoYXQgaGF2ZSB0aGlzIGF0dHJpYnV0ZSBlbmFibGVkIGhhdmUgYSBwdWJsaWMgSVAgYWRkcmVzcyBhc3NpZ25lZCB0byB0aGVpciBwcmltYXJ5IG5ldHdvcmsgaW50ZXJmYWNlLicsXG4gICAgICBsZXZlbDogTmFnTWVzc2FnZUxldmVsLkVSUk9SLFxuICAgICAgcnVsZTogVlBDU3VibmV0QXV0b0Fzc2lnblB1YmxpY0lwRGlzYWJsZWQsXG4gICAgICBub2RlOiBub2RlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFdBRiBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrV0FGKG5vZGU6IENmblJlc291cmNlKTogdm9pZCB7XG4gICAgdGhpcy5hcHBseVJ1bGUoe1xuICAgICAgaW5mbzogJ1RoZSBXQUZ2MiB3ZWIgQUNMIGRvZXMgbm90IGhhdmUgbG9nZ2luZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxMC4xLMKgMTAuMy4xLMKgMTAuMy4yLMKgMTAuMy4zLMKgMTAuMy40LMKgMTAuMy41LMKgMTAuMy42LMKgMTAuNS40KS4nLFxuICAgICAgZXhwbGFuYXRpb246XG4gICAgICAgICdBV1MgV0FGIGxvZ2dpbmcgcHJvdmlkZXMgZGV0YWlsZWQgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHRyYWZmaWMgdGhhdCBpcyBhbmFseXplZCBieSB5b3VyIHdlYiBBQ0wuIFRoZSBsb2dzIHJlY29yZCB0aGUgdGltZSB0aGF0IEFXUyBXQUYgcmVjZWl2ZWQgdGhlIHJlcXVlc3QgZnJvbSB5b3VyIEFXUyByZXNvdXJjZSwgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHJlcXVlc3QsIGFuZCBhbiBhY3Rpb24gZm9yIHRoZSBydWxlIHRoYXQgZWFjaCByZXF1ZXN0IG1hdGNoZWQuJyxcbiAgICAgIGxldmVsOiBOYWdNZXNzYWdlTGV2ZWwuRVJST1IsXG4gICAgICBydWxlOiBXQUZ2MkxvZ2dpbmdFbmFibGVkLFxuICAgICAgbm9kZTogbm9kZSxcbiAgICB9KTtcbiAgfVxufVxuIl19