#!/usr/bin/python

# Copyright (C) 2019-2020 Vanessa Sochat.

# This Source Code Form is subject to the terms of the
# Mozilla Public License, v. 2.0. If a copy of the MPL was not distributed
# with this file, You can obtain one at http://mozilla.org/MPL/2.0/.

from opencontainers.image.v1 import Image
import os
import pytest


config_invalid_os = {
    "architecture": "amd64",
    "os": 123,
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}

config_invalid_user = {
    "created": "2015-10-31T22:22:56.015925234Z",
    "author": "Alyssa P. Hacker <alyspdev@example.com>",
    "architecture": "amd64",
    "os": "linux",
    "config": {"User": 1234},
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}

config_invalid_history = {
    "history": "should be an array",
    "architecture": "amd64",
    "os": "linux",
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}

config_invalid_envint = {
    "architecture": "amd64",
    "os": "linux",
    "config": {"Env": [7353]},
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}


config_invalid_volumes = {
    "architecture": "amd64",
    "os": "linux",
    "config": {"Volumes": ["/var/job-result-data", "/var/log/my-app-logs"]},
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}

config_valid_with_optional = {
    "created": "2015-10-31T22:22:56.015925234Z",
    "author": "Alyssa P. Hacker <alyspdev@example.com>",
    "architecture": "amd64",
    "os": "linux",
    "config": {
        "User": "1:1",
        "ExposedPorts": {"8080/tcp": {}},
        "Env": [
            "PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin",
            "FOO=docker_is_a_really",
            "BAR=great_tool_you_know",
        ],
        "Entrypoint": ["/bin/sh"],
        "Cmd": ["--foreground", "--config", "/etc/my-app.d/default.cfg"],
        "Volumes": {"/var/job-result-data": {}, "/var/log/my-app-logs": {}},
        "StopSignal": "SIGKILL",
        "WorkingDir": "/home/alice",
        "Labels": {
            "com.example.project.git.url": "https://example.com/project.git",
            "com.example.project.git.commit": "45a939b2999782a3f005621a8d0f29aa387e1d6b",
        },
    },
    "rootfs": {
        "diff_ids": [
            "sha256:9d3dd9504c685a304985025df4ed0283e47ac9ffa9bd0326fddf4d59513f0827",
            "sha256:2b689805fbd00b2db1df73fae47562faac1a626d5f61744bfe29946ecff5d73d",
        ],
        "type": "layers",
    },
    "history": [
        {
            "created": "2015-10-31T22:22:54.690851953Z",
            "created_by": "/bin/sh -c #(nop) ADD file:a3bc1e842b69636f9df5256c49c5374fb4eef1e281fe3f282c65fb853ee171c5 in /",
        },
        {
            "created": "2015-10-31T22:22:55.613815829Z",
            "created_by": '/bin/sh -c #(nop) CMD ["sh"]',
            "empty_layer": True,
        },
    ],
}

config_valid_required = {
    "architecture": "amd64",
    "os": "linux",
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}


config_invalid_env = {
    "architecture": "amd64",
    "os": "linux",
    "config": {"Env": ["foo"]},
    "rootfs": {
        "diff_ids": [
            "sha256:5f70bf18a086007016e948b04aed3b82103a36bea41755b6cddfaf10ace3c6ef"
        ],
        "type": "layers",
    },
}


def test_example_config(tmp_path):
    """test creation of an opencontainers Image"""
    image = Image()

    # OS is int, and is invalid
    with pytest.raises(SystemExit):
        image.load(config_invalid_os)

    # User should be string
    with pytest.raises(SystemExit):
        image.load(config_invalid_user)

    # History should be list
    with pytest.raises(SystemExit):
        image.load(config_invalid_history)

    # Env is numeric, must be list of strings
    with pytest.raises(SystemExit):
        image.load(config_invalid_envint)

    # volumes cannot be list
    with pytest.raises(SystemExit):
        image.load(config_invalid_volumes)

    # invalid environment
    with pytest.raises(SystemExit):
        image.load(config_invalid_env)

    # valid config with optional fields
    image.load(config_valid_with_optional)
    assert image.validate()

    # minimum valid required
    image.load(config_valid_required)
    assert image.validate()
