"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
describe('apex domain', () => {
    test('returns right domain', () => {
        expect(util_1.apexDomain('domain.com')).toEqual('domain.com');
        expect(util_1.apexDomain('test.domain.com')).toEqual('domain.com');
    });
    test('understands eTLDs', () => {
        expect(util_1.apexDomain('test.domain.co.uk')).toEqual('domain.co.uk');
    });
});
describe('isDnsValidatedCertificate', () => {
    test('new DnsValidatedCertificate is a DnsValidatedCertificate', () => {
        const stack = new core_1.Stack();
        const hostedZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com',
        });
        const cert = new lib_1.DnsValidatedCertificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            hostedZone,
        });
        expect(util_1.isDnsValidatedCertificate(cert)).toBeTruthy();
    });
    test('new Certificate is not a DnsValidatedCertificate', () => {
        const stack = new core_1.Stack();
        const cert = new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
        });
        expect(util_1.isDnsValidatedCertificate(cert)).toBeFalsy();
    });
    test('fromCertificateArn is not a DnsValidatedCertificate', () => {
        const stack = new core_1.Stack();
        const cert = lib_1.Certificate.fromCertificateArn(stack, 'Certificate', 'cert-arn');
        expect(util_1.isDnsValidatedCertificate(cert)).toBeFalsy();
    });
});
describe('getCertificateRegion', () => {
    test('from stack', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'RegionStack', { env: { region: 'eu-west-1' } });
        const certificate = new lib_1.Certificate(stack, 'TestCertificate', {
            domainName: 'www.example.com',
        });
        expect(util_1.getCertificateRegion(certificate)).toEqual('eu-west-1');
    });
    test('from DnsValidatedCertificate region', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'RegionStack', { env: { region: 'eu-west-1' } });
        const hostedZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com',
        });
        const certificate = new lib_1.DnsValidatedCertificate(stack, 'TestCertificate', {
            domainName: 'www.example.com',
            hostedZone,
            region: 'eu-west-3',
        });
        expect(util_1.getCertificateRegion(certificate)).toEqual('eu-west-3');
    });
    test('fromCertificateArn', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'RegionStack', { env: { region: 'eu-west-1' } });
        const certificate = lib_1.Certificate.fromCertificateArn(stack, 'TestCertificate', 'arn:aws:acm:us-east-2:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
        expect(util_1.getCertificateRegion(certificate)).toEqual('us-east-2');
    });
    test('region agnostic stack', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const certificate = new lib_1.Certificate(stack, 'TestCertificate', {
            domainName: 'www.example.com',
        });
        expect(util_1.getCertificateRegion(certificate)).toEqual('${Token[AWS::Region.4]}');
    });
});
//# sourceMappingURL=data:application/json;base64,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